/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.jenkins.internal.http;

import com.atlassian.bitbucket.jenkins.internal.client.HttpRequestExecutor;
import com.atlassian.bitbucket.jenkins.internal.client.exception.AuthorizationException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.BadRequestException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.BitbucketClientException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.ConnectionFailureException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.NotFoundException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.ServerErrorException;
import com.atlassian.bitbucket.jenkins.internal.client.exception.UnhandledErrorException;
import com.atlassian.bitbucket.jenkins.internal.credentials.BitbucketCredentials;
import hudson.Plugin;
import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.inject.Inject;
import jenkins.model.Jenkins;
import okhttp3.Call;
import okhttp3.Headers;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

public class HttpRequestExecutorImpl
implements HttpRequestExecutor {
    private static final int BAD_REQUEST_FAMILY = 4;
    private static final MediaType JSON = MediaType.parse((String)"application/json; charset=utf-8");
    private static final Logger log = Logger.getLogger(HttpRequestExecutorImpl.class.getName());
    private static final int SERVER_ERROR_FAMILY = 5;
    private final Call.Factory httpCallFactory;

    @Inject
    public HttpRequestExecutorImpl() {
        this((Call.Factory)new OkHttpClient.Builder().addInterceptor((Interceptor)new UserAgentInterceptor()).build());
    }

    public HttpRequestExecutorImpl(Call.Factory httpCallFactory) {
        this.httpCallFactory = httpCallFactory;
    }

    @Override
    public void executeDelete(HttpUrl url, BitbucketCredentials credentials) {
        Request.Builder requestBuilder = new Request.Builder().url(url).delete();
        this.executeRequest(requestBuilder, credentials, HttpRequestExecutor.ResponseConsumer.EMPTY_RESPONSE);
    }

    @Override
    public <T> T executeGet(HttpUrl url, BitbucketCredentials credentials, HttpRequestExecutor.ResponseConsumer<T> consumer) {
        Request.Builder requestBuilder = new Request.Builder().url(url);
        return this.executeRequest(requestBuilder, credentials, consumer);
    }

    @Override
    public <T> T executePost(HttpUrl url, BitbucketCredentials credential, String requestBodyAsJson, HttpRequestExecutor.ResponseConsumer<T> consumer, Headers headers) {
        Request.Builder requestBuilder = new Request.Builder().post(RequestBody.create((MediaType)JSON, (String)requestBodyAsJson)).url(url).headers(headers);
        return this.executeRequest(requestBuilder, credential, consumer);
    }

    @Override
    public <T> T executePut(HttpUrl url, BitbucketCredentials credentials, String requestBodyAsJson, HttpRequestExecutor.ResponseConsumer<T> consumer) {
        Request.Builder requestBuilder = new Request.Builder().put(RequestBody.create((MediaType)JSON, (String)requestBodyAsJson)).url(url);
        return this.executeRequest(requestBuilder, credentials, consumer);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T> T executeRequest(Request.Builder requestBuilder, BitbucketCredentials credentials, HttpRequestExecutor.ResponseConsumer<T> consumer) {
        try {
            this.addAuthentication(credentials, requestBuilder);
            Response response = this.httpCallFactory.newCall(requestBuilder.build()).execute();
            int responseCode = response.code();
            try (ResponseBody body = response.body();){
                if (response.isSuccessful()) {
                    log.fine("Bitbucket - call successful");
                    T t = consumer.consume(response);
                    return t;
                }
                HttpRequestExecutorImpl.handleError(responseCode, body == null ? null : body.string());
                throw new UnhandledErrorException("Unhandled error", -1, null);
            }
        }
        catch (ConnectException | SocketTimeoutException e) {
            log.log(Level.FINE, "Bitbucket - Connection failed", e);
            throw new ConnectionFailureException(e);
        }
        catch (IOException e) {
            log.log(Level.FINE, "Bitbucket - io exception", e);
            throw new BitbucketClientException(e);
        }
    }

    private void addAuthentication(BitbucketCredentials credential, Request.Builder requestBuilder) {
        if (credential != BitbucketCredentials.ANONYMOUS_CREDENTIALS) {
            requestBuilder.addHeader("Authorization", credential.toHeaderValue());
        }
    }

    private static void handleError(int responseCode, @Nullable String body) throws AuthorizationException {
        switch (responseCode) {
            case 401: 
            case 403: {
                log.info("Bitbucket - responded with not authorized ");
                throw new AuthorizationException("Provided credentials cannot access the resource", responseCode, body);
            }
            case 404: {
                log.info("Bitbucket - Path not found");
                throw new NotFoundException("The requested resource does not exist", body);
            }
        }
        int family = responseCode / 100;
        switch (family) {
            case 4: {
                log.info("Bitbucket - did not accept the request");
                throw new BadRequestException("The request is malformed", responseCode, body);
            }
            case 5: {
                log.info("Bitbucket - failed to service request");
                throw new ServerErrorException("The server failed to service request", responseCode, body);
            }
        }
        throw new UnhandledErrorException("Unhandled error", responseCode, body);
    }

    private static class UserAgentInterceptor
    implements Interceptor {
        private final String bbJenkinsUserAgent;

        UserAgentInterceptor() {
            String version = "unknown";
            try {
                Plugin plugin = Jenkins.get().getPlugin("atlassian-bitbucket-server-integration");
                if (plugin != null) {
                    version = plugin.getWrapper().getVersion();
                }
            }
            catch (IllegalStateException e) {
                org.apache.log4j.Logger.getLogger(UserAgentInterceptor.class).warn((Object)"Jenkins not available", (Throwable)e);
            }
            this.bbJenkinsUserAgent = "bitbucket-jenkins-integration/" + version;
        }

        public Response intercept(Interceptor.Chain chain) throws IOException {
            Request request = chain.request().newBuilder().header("User-Agent", this.bbJenkinsUserAgent).build();
            return chain.proceed(request);
        }
    }
}

