package io.intercom.android.nexus;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;
import java.util.UUID;

import static io.intercom.android.nexus.NexusEvent.ADMIN_AVATAR;
import static io.intercom.android.nexus.NexusEvent.ADMIN_ID;
import static io.intercom.android.nexus.NexusEvent.ADMIN_NAME;
import static io.intercom.android.nexus.NexusEvent.ASSIGNEE_ID;
import static io.intercom.android.nexus.NexusEvent.CONVERSATION_ID;
import static io.intercom.android.nexus.NexusEvent.EVENT_DATA;
import static io.intercom.android.nexus.NexusEvent.EVENT_GUID;
import static io.intercom.android.nexus.NexusEvent.EVENT_NAME;
import static io.intercom.android.nexus.NexusEvent.NX_FROM_USER;
import static io.intercom.android.nexus.NexusEvent.NX_TOPICS;
import static io.intercom.android.nexus.NexusEvent.NX_TO_USER;

public enum NexusEventType {
    NewComment {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            event.put(NX_TO_USER, nexusEvent.getUserId());
            return event;
        }
    },

    ConversationSeen {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            event.put(NX_FROM_USER, nexusEvent.getUserId());
            return event;
        }
    },

    UserIsTyping {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            event.put(NX_FROM_USER, nexusEvent.getUserId());
            return event;
        }
    },

    AdminIsTyping {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.optJSONObject(EVENT_DATA);
            eventData.put(ADMIN_NAME, nexusEvent.getEventData().optString(ADMIN_NAME));
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            eventData.put(ADMIN_AVATAR, nexusEvent.getEventData().optString(ADMIN_AVATAR));
            event.put(NX_TO_USER, nexusEvent.getUserId());
            return event;
        }
    },

    AdminIsTypingANote {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.optJSONObject(EVENT_DATA);
            eventData.put(ADMIN_NAME, nexusEvent.getEventData().optString(ADMIN_NAME));
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            eventData.put(ADMIN_AVATAR, nexusEvent.getEventData().optString(ADMIN_AVATAR));
            event.put(NX_TO_USER, nexusEvent.getUserId());
            return event;
        }
    },

    NewNote {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.optJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            return event;
        }
    },

    ConversationAssigned {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.optJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            eventData.put(ASSIGNEE_ID, nexusEvent.getEventData().optString(ASSIGNEE_ID));
            return event;
        }
    },

    ConversationClosed {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.optJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            return event;
        }
    },

    ConversationReopened {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.getJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            return event;
        }
    },

    UserPresence {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = new JSONObject();
            JSONObject eventData = new JSONObject();
            event.put(EVENT_GUID, UUID.randomUUID().toString());
            event.put(EVENT_NAME, "nx." + name());
            event.put(EVENT_DATA, eventData);
            return event;
        }
    },

    UserContentSeenByAdmin {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            event.put(NX_TO_USER, nexusEvent.getUserId());
            return event;
        }
    },

    Subscribe {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            event.put(EVENT_NAME, "nx." + name());
            return event;
        }
    },

    Unsubscribe {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            event.put(EVENT_NAME, "nx." + name());
            return event;
        }
    },

    //region inbox topic event types
    ThreadAssigned {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.optJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            eventData.put(ASSIGNEE_ID, nexusEvent.getEventData().optString(ASSIGNEE_ID));
            return event;
        }
    },

    ThreadClosed {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.optJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            return event;
        }
    },

    ThreadReopened {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.getJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            return event;
        }
    },

    ThreadUpdated {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            JSONObject eventData = event.getJSONObject(EVENT_DATA);
            eventData.put(ADMIN_ID, nexusEvent.getEventData().optString(ADMIN_ID));
            return event;
        }
    },

    ThreadCreated {
        @Override protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
            JSONObject event = super.toJsonObject(nexusEvent);
            event.put(NX_TO_USER, nexusEvent.getUserId());
            return event;
        }
    },
    //endregion

    UNKNOWN {
        @Override protected String toStringEncodedJsonObject(NexusEvent nexusEvent) {
            return "";
        }
    };

    protected String toStringEncodedJsonObject(NexusEvent nexusEvent){
        try {
            return toJsonObject(nexusEvent).toString();
        } catch (JSONException e) {
            return "";
        }
    }

    protected JSONObject toJsonObject(NexusEvent nexusEvent) throws JSONException {
        JSONObject event = new JSONObject();
        JSONObject eventData = new JSONObject();
        eventData.put(CONVERSATION_ID, nexusEvent.getEventData().optString(CONVERSATION_ID));
        event.put(EVENT_GUID, nexusEvent.getGuid());
        event.put(EVENT_NAME, name());
        event.put(EVENT_DATA, eventData);
        List<String> topics = nexusEvent.getTopics();
        if (!topics.isEmpty()) {
            event.put(NX_TOPICS, new JSONArray(topics));
        }

        return event;
    }

    public static NexusEventType safeValueOf(String name) {
        try {
            return NexusEventType.valueOf(name);
        } catch (Exception e) {
            return NexusEventType.UNKNOWN;
        }
    }
}
