/*
 * Copyright 2017-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.koin.compose.scope

import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import org.koin.compose.ComposeContextWrapper
import org.koin.compose.LocalKoinScopeContext
import org.koin.core.annotation.KoinDelicateAPI
import org.koin.core.annotation.KoinExperimentalAPI
import org.koin.core.annotation.KoinInternalApi
import org.koin.core.scope.Scope

/**
 * Safely provides an existing Koin scope to [LocalKoinScopeContext] without binding it to the Composable lifecycle.
 *
 * This function sets the given scope as the current [LocalKoinScopeContext] for the composable hierarchy,
 * allowing child composables to access it through Koin's injection APIs. The scope must already exist
 * and is provided directly.
 *
 * **Important**: This function does NOT manage the scope lifecycle. The scope is unbound from the
 * Composable and will not be automatically closed when the composable leaves composition. You must
 * manually close the scope when it's no longer needed to prevent memory leaks.
 *
 * Use this when you need to provide a scope that has a different lifecycle than the Composable,
 * such as scopes managed by external systems or shared across multiple composable trees.
 *
 * Example usage:
 * ```kotlin
 * @Composable
 * fun MyFeature(externalScope: Scope, onClose: () -> Unit) {
 *     UnboundKoinScope(scope = externalScope) {
 *         // Child composables can access the scope via LocalKoinScopeContext
 *         val myService = koinInject<MyService>()
 *
 *         // Remember to close the scope externally
 *         DisposableEffect(Unit) {
 *             onDispose { onClose() }
 *         }
 *     }
 * }
 * ```
 *
 * @param scope The existing Koin scope to provide to [LocalKoinScopeContext]
 * @param content The composable content that will have access to the scope via [LocalKoinScopeContext]
 *
 * @see rememberKoinScope for automatically lifecycle-managed scopes
 */
@OptIn(KoinInternalApi::class)
@KoinDelicateAPI
@KoinExperimentalAPI
@Composable
inline fun UnboundKoinScope(
    scope : Scope,
    noinline content: @Composable () -> Unit
) {
    CompositionLocalProvider(
        LocalKoinScopeContext provides ComposeContextWrapper(scope),
    ) {
        content()
    }
}