/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security;

import io.helidon.common.configurable.ThreadPoolSupplier;
import io.helidon.common.serviceloader.HelidonServiceLoader;
import io.helidon.config.Config;
import io.helidon.security.AuditEvent;
import io.helidon.security.AuthenticationResponse;
import io.helidon.security.AuthorizationResponse;
import io.helidon.security.CompositeProviderSelectionPolicy;
import io.helidon.security.DefaultAuditProvider;
import io.helidon.security.FirstProviderSelectionPolicy;
import io.helidon.security.Grant;
import io.helidon.security.NamedProvider;
import io.helidon.security.ProviderRequest;
import io.helidon.security.ProviderSelectionPolicyType;
import io.helidon.security.ReflectionUtil;
import io.helidon.security.Role;
import io.helidon.security.SecurityContext;
import io.helidon.security.SecurityEnvironment;
import io.helidon.security.SecurityException;
import io.helidon.security.SecurityTime;
import io.helidon.security.SecurityUtil;
import io.helidon.security.Subject;
import io.helidon.security.internal.SecurityAuditEvent;
import io.helidon.security.spi.AuditProvider;
import io.helidon.security.spi.AuthenticationProvider;
import io.helidon.security.spi.AuthorizationProvider;
import io.helidon.security.spi.OutboundSecurityProvider;
import io.helidon.security.spi.ProviderSelectionPolicy;
import io.helidon.security.spi.SecurityProvider;
import io.helidon.security.spi.SecurityProviderService;
import io.helidon.security.spi.SubjectMappingProvider;
import io.opentracing.Tracer;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class Security {
    public static final String HEADER_ORIG_URI = "X_ORIG_URI_HEADER";
    private static final Set<String> RESERVED_PROVIDER_KEYS = Set.of("name", "class", "is-authentication-provider", "is-authorization-provider", "is-client-security-provider", "is-audit-provider");
    private static final Set<String> CONFIG_INTERNAL_PREFIXES = Set.of("provider-policy", "providers", "environment");
    private static final Logger LOGGER = Logger.getLogger(Security.class.getName());
    private final Collection<Class<? extends Annotation>> annotations = new LinkedList<Class<? extends Annotation>>();
    private final List<Consumer<AuditProvider.TracedAuditEvent>> auditors = new LinkedList<Consumer<AuditProvider.TracedAuditEvent>>();
    private final Optional<SubjectMappingProvider> subjectMappingProvider;
    private final String instanceUuid;
    private final ProviderSelectionPolicy providerSelectionPolicy;
    private final Tracer securityTracer;
    private final SecurityTime serverTime;
    private final Supplier<ExecutorService> executorService;
    private final Config securityConfig;
    private final boolean enabled;

    private Security(Builder builder) {
        this.enabled = builder.enabled;
        this.instanceUuid = UUID.randomUUID().toString();
        this.serverTime = builder.serverTime;
        this.executorService = builder.executorService;
        this.annotations.addAll(SecurityUtil.getAnnotations(builder.allProviders));
        this.securityTracer = SecurityUtil.getTracer(builder.tracingEnabled, builder.tracer);
        this.subjectMappingProvider = Optional.ofNullable(builder.subjectMappingProvider);
        this.securityConfig = builder.config;
        if (!this.enabled) {
            this.audit(this.instanceUuid, SecurityAuditEvent.info("security.configure", "Security is disabled."));
        }
        final LinkedList<NamedProvider<AuthorizationProvider>> atzProviders = new LinkedList<NamedProvider<AuthorizationProvider>>();
        final LinkedList<NamedProvider<AuthenticationProvider>> atnProviders = new LinkedList<NamedProvider<AuthenticationProvider>>();
        final LinkedList<NamedProvider<OutboundSecurityProvider>> outboundProviders = new LinkedList<NamedProvider<OutboundSecurityProvider>>();
        atzProviders.addAll(builder.atzProviders);
        atnProviders.addAll(builder.atnProviders);
        outboundProviders.addAll(builder.outboundProviders);
        builder.auditProviders.forEach(auditProvider -> this.auditors.add(auditProvider.auditConsumer()));
        this.audit(this.instanceUuid, SecurityAuditEvent.info("security.configure", "Security initialized. Providers: audit: \"%s\"; authn: \"%s\"; authz: \"%s\"; identity propagation: \"%s\";").addParam(AuditEvent.AuditParam.plain("auditProviders", SecurityUtil.forAudit(builder.auditProviders))).addParam(AuditEvent.AuditParam.plain("authenticationProvider", SecurityUtil.forAuditNamed(atnProviders))).addParam(AuditEvent.AuditParam.plain("authorizationProvider", SecurityUtil.forAuditNamed(atzProviders))).addParam(AuditEvent.AuditParam.plain("identityPropagationProvider", SecurityUtil.forAuditNamed(outboundProviders))));
        final NamedProvider<AuthenticationProvider> authnProvider = builder.authnProvider;
        final NamedProvider<AuthorizationProvider> authzProvider = builder.authzProvider;
        this.providerSelectionPolicy = builder.providerSelectionPolicy.apply(new ProviderSelectionPolicy.Providers(){

            @Override
            public <T extends SecurityProvider> List<NamedProvider<T>> getProviders(Class<T> providerType) {
                if (providerType.equals(AuthenticationProvider.class)) {
                    LinkedList result = new LinkedList();
                    result.add(authnProvider);
                    atnProviders.stream().filter(pr -> pr != authnProvider).forEach(atn -> result.add((NamedProvider)atn));
                    return result;
                }
                if (providerType.equals(AuthorizationProvider.class)) {
                    LinkedList result = new LinkedList();
                    result.add(authzProvider);
                    atzProviders.stream().filter(pr -> pr != authzProvider).forEach(atn -> result.add((NamedProvider)atn));
                    return result;
                }
                if (providerType.equals(OutboundSecurityProvider.class)) {
                    LinkedList result = new LinkedList();
                    outboundProviders.forEach(atn -> result.add((NamedProvider)atn));
                    return result;
                }
                throw new SecurityException("Security only supports AuthenticationProvider, AuthorizationProvider and OutboundSecurityProvider in provider selection policy, not " + providerType.getName());
            }
        });
    }

    public static Security create(Config config) {
        Objects.requireNonNull(config, "Configuration must not be null");
        return Security.builder().config(config).build();
    }

    public static Builder builder(Config config) {
        Objects.requireNonNull(config, "Configuration must not be null");
        return Security.builder().config(config);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Set<String> getRoles(Subject subject) {
        return subject.grants(Role.class).stream().map(Grant::getName).collect(Collectors.toSet());
    }

    void audit(String tracingId, AuditEvent event) {
        AuditProvider.AuditSource auditSource = AuditProvider.AuditSource.create();
        for (Consumer<AuditProvider.TracedAuditEvent> auditor : this.auditors) {
            auditor.accept(SecurityUtil.wrapEvent(tracingId, auditSource, event));
        }
    }

    public SecurityTime serverTime() {
        return this.serverTime;
    }

    Supplier<ExecutorService> executorService() {
        return this.executorService;
    }

    ProviderSelectionPolicy providerSelectionPolicy() {
        return this.providerSelectionPolicy;
    }

    public SecurityContext.Builder contextBuilder(String id) {
        String newId = null == id || id.isEmpty() ? this.instanceUuid + ":?" : this.instanceUuid + ":" + id;
        return new SecurityContext.Builder(this).id(newId).executorService(this.executorService).tracingTracer(this.securityTracer).serverTime(this.serverTime);
    }

    public SecurityContext createContext(String id) {
        return this.contextBuilder(id).build();
    }

    public Tracer tracer() {
        return this.securityTracer;
    }

    public Collection<Class<? extends Annotation>> customAnnotations() {
        return this.annotations;
    }

    public Config configFor(String child) {
        String test = child.trim();
        if (test.isEmpty()) {
            throw new IllegalArgumentException("Root of security configuration is not available");
        }
        for (String prefix : CONFIG_INTERNAL_PREFIXES) {
            if (!child.equals(prefix) && !child.startsWith(prefix + ".")) continue;
            throw new IllegalArgumentException("Security configuration for " + prefix + " is not available");
        }
        return this.securityConfig.get(child);
    }

    Optional<? extends AuthenticationProvider> resolveAtnProvider(String providerName) {
        return this.resolveProvider(AuthenticationProvider.class, providerName);
    }

    Optional<AuthorizationProvider> resolveAtzProvider(String providerName) {
        return this.resolveProvider(AuthorizationProvider.class, providerName);
    }

    List<? extends OutboundSecurityProvider> resolveOutboundProvider(String providerName) {
        if (null != providerName) {
            return this.resolveProvider(OutboundSecurityProvider.class, providerName).map(List::of).orElse(List.of());
        }
        return this.providerSelectionPolicy.selectOutboundProviders();
    }

    private <T extends SecurityProvider> Optional<T> resolveProvider(Class<T> providerClass, String providerName) {
        if (null == providerName) {
            return this.providerSelectionPolicy.selectProvider(providerClass);
        }
        Optional<T> instance = this.providerSelectionPolicy.selectProvider(providerClass, providerName);
        if (instance.isPresent()) {
            return instance;
        }
        throw new SecurityException("Named " + providerClass.getSimpleName() + " expected for name \"" + providerName + "\" yet none is configured for such a name");
    }

    public SecurityEnvironment.Builder environmentBuilder() {
        return SecurityEnvironment.builder(this.serverTime);
    }

    public Optional<SubjectMappingProvider> subjectMapper() {
        return this.subjectMappingProvider;
    }

    public boolean enabled() {
        return this.enabled;
    }

    public static final class Builder
    implements io.helidon.common.Builder<Security> {
        private final Set<AuditProvider> auditProviders = new LinkedHashSet<AuditProvider>();
        private final List<NamedProvider<AuthenticationProvider>> atnProviders = new LinkedList<NamedProvider<AuthenticationProvider>>();
        private final List<NamedProvider<AuthorizationProvider>> atzProviders = new LinkedList<NamedProvider<AuthorizationProvider>>();
        private final List<NamedProvider<OutboundSecurityProvider>> outboundProviders = new LinkedList<NamedProvider<OutboundSecurityProvider>>();
        private final Map<SecurityProvider, Boolean> allProviders = new IdentityHashMap<SecurityProvider, Boolean>();
        private NamedProvider<AuthenticationProvider> authnProvider;
        private NamedProvider<AuthorizationProvider> authzProvider;
        private SubjectMappingProvider subjectMappingProvider;
        private Config config = Config.empty();
        private Function<ProviderSelectionPolicy.Providers, ProviderSelectionPolicy> providerSelectionPolicy = FirstProviderSelectionPolicy::new;
        private Tracer tracer;
        private boolean tracingEnabled = true;
        private SecurityTime serverTime = SecurityTime.builder().build();
        private Supplier<ExecutorService> executorService = ThreadPoolSupplier.create();
        private boolean enabled = true;
        private Set<String> providerNames = new HashSet<String>();

        private Builder() {
        }

        public Builder providerSelectionPolicy(Function<ProviderSelectionPolicy.Providers, ProviderSelectionPolicy> pspFunction) {
            this.providerSelectionPolicy = pspFunction;
            return this;
        }

        public Builder serverTime(SecurityTime time) {
            this.serverTime = time;
            return this;
        }

        public Builder tracer(Tracer tracer) {
            this.tracer = tracer;
            this.tracingEnabled(null != tracer);
            return this;
        }

        public Builder tracingEnabled(boolean tracingEnabled) {
            this.tracingEnabled = tracingEnabled;
            return this;
        }

        public Builder disableTracing() {
            return this.tracingEnabled(false);
        }

        public Builder addProvider(SecurityProvider provider) {
            return this.addProvider(provider, provider.getClass().getSimpleName());
        }

        public Builder addProvider(Supplier<? extends SecurityProvider> providerBuilder) {
            return this.addProvider(providerBuilder.get());
        }

        public Builder addProvider(SecurityProvider provider, String name) {
            Objects.requireNonNull(provider);
            if (provider instanceof AuthenticationProvider) {
                this.addAuthenticationProvider((AuthenticationProvider)provider, name);
            }
            if (provider instanceof AuthorizationProvider) {
                this.addAuthorizationProvider((AuthorizationProvider)provider, name);
            }
            if (provider instanceof OutboundSecurityProvider) {
                this.addOutboundSecurityProvider((OutboundSecurityProvider)provider, name);
            }
            if (provider instanceof AuditProvider) {
                this.addAuditProvider((AuditProvider)provider);
            }
            if (provider instanceof SubjectMappingProvider) {
                this.subjectMappingProvider((SubjectMappingProvider)provider);
            }
            return this;
        }

        public Builder addProvider(Supplier<? extends SecurityProvider> providerBuilder, String name) {
            return this.addProvider(providerBuilder.get(), name);
        }

        public Builder authenticationProvider(AuthenticationProvider provider) {
            this.authnProvider = new NamedProvider<AuthenticationProvider>(provider.getClass().getSimpleName(), provider);
            return this.addAuthenticationProvider(provider, provider.getClass().getSimpleName());
        }

        public Builder authenticationProvider(Supplier<? extends AuthenticationProvider> builder) {
            return this.authenticationProvider(builder.get());
        }

        public Builder authorizationProvider(AuthorizationProvider provider) {
            this.authzProvider = new NamedProvider<AuthorizationProvider>(provider.getClass().getSimpleName(), provider);
            return this.addAuthorizationProvider(provider, provider.getClass().getSimpleName());
        }

        public Builder authorizationProvider(Supplier<? extends AuthorizationProvider> builder) {
            return this.authorizationProvider(builder.get());
        }

        public Builder addAuthenticationProvider(AuthenticationProvider provider) {
            return this.addAuthenticationProvider(provider, provider.getClass().getSimpleName());
        }

        public Builder addAuthenticationProvider(Supplier<? extends AuthenticationProvider> builder) {
            return this.addAuthenticationProvider(builder.get());
        }

        public Builder addAuthenticationProvider(AuthenticationProvider provider, String name) {
            Objects.requireNonNull(provider);
            NamedProvider<AuthenticationProvider> namedProvider = new NamedProvider<AuthenticationProvider>(name, provider);
            if (null == this.authnProvider) {
                this.authnProvider = namedProvider;
            }
            this.atnProviders.add(namedProvider);
            this.allProviders.put(provider, true);
            if (null != name) {
                this.providerNames.add(name);
            }
            return this;
        }

        public Builder addAuthenticationProvider(Supplier<? extends AuthenticationProvider> builder, String name) {
            return this.addAuthenticationProvider(builder.get(), name);
        }

        public Builder addAuthorizationProvider(AuthorizationProvider provider) {
            return this.addAuthorizationProvider(provider, provider.getClass().getSimpleName());
        }

        public Builder addAuthorizationProvider(Supplier<? extends AuthorizationProvider> builder) {
            return this.addAuthorizationProvider(builder.get());
        }

        public Builder addAuthorizationProvider(AuthorizationProvider provider, String name) {
            Objects.requireNonNull(provider);
            NamedProvider<AuthorizationProvider> namedProvider = new NamedProvider<AuthorizationProvider>(name, provider);
            if (null == this.authzProvider) {
                this.authzProvider = namedProvider;
            }
            this.atzProviders.add(namedProvider);
            this.allProviders.put(provider, true);
            if (null != name) {
                this.providerNames.add(name);
            }
            return this;
        }

        public Builder addAuthorizationProvider(Supplier<? extends AuthorizationProvider> builder, String name) {
            return this.addAuthorizationProvider(builder.get(), name);
        }

        public Builder addOutboundSecurityProvider(OutboundSecurityProvider provider) {
            return this.addOutboundSecurityProvider(provider, provider.getClass().getSimpleName());
        }

        public Builder addOutboundSecurityProvider(Supplier<? extends OutboundSecurityProvider> builder) {
            return this.addOutboundSecurityProvider(builder.get());
        }

        public Builder addOutboundSecurityProvider(Supplier<? extends OutboundSecurityProvider> build, String name) {
            return this.addOutboundSecurityProvider(build.get(), name);
        }

        public Builder addOutboundSecurityProvider(OutboundSecurityProvider provider, String name) {
            Objects.requireNonNull(provider);
            Objects.requireNonNull(name);
            this.outboundProviders.add(new NamedProvider<OutboundSecurityProvider>(name, provider));
            this.allProviders.put(provider, true);
            this.providerNames.add(name);
            return this;
        }

        public Builder addAuditProvider(AuditProvider provider) {
            this.auditProviders.add(provider);
            this.allProviders.put(provider, true);
            return this;
        }

        public Builder subjectMappingProvider(SubjectMappingProvider provider) {
            this.subjectMappingProvider = provider;
            this.allProviders.put(provider, true);
            return this;
        }

        public Builder addAuditProvider(Supplier<? extends AuditProvider> builder) {
            return this.addAuditProvider(builder.get());
        }

        public Builder config(Config config) {
            this.config = config;
            this.fromConfig(config);
            return this;
        }

        public Builder enabled(boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public Security build() {
            if (this.allProviders.isEmpty() && this.enabled) {
                LOGGER.warning("Security component is NOT configured with any security providers.");
            }
            if (this.auditProviders.isEmpty()) {
                DefaultAuditProvider provider = DefaultAuditProvider.create(this.config);
                this.addAuditProvider(provider);
            }
            if (this.atnProviders.isEmpty()) {
                this.addAuthenticationProvider((ProviderRequest context) -> CompletableFuture.completedFuture(AuthenticationResponse.success(SecurityContext.ANONYMOUS)), "default");
            }
            if (this.atzProviders.isEmpty()) {
                this.addAuthorizationProvider(new DefaultAtzProvider(), "default");
            }
            if (!this.enabled) {
                this.providerSelectionPolicy(FirstProviderSelectionPolicy::new);
            }
            return new Security(this);
        }

        private void fromConfig(Config config) {
            String defaultAtzProvider;
            config.get("enabled").asBoolean().ifPresent(this::enabled);
            if (!this.enabled) {
                LOGGER.info("Security is disabled, ignoring provider configuration");
                return;
            }
            config.get("environment.server-time").as(SecurityTime::create).ifPresent(this::serverTime);
            this.executorSupplier((Supplier<ExecutorService>)ThreadPoolSupplier.create((Config)config.get("environment.executor-service")));
            HashMap<String, SecurityProviderService> configKeyToService = new HashMap<String, SecurityProviderService>();
            HashMap<String, SecurityProviderService> classNameToService = new HashMap<String, SecurityProviderService>();
            String knownKeys = this.loadProviderServices(configKeyToService, classNameToService);
            config.get("tracing.enabled").as(Boolean.class).ifPresent(this::tracingEnabled);
            config.get("providers").asList(Config.class).ifPresent(confList -> confList.forEach(pConf -> this.providerFromConfig(configKeyToService, classNameToService, knownKeys, (Config)pConf)));
            String defaultAtnProvider = (String)config.get("default-authentication-provider").asString().orElse(null);
            if (null != defaultAtnProvider) {
                this.authenticationProvider(this.atnProviders.stream().filter(nsp -> nsp.getName().equals(defaultAtnProvider)).findFirst().map(NamedProvider::getProvider).orElseThrow(() -> new SecurityException("Authentication provider named \"" + defaultAtnProvider + "\" is set as default, yet no provider configuration exists")));
            }
            if (null != (defaultAtzProvider = (String)config.get("default-authorization-provider").asString().orElse(null))) {
                this.authorizationProvider(this.atzProviders.stream().filter(nsp -> nsp.getName().equals(defaultAtzProvider)).findFirst().map(NamedProvider::getProvider).orElseThrow(() -> new SecurityException("Authorization provider named \"" + defaultAtzProvider + "\" is set as default, yet no provider configuration exists")));
            }
            config = config.get("provider-policy");
            ProviderSelectionPolicyType pType = config.get("type").asString().map(ProviderSelectionPolicyType::valueOf).orElse(ProviderSelectionPolicyType.FIRST);
            switch (pType) {
                case FIRST: {
                    this.providerSelectionPolicy = FirstProviderSelectionPolicy::new;
                    break;
                }
                case COMPOSITE: {
                    this.providerSelectionPolicy = CompositeProviderSelectionPolicy.create(config);
                    break;
                }
                case CLASS: {
                    this.providerSelectionPolicy = this.findProviderSelectionPolicy(config);
                    break;
                }
                default: {
                    throw new IllegalStateException("Invalid enum option: " + pType + ", probably version mis-match");
                }
            }
        }

        private void providerFromConfig(Map<String, SecurityProviderService> configKeyToService, Map<String, SecurityProviderService> classNameToService, String knownKeys, Config pConf) {
            AtomicReference<SecurityProviderService> service = new AtomicReference<SecurityProviderService>();
            AtomicReference<Config> providerSpecific = new AtomicReference<Config>();
            String className = (String)pConf.get("class").asString().orElse(null);
            if (null == className) {
                this.findProviderService(configKeyToService, knownKeys, pConf, service, providerSpecific);
            } else {
                SecurityProviderService providerService = classNameToService.get(className);
                if (null == providerService) {
                    this.findProviderSpecificConfig(pConf, providerSpecific);
                } else {
                    service.set(providerService);
                    providerSpecific.set(pConf.get(providerService.providerConfigKey()));
                }
            }
            Config providerSpecificConfig = providerSpecific.get();
            SecurityProviderService providerService = service.get();
            if (null == className && null == providerService) {
                throw new SecurityException("Each configured provider MUST have a \"class\" configuration property defined or a custom configuration section mapped to that provider, supported keys: " + knownKeys);
            }
            String name = this.resolveProviderName(pConf, className, providerSpecificConfig, providerService);
            boolean isAuthn = (Boolean)pConf.get("is-authentication-provider").asBoolean().orElse((Object)true);
            boolean isAuthz = (Boolean)pConf.get("is-authorization-provider").asBoolean().orElse((Object)true);
            boolean isClientSec = (Boolean)pConf.get("is-client-security-provider").asBoolean().orElse((Object)true);
            boolean isAudit = (Boolean)pConf.get("is-audit-provider").asBoolean().orElse((Object)true);
            boolean isSubjectMapper = (Boolean)pConf.get("is-subject-mapper").asBoolean().orElse((Object)true);
            SecurityProvider provider = null == providerService ? SecurityUtil.instantiate(className, SecurityProvider.class, providerSpecificConfig) : providerService.providerInstance(providerSpecificConfig);
            if (isAuthn && provider instanceof AuthenticationProvider) {
                this.addAuthenticationProvider((AuthenticationProvider)provider, name);
            }
            if (isAuthz && provider instanceof AuthorizationProvider) {
                this.addAuthorizationProvider((AuthorizationProvider)provider, name);
            }
            if (isClientSec && provider instanceof OutboundSecurityProvider) {
                this.addOutboundSecurityProvider((OutboundSecurityProvider)provider, name);
            }
            if (isAudit && provider instanceof AuditProvider) {
                this.addAuditProvider((AuditProvider)provider);
            }
            if (isSubjectMapper && provider instanceof SubjectMappingProvider) {
                this.subjectMappingProvider((SubjectMappingProvider)provider);
            }
        }

        private void executorSupplier(Supplier<ExecutorService> supplier) {
            this.executorService = supplier;
        }

        private String resolveProviderName(Config pConf, String className, Config providerSpecificConfig, SecurityProviderService providerService) {
            return (String)pConf.get("name").asString().orElseGet(() -> {
                if (null == providerSpecificConfig) {
                    if (null == className) {
                        return providerService.providerClass().getSimpleName();
                    }
                    int index = className.indexOf(46);
                    if (index > -1) {
                        return className.substring(index + 1);
                    }
                    return className;
                }
                return providerSpecificConfig.name();
            });
        }

        private void findProviderSpecificConfig(Config pConf, AtomicReference<Config> providerSpecific) {
            ((List)pConf.asNodeList().get()).stream().filter(this::notReservedProviderKey).forEach(providerSpecificConf -> {
                if (!providerSpecific.compareAndSet((Config)null, (Config)providerSpecificConf)) {
                    throw new SecurityException("More than one provider configurations found, each provider can only have one provide specific config. Conflict: " + ((Config)providerSpecific.get()).key() + " and " + providerSpecificConf.key());
                }
            });
        }

        private void findProviderService(Map<String, SecurityProviderService> configKeyToService, String knownKeys, Config pConf, AtomicReference<SecurityProviderService> service, AtomicReference<Config> providerSpecific) {
            ((List)pConf.asNodeList().get()).stream().filter(this::notReservedProviderKey).forEach(providerSpecificConf -> {
                if (!providerSpecific.compareAndSet((Config)null, (Config)providerSpecificConf)) {
                    throw new SecurityException("More than one provider configurations found, each provider can only have one provider specific config. Conflict: " + ((Config)providerSpecific.get()).key() + " and " + providerSpecificConf.key());
                }
                String keyName = providerSpecificConf.name();
                if (!configKeyToService.containsKey(keyName)) {
                    throw new SecurityException("Configuration key " + providerSpecificConf.key() + " is not a valid provider configuration. Supported keys: " + knownKeys);
                }
                service.set((SecurityProviderService)configKeyToService.get(keyName));
            });
        }

        private String loadProviderServices(Map<String, SecurityProviderService> configKeyToService, Map<String, SecurityProviderService> classNameToService) {
            HashSet configKeys = new HashSet();
            HelidonServiceLoader loader = HelidonServiceLoader.create(ServiceLoader.load(SecurityProviderService.class));
            loader.forEach(service -> {
                String configKey = service.providerConfigKey();
                if (null != configKey) {
                    configKeyToService.put(configKey, (SecurityProviderService)service);
                    configKeys.add(configKey);
                }
                Class<? extends SecurityProvider> theClass = service.providerClass();
                classNameToService.put(theClass.getName(), (SecurityProviderService)service);
            });
            return String.join((CharSequence)", ", configKeys);
        }

        private boolean notReservedProviderKey(Config config) {
            return !RESERVED_PROVIDER_KEYS.contains(config.name());
        }

        private Function<ProviderSelectionPolicy.Providers, ProviderSelectionPolicy> findProviderSelectionPolicy(Config config) {
            Class clazz = (Class)config.get("class-name").as(Class.class).orElseThrow(() -> new java.lang.SecurityException("You have configured a CLASS provider selection without configuring class-name"));
            if (!ProviderSelectionPolicy.class.isAssignableFrom(clazz)) {
                throw new SecurityException("Class " + clazz.getName() + " does not implement ProviderSelectionPolicy");
            }
            Class pspClass = clazz;
            try {
                Constructor constructor = pspClass.getConstructor(ProviderSelectionPolicy.Providers.class, Config.class);
                if (ReflectionUtil.canAccess(this.getClass(), constructor)) {
                    return providers -> {
                        try {
                            return (ProviderSelectionPolicy)constructor.newInstance(providers, config);
                        }
                        catch (Exception e) {
                            throw new SecurityException("Failed to instantiate ProviderSelectionPolicy", e);
                        }
                    };
                }
                throw new SecurityException("Constructor " + constructor + " of class " + clazz.getName() + " is not accessible");
            }
            catch (NoSuchMethodException constructor) {
                try {
                    Constructor constructor2 = pspClass.getConstructor(ProviderSelectionPolicy.Providers.class);
                    if (ReflectionUtil.canAccess(this.getClass(), constructor2)) {
                        return providers -> {
                            try {
                                return (ProviderSelectionPolicy)constructor2.newInstance(providers);
                            }
                            catch (Exception e) {
                                throw new SecurityException("Failed to instantiate ProviderSelectionPolicy", e);
                            }
                        };
                    }
                    throw new SecurityException("Constructor " + constructor2 + " of class " + clazz.getName() + " is not accessible");
                }
                catch (NoSuchMethodException e) {
                    throw new SecurityException("You have configured " + clazz.getName() + " as provider selection policy class, yet it is missing public constructor with Providers or Providers and Config as parameters.", e);
                }
            }
        }

        public boolean noProvider(Class<? extends SecurityProvider> providerClass) {
            if (providerClass.equals(AuthenticationProvider.class)) {
                return this.atnProviders.isEmpty();
            }
            if (providerClass.equals(AuthorizationProvider.class)) {
                return this.atzProviders.isEmpty();
            }
            if (providerClass.equals(OutboundSecurityProvider.class)) {
                return this.outboundProviders.isEmpty();
            }
            if (providerClass.equals(AuditProvider.class)) {
                return this.auditProviders.isEmpty();
            }
            if (providerClass.equals(SubjectMappingProvider.class)) {
                return this.subjectMappingProvider == null;
            }
            return this.allProviders.isEmpty();
        }

        public boolean hasProvider(String name) {
            return this.providerNames.stream().anyMatch(name::equals);
        }

        private static class DefaultAtzProvider
        implements AuthorizationProvider {
            private DefaultAtzProvider() {
            }

            @Override
            public CompletionStage<AuthorizationResponse> authorize(ProviderRequest context) {
                return CompletableFuture.completedFuture(AuthorizationResponse.permit());
            }
        }
    }
}

