/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.build.util;

import io.helidon.build.util.Constants;
import io.helidon.build.util.Log;
import io.helidon.build.util.Requirements;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public final class FileUtils {
    public static final Path WORKING_DIR = FileUtils.requiredDirectoryFromProperty("user.dir", false);
    public static final Path USER_HOME_DIR = FileUtils.requiredDirectoryFromProperty("user.home", false);
    private static final String JAVA_BINARY_NAME = Constants.OS.javaExecutable();
    private static final String JAVA_HOME_VAR = "JAVA_HOME";
    private static final String PATH_VAR = "PATH";
    private static final String BIN_DIR_NAME = "bin";

    public static Path requiredDirectoryFromProperty(String systemPropertyName, boolean createIfRequired) {
        String path = Requirements.requireNonNull(System.getProperty(systemPropertyName), "Required system property %s not set", systemPropertyName);
        return FileUtils.requiredDirectory(path, createIfRequired);
    }

    public static Path requiredDirectory(String path, boolean createIfRequired) {
        Path dir = Path.of(Objects.requireNonNull(path, "valid path required"), new String[0]);
        return createIfRequired ? FileUtils.ensureDirectory(dir, new FileAttribute[0]) : FileUtils.assertDir(dir);
    }

    public static Path fromWorking(Path path) {
        try {
            Path relativePath = WORKING_DIR.relativize(path);
            if (relativePath.getName(0).toString().equals("..")) {
                return path;
            }
            return relativePath;
        }
        catch (IllegalArgumentException e) {
            return path;
        }
    }

    public static Path ensureDirectory(Path path, FileAttribute<?> ... attrs) {
        if (Files.exists(Objects.requireNonNull(path), new LinkOption[0])) {
            return FileUtils.assertDir(path);
        }
        try {
            return Files.createDirectories(path, attrs);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Path copyDirectory(Path source, Path destination) {
        FileUtils.assertDoesNotExist(destination);
        try (Stream<Path> stream = Files.walk(source, new FileVisitOption[0]);){
            stream.forEach(src -> {
                try {
                    Path dst = destination.resolve(source.relativize((Path)src));
                    if (Files.isDirectory(src, new LinkOption[0])) {
                        Files.createDirectory(dst, new FileAttribute[0]);
                    } else {
                        Files.copy(src, dst, StandardCopyOption.COPY_ATTRIBUTES);
                    }
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
        }
        catch (UncheckedIOException e) {
            throw e;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return destination.toAbsolutePath().normalize();
    }

    public static List<Path> listFiles(Path directory, Predicate<String> fileNameFilter) {
        return FileUtils.listFiles(directory, fileNameFilter, 1);
    }

    public static List<Path> listFiles(Path directory, BiPredicate<Path, BasicFileAttributes> pathFilter) {
        return FileUtils.listFiles(directory, pathFilter, 1);
    }

    public static List<Path> listFiles(Path directory, Predicate<String> fileNameFilter, int maxDepth) {
        return FileUtils.listFiles(directory, (Path path, BasicFileAttributes attrs) -> fileNameFilter.test(path.getFileName().toString()), maxDepth);
    }

    public static List<Path> listFiles(Path directory, BiPredicate<Path, BasicFileAttributes> pathFilter, int maxDepth) {
        try {
            return Files.find(FileUtils.assertDir(directory), maxDepth, pathFilter, new FileVisitOption[0]).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static List<Path> list(Path directory) {
        return FileUtils.list(directory, 1);
    }

    public static List<Path> list(Path directory, int maxDepth) {
        try {
            return Files.find(FileUtils.assertDir(directory), maxDepth, (path, attrs) -> true, new FileVisitOption[0]).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Path assertDir(Path directory) {
        Path result = FileUtils.assertExists(directory);
        if (Files.isDirectory(result, new LinkOption[0])) {
            return result;
        }
        throw new IllegalArgumentException(directory + " is not a directory");
    }

    public static Path assertFile(Path file) {
        Path result = FileUtils.assertExists(file);
        if (Files.isRegularFile(result, new LinkOption[0])) {
            return result;
        }
        throw new IllegalArgumentException(file + " is not a file");
    }

    public static Path assertExists(Path path) {
        if (Files.exists(Objects.requireNonNull(path), new LinkOption[0])) {
            return path.toAbsolutePath().normalize();
        }
        throw new IllegalArgumentException(path + " does not exist");
    }

    public static Path assertDoesNotExist(Path path) {
        if (Files.exists(Objects.requireNonNull(path), new LinkOption[0])) {
            throw new IllegalArgumentException(path + " exists");
        }
        return path.toAbsolutePath().normalize();
    }

    public static Path delete(Path fileOrDirectory) throws IOException {
        if (Files.exists(fileOrDirectory, new LinkOption[0])) {
            if (Files.isRegularFile(fileOrDirectory, new LinkOption[0])) {
                Files.delete(fileOrDirectory);
            } else {
                FileUtils.deleteDirectory(fileOrDirectory);
            }
        }
        return fileOrDirectory;
    }

    public static Path deleteDirectory(Path directory) throws IOException {
        if (Files.exists(directory, new LinkOption[0])) {
            if (Files.isDirectory(directory, new LinkOption[0])) {
                try (Stream<Path> stream = Files.walk(directory, new FileVisitOption[0]);){
                    stream.sorted(Comparator.reverseOrder()).forEach(file -> {
                        try {
                            Files.delete(file);
                        }
                        catch (IOException e) {
                            throw new UncheckedIOException(e);
                        }
                    });
                }
            } else {
                throw new IllegalArgumentException(directory + " is not a directory");
            }
        }
        return directory;
    }

    public static Path deleteDirectoryContent(Path directory) throws IOException {
        if (Files.exists(directory, new LinkOption[0])) {
            if (Files.isDirectory(directory, new LinkOption[0])) {
                try (Stream<Path> stream = Files.walk(directory, new FileVisitOption[0]);){
                    stream.sorted(Comparator.reverseOrder()).filter(file -> !file.equals(directory)).forEach(file -> {
                        try {
                            Files.delete(file);
                        }
                        catch (IOException e) {
                            throw new UncheckedIOException(e);
                        }
                    });
                }
            } else {
                throw new IllegalArgumentException(directory + " is not a directory");
            }
        }
        return directory;
    }

    public static long sizeOf(Path path) {
        try {
            if (Files.isRegularFile(path, new LinkOption[0])) {
                return Files.size(path);
            }
            final AtomicLong size = new AtomicLong();
            Files.walkFileTree(path, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    size.addAndGet(attrs.size());
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }
            });
            return size.get();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static long lastModifiedSeconds(Path file) {
        return FileUtils.lastModifiedTime(file).to(TimeUnit.SECONDS);
    }

    public static long lastModifiedMillis(Path file) {
        return FileUtils.lastModifiedTime(file).to(TimeUnit.MILLISECONDS);
    }

    public static FileTime lastModifiedTime(Path file) {
        try {
            return Files.getLastModifiedTime(file, new LinkOption[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Optional<FileTime> newerThan(Path file, FileTime baseTime) {
        FileTime modTime = FileUtils.lastModifiedTime(file);
        if (FileUtils.newerThan(modTime, baseTime)) {
            return Optional.of(modTime);
        }
        return Optional.empty();
    }

    public static Optional<FileTime> olderThan(Path file, FileTime baseTime) {
        FileTime modTime = FileUtils.lastModifiedTime(file);
        if (FileUtils.olderThan(modTime, baseTime)) {
            return Optional.of(modTime);
        }
        return Optional.empty();
    }

    public static boolean newerThan(FileTime changeTime, FileTime baseTime) {
        return baseTime == null || changeTime.compareTo(baseTime) > 0;
    }

    public static boolean olderThan(FileTime changeTime, FileTime baseTime) {
        return baseTime == null || changeTime.compareTo(baseTime) < 0;
    }

    public static String fileName(Path file) {
        return Objects.requireNonNull(file.getFileName()).toString();
    }

    public static Optional<Path> findExecutableInPath(String executableName) {
        return Arrays.stream(Objects.requireNonNull(System.getenv(PATH_VAR)).split(File.pathSeparator)).map(x$0 -> Paths.get(x$0, new String[0])).map(path -> path.resolve(executableName)).filter(path -> !Constants.OS.isPosix() && Files.exists(path, new LinkOption[0]) || Files.isExecutable(path)).findFirst();
    }

    public static Optional<Path> javaExecutable() {
        Optional<Path> path = FileUtils.javaExecutableInPath();
        if (path.isPresent()) {
            return path;
        }
        return FileUtils.javaExecutableInJavaHome();
    }

    public static Path assertJavaExecutable() {
        return FileUtils.javaExecutable().orElseThrow(() -> new IllegalStateException(JAVA_BINARY_NAME + " not found. Please add it to your PATH or set the JAVA_HOME or variable."));
    }

    public static Optional<Path> javaExecutableInPath() {
        return FileUtils.findExecutableInPath(JAVA_BINARY_NAME);
    }

    public static Optional<Path> javaExecutableInJavaHome() {
        String javaHomePath = System.getenv(JAVA_HOME_VAR);
        if (javaHomePath != null) {
            Path javaHome = Paths.get(javaHomePath, new String[0]);
            Path binary = javaHome.resolve(BIN_DIR_NAME).resolve(JAVA_BINARY_NAME);
            if (!Constants.OS.isPosix() || Files.isExecutable(binary)) {
                return Optional.of(binary);
            }
            throw new IllegalStateException(JAVA_BINARY_NAME + " not found in JAVA_HOME path: " + javaHomePath);
        }
        return Optional.empty();
    }

    public static Path ensureFile(Path file) {
        if (!Files.exists(file, new LinkOption[0])) {
            try {
                Files.createFile(file, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return file;
    }

    public static Path touch(Path file) {
        if (Files.exists(file, new LinkOption[0])) {
            long currentTime = System.currentTimeMillis();
            long lastModified = FileUtils.lastModifiedSeconds(file);
            long lastModifiedPlusOneSecond = lastModified + 1000L;
            long newTime = Math.max(currentTime, lastModifiedPlusOneSecond);
            try {
                Files.setLastModifiedTime(file, FileTime.fromMillis(newTime));
                return file;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return FileUtils.ensureFile(file);
    }

    public static Optional<FileTime> changedSince(Path directory, FileTime baseTime, final Predicate<Path> dirFilter, final Predicate<Path> fileFilter, ChangeDetectionType type) {
        FileTime base = baseTime == null ? FileTime.fromMillis(0L) : baseTime;
        final AtomicReference<FileTime> checkTime = new AtomicReference<FileTime>(base);
        final AtomicReference changeTime = new AtomicReference();
        final boolean checkAllFiles = type == ChangeDetectionType.LATEST;
        Log.debug("Checking if project has files newer than last check time %s", checkTime.get());
        try {
            Files.walkFileTree(directory, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                    return dirFilter.test(dir) ? FileVisitResult.CONTINUE : FileVisitResult.SKIP_SUBTREE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    FileTime fileTime;
                    if (fileFilter.test(file) && (fileTime = FileUtils.lastModifiedTime(file)).compareTo((FileTime)checkTime.get()) > 0) {
                        Log.debug("%s @ %s is newer than last check time %s", file, fileTime, checkTime.get());
                        changeTime.set(fileTime);
                        if (checkAllFiles) {
                            checkTime.set(fileTime);
                        } else {
                            return FileVisitResult.TERMINATE;
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    changeTime.set(null);
                    return FileVisitResult.TERMINATE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }
            });
            return Optional.ofNullable((FileTime)changeTime.get());
        }
        catch (Exception e) {
            Log.warn(e.getMessage(), new Object[0]);
            return Optional.of(FileTime.fromMillis(System.currentTimeMillis()));
        }
    }

    private FileUtils() {
    }

    public static enum ChangeDetectionType {
        FIRST,
        LATEST;

    }
}

