/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.management.service.impl;

import io.gravitee.management.model.ApiKeyEntity;
import io.gravitee.management.model.ApiModelEntity;
import io.gravitee.management.model.ApplicationEntity;
import io.gravitee.management.model.PlanEntity;
import io.gravitee.management.model.PrimaryOwnerEntity;
import io.gravitee.management.model.SubscriptionEntity;
import io.gravitee.management.model.SubscriptionStatus;
import io.gravitee.management.service.ApiKeyGenerator;
import io.gravitee.management.service.ApiKeyService;
import io.gravitee.management.service.ApiService;
import io.gravitee.management.service.ApplicationService;
import io.gravitee.management.service.AuditService;
import io.gravitee.management.service.NotifierService;
import io.gravitee.management.service.PlanService;
import io.gravitee.management.service.SubscriptionService;
import io.gravitee.management.service.exceptions.ApiKeyAlreadyActivatedException;
import io.gravitee.management.service.exceptions.ApiKeyAlreadyExpiredException;
import io.gravitee.management.service.exceptions.ApiKeyNotFoundException;
import io.gravitee.management.service.exceptions.SubscriptionClosedException;
import io.gravitee.management.service.exceptions.SubscriptionNotActiveException;
import io.gravitee.management.service.exceptions.TechnicalManagementException;
import io.gravitee.management.service.impl.TransactionalService;
import io.gravitee.management.service.notification.ApiHook;
import io.gravitee.management.service.notification.NotificationParamsBuilder;
import io.gravitee.repository.exceptions.TechnicalException;
import io.gravitee.repository.management.api.ApiKeyRepository;
import io.gravitee.repository.management.model.ApiKey;
import io.gravitee.repository.management.model.Audit;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class ApiKeyServiceImpl
extends TransactionalService
implements ApiKeyService {
    private final Logger LOGGER = LoggerFactory.getLogger(ApiKeyServiceImpl.class);
    @Autowired
    private ApiKeyRepository apiKeyRepository;
    @Autowired
    private SubscriptionService subscriptionService;
    @Autowired
    private ApiKeyGenerator apiKeyGenerator;
    @Autowired
    private ApplicationService applicationService;
    @Autowired
    private ApiService apiService;
    @Autowired
    private PlanService planService;
    @Autowired
    private AuditService auditService;
    @Autowired
    private NotifierService notifierService;

    @Override
    public ApiKeyEntity generate(String subscription) {
        try {
            this.LOGGER.debug("Generate an API Key for subscription {}", (Object)subscription);
            ApiKey apiKey = this.generateForSubscription(subscription);
            apiKey = this.apiKeyRepository.create(apiKey);
            PlanEntity plan = this.planService.findById(apiKey.getPlan());
            LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
            properties.put(Audit.AuditProperties.API_KEY, apiKey.getKey());
            properties.put(Audit.AuditProperties.API, (String)plan.getApis().iterator().next());
            properties.put(Audit.AuditProperties.APPLICATION, apiKey.getApplication());
            this.auditService.createApiAuditLog((String)plan.getApis().iterator().next(), properties, (Audit.AuditEvent)ApiKey.AuditEvent.APIKEY_CREATED, apiKey.getCreatedAt(), null, apiKey);
            return ApiKeyServiceImpl.convert(apiKey);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to generate an API Key for {} - {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to generate an API Key for %s", subscription), ex);
        }
    }

    @Override
    public ApiKeyEntity renew(String subscription) {
        try {
            this.LOGGER.debug("Renew API Key for subscription {}", (Object)subscription);
            ApiKey newApiKey = this.generateForSubscription(subscription);
            newApiKey = this.apiKeyRepository.create(newApiKey);
            Instant expirationInst = newApiKey.getCreatedAt().toInstant().plus(Duration.ofHours(2L));
            Date expirationDate = Date.from(expirationInst);
            Set oldKeys = this.apiKeyRepository.findBySubscription(subscription);
            for (ApiKey oldKey : oldKeys) {
                if (oldKey.equals((Object)newApiKey) || ApiKeyServiceImpl.convert(oldKey).isExpired()) continue;
                this.setExpiration(expirationDate, oldKey);
            }
            PlanEntity plan = this.planService.findById(newApiKey.getPlan());
            LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
            properties.put(Audit.AuditProperties.API_KEY, newApiKey.getKey());
            properties.put(Audit.AuditProperties.API, (String)plan.getApis().iterator().next());
            properties.put(Audit.AuditProperties.APPLICATION, newApiKey.getApplication());
            this.auditService.createApiAuditLog((String)plan.getApis().iterator().next(), properties, (Audit.AuditEvent)ApiKey.AuditEvent.APIKEY_RENEWED, newApiKey.getCreatedAt(), null, newApiKey);
            ApplicationEntity application = this.applicationService.findById(newApiKey.getApplication());
            ApiModelEntity api = this.apiService.findByIdForTemplates((String)plan.getApis().iterator().next());
            PrimaryOwnerEntity owner = application.getPrimaryOwner();
            Map<String, Object> params = new NotificationParamsBuilder().application(application).plan(plan).api(api).owner(owner).apikey(newApiKey).build();
            this.notifierService.trigger(ApiHook.APIKEY_RENEWED, (String)plan.getApis().iterator().next(), params);
            return ApiKeyServiceImpl.convert(newApiKey);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to renew an API Key for {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to renew an API Key for %s", subscription), ex);
        }
    }

    private ApiKey generateForSubscription(String subscription) {
        SubscriptionEntity subscriptionEntity = this.subscriptionService.findById(subscription);
        Date now = new Date();
        if (subscriptionEntity.getEndingAt() != null && subscriptionEntity.getEndingAt().before(now)) {
            throw new SubscriptionClosedException(subscription);
        }
        ApiKey apiKey = new ApiKey();
        apiKey.setSubscription(subscription);
        apiKey.setApplication(subscriptionEntity.getApplication());
        apiKey.setPlan(subscriptionEntity.getPlan());
        apiKey.setCreatedAt(new Date());
        apiKey.setUpdatedAt(apiKey.getCreatedAt());
        apiKey.setKey(this.apiKeyGenerator.generate());
        apiKey.setExpireAt(subscriptionEntity.getEndingAt());
        return apiKey;
    }

    @Override
    public void revoke(String apiKey, boolean notify) {
        try {
            this.LOGGER.debug("Revoke API Key {}", (Object)apiKey);
            Optional optKey = this.apiKeyRepository.findById(apiKey);
            if (!optKey.isPresent()) {
                throw new ApiKeyNotFoundException();
            }
            ApiKey key = (ApiKey)optKey.get();
            this.checkApiKeyExpired(key);
            ApiKey previousApiKey = new ApiKey(key);
            key.setRevoked(true);
            key.setUpdatedAt(new Date());
            key.setRevokedAt(key.getUpdatedAt());
            this.apiKeyRepository.update(key);
            PlanEntity plan = this.planService.findById(key.getPlan());
            LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
            properties.put(Audit.AuditProperties.API_KEY, key.getKey());
            properties.put(Audit.AuditProperties.API, (String)plan.getApis().iterator().next());
            properties.put(Audit.AuditProperties.APPLICATION, key.getApplication());
            this.auditService.createApiAuditLog((String)plan.getApis().iterator().next(), properties, (Audit.AuditEvent)ApiKey.AuditEvent.APIKEY_REVOKED, key.getUpdatedAt(), previousApiKey, key);
            if (notify) {
                ApplicationEntity application = this.applicationService.findById(key.getApplication());
                ApiModelEntity api = this.apiService.findByIdForTemplates((String)plan.getApis().iterator().next());
                PrimaryOwnerEntity owner = application.getPrimaryOwner();
                Map<String, Object> params = new NotificationParamsBuilder().application(application).plan(plan).api(api).owner(owner).apikey(key).build();
                this.notifierService.trigger(ApiHook.APIKEY_REVOKED, api.getId(), params);
            }
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to revoke a key {}", (Object)apiKey, (Object)ex);
            throw new TechnicalManagementException("An error occurs while trying to revoke a key " + apiKey, ex);
        }
    }

    @Override
    public ApiKeyEntity reactivate(String apiKey) {
        try {
            this.LOGGER.debug("Reactivate API Key {}", (Object)apiKey);
            Optional optKey = this.apiKeyRepository.findById(apiKey);
            if (!optKey.isPresent()) {
                throw new ApiKeyNotFoundException();
            }
            ApiKey key = (ApiKey)optKey.get();
            if (!key.isRevoked() && !ApiKeyServiceImpl.convert(key).isExpired()) {
                throw new ApiKeyAlreadyActivatedException("The API key is already activated");
            }
            SubscriptionEntity subscription = this.subscriptionService.findById(key.getSubscription());
            if (subscription.getStatus() != SubscriptionStatus.PAUSED && subscription.getStatus() != SubscriptionStatus.ACCEPTED) {
                throw new SubscriptionNotActiveException(subscription);
            }
            ApiKey previousApiKey = new ApiKey(key);
            key.setRevoked(false);
            key.setUpdatedAt(new Date());
            key.setRevokedAt(null);
            key.setExpireAt(subscription.getEndingAt());
            ApiKey updated = this.apiKeyRepository.update(key);
            LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
            properties.put(Audit.AuditProperties.API_KEY, key.getKey());
            properties.put(Audit.AuditProperties.API, subscription.getApi());
            properties.put(Audit.AuditProperties.APPLICATION, key.getApplication());
            this.auditService.createApiAuditLog(subscription.getApi(), properties, (Audit.AuditEvent)ApiKey.AuditEvent.APIKEY_REACTIVATED, key.getUpdatedAt(), previousApiKey, key);
            return ApiKeyServiceImpl.convert(updated);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to reactivate a key {}", (Object)apiKey, (Object)ex);
            throw new TechnicalManagementException("An error occurs while trying to reactivate a key " + apiKey, ex);
        }
    }

    private void checkApiKeyExpired(ApiKey key) {
        if (key.isRevoked() || ApiKeyServiceImpl.convert(key).isExpired()) {
            throw new ApiKeyAlreadyExpiredException("The API key is already expired");
        }
    }

    @Override
    public List<ApiKeyEntity> findBySubscription(String subscription) {
        try {
            this.LOGGER.debug("Find API Keys for subscription {}", (Object)subscription);
            SubscriptionEntity subscriptionEntity = this.subscriptionService.findById(subscription);
            Set keys = this.apiKeyRepository.findBySubscription(subscriptionEntity.getId());
            return keys.stream().map(ApiKeyServiceImpl::convert).sorted((o1, o2) -> o2.getCreatedAt().compareTo(o1.getCreatedAt())).collect(Collectors.toList());
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while finding API keys for subscription {}", (Object)subscription, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while finding API keys for subscription %s", subscription), ex);
        }
    }

    @Override
    public ApiKeyEntity findByKey(String apiKey) {
        try {
            this.LOGGER.debug("Find an API Key by key: {}", (Object)apiKey);
            Optional optApiKey = this.apiKeyRepository.findById(apiKey);
            if (optApiKey.isPresent()) {
                return ApiKeyServiceImpl.convert((ApiKey)optApiKey.get());
            }
            throw new ApiKeyNotFoundException();
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to find an API Key by key {}", (Object)apiKey, (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while trying to find an API Key by key: %s", apiKey), ex);
        }
    }

    @Override
    public ApiKeyEntity update(ApiKeyEntity apiKeyEntity) {
        try {
            this.LOGGER.debug("Update API Key {}", (Object)apiKeyEntity.getKey());
            Optional optKey = this.apiKeyRepository.findById(apiKeyEntity.getKey());
            if (!optKey.isPresent()) {
                throw new ApiKeyNotFoundException();
            }
            ApiKey key = (ApiKey)optKey.get();
            this.checkApiKeyExpired(key);
            key.setPaused(apiKeyEntity.isPaused());
            key.setPlan(apiKeyEntity.getPlan());
            if (apiKeyEntity.getExpireAt() != null) {
                this.setExpiration(apiKeyEntity.getExpireAt(), key);
            } else {
                key.setUpdatedAt(new Date());
                this.apiKeyRepository.update(key);
            }
            return ApiKeyServiceImpl.convert(key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while updating an API Key {}", (Object)apiKeyEntity.getKey(), (Object)ex);
            throw new TechnicalManagementException(String.format("An error occurs while updating an API Key %s", apiKeyEntity.getKey()), ex);
        }
    }

    @Override
    public void delete(String apiKey) {
    }

    private void setExpiration(Date expirationDate, ApiKey key) throws TechnicalException {
        Date now = new Date();
        if (now.after(expirationDate)) {
            expirationDate = now;
        }
        key.setUpdatedAt(now);
        if (!key.isRevoked()) {
            SubscriptionEntity subscription = this.subscriptionService.findById(key.getSubscription());
            if (subscription.getEndingAt() != null && (expirationDate == null || subscription.getEndingAt().compareTo(expirationDate) < 0)) {
                expirationDate = subscription.getEndingAt();
            }
            ApiKey oldkey = new ApiKey(key);
            key.setExpireAt(expirationDate);
            this.apiKeyRepository.update(key);
            ApplicationEntity application = this.applicationService.findById(key.getApplication());
            PlanEntity plan = this.planService.findById(key.getPlan());
            ApiModelEntity api = this.apiService.findByIdForTemplates((String)plan.getApis().iterator().next());
            PrimaryOwnerEntity owner = application.getPrimaryOwner();
            NotificationParamsBuilder paramsBuilder = new NotificationParamsBuilder();
            paramsBuilder.api(api).application(application).apikey(key).plan(plan).owner(owner);
            if (key.getExpireAt() != null && now.before(key.getExpireAt())) {
                paramsBuilder.expirationDate(key.getExpireAt());
            }
            Map<String, Object> params = paramsBuilder.build();
            this.notifierService.trigger(ApiHook.APIKEY_EXPIRED, api.getId(), params);
            LinkedHashMap<Audit.AuditProperties, String> properties = new LinkedHashMap<Audit.AuditProperties, String>();
            properties.put(Audit.AuditProperties.API_KEY, key.getKey());
            properties.put(Audit.AuditProperties.API, api.getId());
            properties.put(Audit.AuditProperties.APPLICATION, application.getId());
            this.auditService.createApiAuditLog((String)plan.getApis().iterator().next(), properties, (Audit.AuditEvent)ApiKey.AuditEvent.APIKEY_EXPIRED, key.getUpdatedAt(), oldkey, key);
        } else {
            this.apiKeyRepository.update(key);
        }
    }

    private static ApiKeyEntity convert(ApiKey apiKey) {
        ApiKeyEntity apiKeyEntity = new ApiKeyEntity();
        apiKeyEntity.setKey(apiKey.getKey());
        apiKeyEntity.setCreatedAt(apiKey.getCreatedAt());
        apiKeyEntity.setExpireAt(apiKey.getExpireAt());
        apiKeyEntity.setExpired(apiKey.getExpireAt() != null && new Date().after(apiKey.getExpireAt()));
        apiKeyEntity.setRevoked(apiKey.isRevoked());
        apiKeyEntity.setRevokedAt(apiKey.getRevokedAt());
        apiKeyEntity.setUpdatedAt(apiKey.getUpdatedAt());
        apiKeyEntity.setSubscription(apiKey.getSubscription());
        apiKeyEntity.setApplication(apiKey.getApplication());
        apiKeyEntity.setPlan(apiKey.getPlan());
        return apiKeyEntity;
    }
}

