/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.management.service.impl;

import com.google.common.collect.ImmutableMap;
import io.gravitee.management.model.ApiKeyEntity;
import io.gravitee.management.model.ApplicationEntity;
import io.gravitee.management.model.PrimaryOwnerEntity;
import io.gravitee.management.service.ApiKeyGenerator;
import io.gravitee.management.service.ApiKeyService;
import io.gravitee.management.service.ApplicationService;
import io.gravitee.management.service.EmailService;
import io.gravitee.management.service.builder.EmailNotificationBuilder;
import io.gravitee.management.service.exceptions.ApiKeyNotFoundException;
import io.gravitee.management.service.exceptions.TechnicalManagementException;
import io.gravitee.management.service.impl.TransactionalService;
import io.gravitee.repository.exceptions.TechnicalException;
import io.gravitee.repository.management.api.ApiKeyRepository;
import io.gravitee.repository.management.model.ApiKey;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class ApiKeyServiceImpl
extends TransactionalService
implements ApiKeyService {
    private final Logger LOGGER = LoggerFactory.getLogger(ApiKeyServiceImpl.class);
    @Autowired
    private ApiKeyRepository apiKeyRepository;
    @Autowired
    private ApiKeyGenerator apiKeyGenerator;
    @Autowired
    private EmailService emailService;
    @Autowired
    private ApplicationService applicationService;

    @Override
    public ApiKeyEntity generateOrRenew(String applicationName, String apiName) {
        try {
            this.LOGGER.debug("Generate a new key for {} - {}", (Object)applicationName, (Object)apiName);
            ApiKey apiKey = new ApiKey();
            apiKey.setApplication(applicationName);
            apiKey.setApi(apiName);
            apiKey.setCreatedAt(new Date());
            apiKey.setKey(this.apiKeyGenerator.generate());
            apiKey.setRevoked(false);
            Instant expirationInst = apiKey.getCreatedAt().toInstant().plus(Duration.ofHours(1L));
            Date expirationDate = Date.from(expirationInst);
            Set oldKeys = this.apiKeyRepository.findByApplicationAndApi(applicationName, apiName);
            for (ApiKey oldKey : oldKeys) {
                this.setExpiration(expirationDate, oldKey);
            }
            apiKey = this.apiKeyRepository.create(applicationName, apiName, apiKey);
            return ApiKeyServiceImpl.convert(apiKey);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to generate a key for {} - {}", new Object[]{applicationName, apiName, ex});
            throw new TechnicalManagementException("An error occurs while trying to generate a key for " + applicationName + " - " + apiName, ex);
        }
    }

    @Override
    public void revoke(String apiKey) {
        try {
            this.LOGGER.debug("Revoke key {}", (Object)apiKey);
            Optional optKey = this.apiKeyRepository.retrieve(apiKey);
            if (!optKey.isPresent()) {
                throw new ApiKeyNotFoundException();
            }
            ApiKey key = (ApiKey)optKey.get();
            if (!key.isRevoked()) {
                key.setRevoked(true);
                key.setRevokeAt(new Date());
                this.apiKeyRepository.update(key);
                ApplicationEntity applicationEntity = this.applicationService.findById(key.getApplication());
                PrimaryOwnerEntity owner = applicationEntity.getPrimaryOwner();
                if (owner != null && owner.getEmail() != null && !owner.getEmail().isEmpty()) {
                    this.emailService.sendEmailNotification(new EmailNotificationBuilder().to(owner.getEmail()).subject("An API key has been revoked on API " + key.getApi() + " for application " + key.getApplication()).content("apiKeyRevoked.html").params((Map)ImmutableMap.of((Object)"owner", (Object)owner.getUsername(), (Object)"api", (Object)key.getApi(), (Object)"application", (Object)key.getApplication(), (Object)"apiKey", (Object)key.getKey())).build());
                }
            }
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to revoke a key {}", (Object)apiKey, (Object)ex);
            throw new TechnicalManagementException("An error occurs while trying to revoke a key " + apiKey, ex);
        }
    }

    @Override
    public Optional<ApiKeyEntity> getCurrent(String applicationName, String apiName) {
        try {
            this.LOGGER.debug("Generate a new key for {} - {}", (Object)applicationName, (Object)apiName);
            Set apiKeys = this.apiKeyRepository.findByApplicationAndApi(applicationName, apiName);
            if (apiKeys == null || apiKeys.isEmpty()) {
                return Optional.empty();
            }
            return apiKeys.stream().filter(apiKey -> !apiKey.isRevoked()).max((o1, o2) -> o1.getCreatedAt().compareTo(o2.getCreatedAt())).map(ApiKeyServiceImpl::convert);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while getting current API key for {} - {}", new Object[]{applicationName, apiName, ex});
            throw new TechnicalManagementException("An error occurs while getting current API key for " + applicationName + " - " + apiName, ex);
        }
    }

    @Override
    public Map<String, List<ApiKeyEntity>> findByApplication(String applicationId) {
        try {
            this.LOGGER.debug("Find all API keys for application {}", (Object)applicationId);
            Set keys = this.apiKeyRepository.findByApplication(applicationId);
            HashMap<String, Set> keysByApi = new HashMap<String, Set>();
            keys.forEach(apiKey -> {
                Set values = keysByApi.getOrDefault(apiKey.getApi(), new HashSet());
                values.add(apiKey);
                keysByApi.put(apiKey.getApi(), values);
            });
            HashMap<String, List<ApiKeyEntity>> keysByApiResult = new HashMap<String, List<ApiKeyEntity>>(keysByApi.size());
            keysByApi.forEach((api, apiKeys) -> keysByApiResult.put((String)api, apiKeys.stream().sorted((key1, key2) -> key2.getCreatedAt().compareTo(key1.getCreatedAt())).map(ApiKeyServiceImpl::convert).collect(Collectors.toList())));
            return keysByApiResult;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while getting all API keys for application {}", (Object)applicationId, (Object)ex);
            throw new TechnicalManagementException("An error occurs while getting all API keys for application " + applicationId, ex);
        }
    }

    @Override
    public Map<String, List<ApiKeyEntity>> findByApi(String apiId) {
        try {
            this.LOGGER.debug("Find all API keys for API {}", (Object)apiId);
            Set keys = this.apiKeyRepository.findByApi(apiId);
            HashMap<String, Set> keysByApplication = new HashMap<String, Set>();
            keys.forEach(apiKey -> {
                Set values = keysByApplication.getOrDefault(apiKey.getApplication(), new HashSet());
                values.add(apiKey);
                keysByApplication.put(apiKey.getApplication(), values);
            });
            HashMap<String, List<ApiKeyEntity>> keysByApplicationResult = new HashMap<String, List<ApiKeyEntity>>(keysByApplication.size());
            keysByApplication.forEach((api, apiKeys) -> keysByApplicationResult.put((String)api, apiKeys.stream().sorted((key1, key2) -> key2.getCreatedAt().compareTo(key1.getCreatedAt())).map(ApiKeyServiceImpl::convert).collect(Collectors.toList())));
            return keysByApplicationResult;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while getting all API keys for API {}", (Object)apiId, (Object)ex);
            throw new TechnicalManagementException("An error occurs while getting all API keys for API " + apiId, ex);
        }
    }

    @Override
    public ApiKeyEntity update(String apiKey, ApiKeyEntity apiKeyEntity) {
        try {
            this.LOGGER.debug("Trying to update key {}", (Object)apiKey);
            Optional optKey = this.apiKeyRepository.retrieve(apiKey);
            if (!optKey.isPresent()) {
                throw new ApiKeyNotFoundException();
            }
            ApiKey key = (ApiKey)optKey.get();
            this.setExpiration(apiKeyEntity.getExpireOn(), key);
            return ApiKeyServiceImpl.convert(key);
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to update a key {}", (Object)apiKey, (Object)ex);
            throw new TechnicalManagementException("An error occurs while trying to update a key " + apiKey, ex);
        }
    }

    private void setExpiration(Date expirationDate, ApiKey key) throws TechnicalException {
        if (!key.isRevoked() && key.getExpiration() == null) {
            key.setExpiration(expirationDate);
            this.apiKeyRepository.update(key);
            ApplicationEntity applicationEntity = this.applicationService.findById(key.getApplication());
            PrimaryOwnerEntity owner = applicationEntity.getPrimaryOwner();
            if (owner != null && owner.getEmail() != null && !owner.getEmail().isEmpty()) {
                this.emailService.sendEmailNotification(new EmailNotificationBuilder().to(owner.getEmail()).subject("An API key has been revoked on API " + key.getApi() + " for application " + key.getApplication()).content("apiKeyRevoked.html").params((Map)ImmutableMap.of((Object)"owner", (Object)owner.getUsername(), (Object)"api", (Object)key.getApi(), (Object)"application", (Object)key.getApplication(), (Object)"apiKey", (Object)key.getKey())).build());
            }
        }
    }

    @Override
    public Set<ApiKeyEntity> findAll(String applicationName, String apiName) {
        try {
            this.LOGGER.debug("Find all API Keys for {} - {}", (Object)applicationName, (Object)apiName);
            Set apiKeys = this.apiKeyRepository.findByApplicationAndApi(applicationName, apiName);
            if (apiKeys == null || apiKeys.isEmpty()) {
                return Collections.emptySet();
            }
            HashSet<ApiKeyEntity> apiKeyEntities = new HashSet<ApiKeyEntity>(apiKeys.size());
            apiKeyEntities.addAll(apiKeys.stream().map(ApiKeyServiceImpl::convert).collect(Collectors.toSet()));
            return apiKeyEntities;
        }
        catch (TechnicalException ex) {
            this.LOGGER.error("An error occurs while trying to find all API Keys for {} - {}", (Object)applicationName, (Object)apiName);
            throw new TechnicalManagementException("An error occurs while trying to find find all API Keys for " + applicationName + " - " + apiName, ex);
        }
    }

    private static ApiKeyEntity convert(ApiKey apiKey) {
        ApiKeyEntity apiKeyEntity = new ApiKeyEntity();
        apiKeyEntity.setKey(apiKey.getKey());
        apiKeyEntity.setCreatedAt(apiKey.getCreatedAt());
        apiKeyEntity.setExpireOn(apiKey.getExpiration());
        apiKeyEntity.setRevoked(apiKey.isRevoked());
        apiKeyEntity.setRevokeAt(apiKey.getRevokeAt());
        apiKeyEntity.setApi(apiKey.getApi());
        apiKeyEntity.setApplication(apiKey.getApplication());
        return apiKeyEntity;
    }
}

