package io.glimr.sdk.beacon;

import android.Manifest;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.pm.PackageManager;
import android.os.RemoteException;
import android.util.Log;

import io.glimr.sdk.engine.KATActionType;
import io.glimr.sdk.engine.KATRequestAdvertPayload;
import io.glimr.sdk.network.KATEndPoints;
import io.glimr.sdk.network.KATPostReqAsync;
import io.glimr.sdk.network.KATRequest;
import io.glimr.sdk.network.KATRequestDone;
import io.glimr.sdk.network.KATResponse;
import io.glimr.sdk.utils.KATSystemInformation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public class KATBeaconManager {

    private static KATBeaconManager instance;
    private final HashMap<String, IBeacon> mEnterBeacons = new HashMap<String, IBeacon>();
    private final HashMap<String, Integer> mExitBeacons = new HashMap<String, Integer>();
    public boolean isApplicationOpen = false;
    private ArrayList<String> mBeaconTokens;
    private String apiToken;
    private IBeaconManager iBeaconManager;
    private Context context;
    private MonitorNotifier mMonitorNotifier = new MonitorNotifier() {
        @Override
        public void didEnterRegion(Region region) {
            Log.d("GlimrSDK", "#Beacon enter region: " + region.getProximityUuid());
            sendRegionRequestToServer(region, KATActionType.KATActionTypeEnter);
        }

        @Override
        public void didExitRegion(Region region) {
            Log.d("GlimrSDK", "#Beacon exit region: " + region.getUniqueId());
            sendRegionRequestToServer(region, KATActionType.KATActionTypeExit);
        }

        @Override
        public void didDetermineStateForRegion(int state, Region region) {
            Log.d("GlimrSDK", "#Beacon state for region: " + region.getUniqueId());
        }
    };
    private RangeNotifier mRangeNotifier = new RangeNotifier() {
        @Override
        public void didRangeBeaconsInRegion(Collection<IBeacon> iBeacons, Region region) {
            beaconSendRequests(iBeacons, region);
            // rangeBeaconsInRegion(iBeacons, region);
        }
    };
    private IBeaconConsumer mBeaconConsumer = new IBeaconConsumer() {
        @Override
        public void onIBeaconServiceConnect() {
            iBeaconManager.setMonitorNotifier(mMonitorNotifier);
            iBeaconManager.setRangeNotifier(mRangeNotifier);

            for (String token : mBeaconTokens) {
                String dashlessUuid = token.toLowerCase().replaceAll("[\\-\\s]", "");
                if (dashlessUuid.length() == 32) {
                    try {
                        iBeaconManager.startMonitoringBeaconsInRegion(
                                new Region(token, token, null, null));
                    } catch (RemoteException e) {
                        Log.e("GlimrSDK", "#Beacon beaconError: " + e.getMessage());
                    }

                    try {
                        iBeaconManager
                                .startRangingBeaconsInRegion(new Region(token, token, null, null));
                    } catch (RemoteException e) {
                        Log.e("GlimrSDK", "#Beacon beaconError: " + e.getMessage());
                    }
                }
            }
        }

        @Override
        public Context getApplicationContext() {
            return context.getApplicationContext();
        }

        @Override
        public void unbindService(ServiceConnection connection) {
            if (connection != null && iBeaconManager != null && iBeaconManager.getIsServiceBound()) {
                try {
                    getApplicationContext().unbindService(connection);
                } catch (Exception e) {
                    Log.e("GlimrSDK", "#Beacon unbindService: " + e.getMessage());
                }
            }
        }

        @Override
        public boolean bindService(Intent intent, ServiceConnection connection, int mode) {
            return getApplicationContext().bindService(intent, connection, mode);
        }
    };

    protected KATBeaconManager(Context context) {
        this.context = context;
        iBeaconManager = IBeaconManager.getInstanceForApplication(context.getApplicationContext());
    }

    /**
     * Get the instance of the class
     *
     * @param context app context
     * @return instance
     */
    public static KATBeaconManager getInstance(Context context) {
        if (instance == null) {
            instance = new KATBeaconManager(context);
            KATSystemInformation.initSettings(context);
        }
        return instance;
    }

    public String getApiToken() {
        return apiToken;
    }

    public void setApiToken(String mApiToken) {
        this.apiToken = mApiToken;
    }

    public void stop() {
        try {
            iBeaconManager.stopScan();
        } catch (RemoteException e) {
            e.printStackTrace();
        }
    }

    /**
     * Bind the beacons so that the app will listen to beacons,
     * this must be done to get the beacons to work
     *
     * @param beaconIDs This is the ID the beacons should listen to, they will ignore everything
     *                  else
     * @param apiToken  The applications API token to be verified by the back-end system
     */
    public void bindBeacons(ArrayList<String> beaconIDs, String apiToken) {
        if (KATSystemInformation.canDoBle(context) &&
                context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE)) {
            mBeaconTokens = beaconIDs;
            this.apiToken = apiToken;
            Log.d("GlimrSDK", "#Beacon bind");
            if (!iBeaconManager.isBound(mBeaconConsumer)) {
                iBeaconManager.bind(mBeaconConsumer);
            }
        }
    }

    /**
     * This should be implemented in onPause in the given activity
     */
    public void unbindBeacons() {
        if (context != null &&
                KATSystemInformation.canDoBle(context) &&
                context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE)) {
            PackageManager pm = context.getPackageManager();
            if ((pm.checkPermission(Manifest.permission.BLUETOOTH, context.getPackageName())
                    == PackageManager.PERMISSION_GRANTED) &&
                    (pm.checkPermission(Manifest.permission.BLUETOOTH_ADMIN,
                            context.getPackageName()) == PackageManager.PERMISSION_GRANTED)) {
                Log.d("GlimrSDK", "#Beacon unbind");
                if (iBeaconManager.isBound(mBeaconConsumer)) {
                    iBeaconManager.unBind(mBeaconConsumer);
                }
            } else {
                Log.d("GlimrSDK",
                        "YOU NEED TO ADD BLUETOOTH AND BLUETOOTH ADMIN PERMISSION IN THE MANIFEST");
            }
        }
    }

    private void beaconSendRequests(Collection<IBeacon> iBeacons, Region region) {
        ArrayList<IBeacon> beacons = new ArrayList<IBeacon>(iBeacons);

        for (IBeacon beacon : beacons) {
            String key = keyForBeacon(beacon);

            if (!mEnterBeacons.containsKey(key)) {
                sendBeaconRequestToServer(beacon, region, KATActionType.KATActionTypeEnter);
                mEnterBeacons.put(key, beacon);
                Log.d("GlimrSDK", "#Make send " + key);
            }

            if (beacon.getRssi() != 0 && mExitBeacons.containsKey(key)) {
                mExitBeacons.remove(key);
            }
        }

        try {
            Iterator it = mEnterBeacons.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry pairs = (Map.Entry) it.next();
                IBeacon beaconId = (IBeacon) pairs.getValue();
                String key = keyForBeacon(beaconId);

                boolean isActive = false;
                for (IBeacon beacon : beacons) {
                    if (key.equalsIgnoreCase(keyForBeacon(beacon))) {
                        isActive = true;
                        break;
                    }
                }

                if (!isActive && !mExitBeacons.containsKey(key)) {
                    mExitBeacons.put(key, 2);
                }
            }

            it = mExitBeacons.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry pairs = (Map.Entry) it.next();
                String beaconId = (String) pairs.getKey();

                int current = (Integer) pairs.getValue();
                if (current >= 2) {
                    // sendBeaconRequestToServer(beacon, region, KATActionType.KATActionTypeExit);
                    mExitBeacons.remove(beaconId);
                    mEnterBeacons.remove(beaconId);
                } else if (current < 2) {
                    mExitBeacons.put(beaconId, current + 1);
                }
            }
        } catch (ConcurrentModificationException e) {

        }
    }

    private String keyForBeacon(IBeacon beacon) {
        return beacon.getProximityUuid() + "-" + beacon.getMajor() + "-" + beacon.getMinor();
    }

    private void sendBeaconRequestToServer(IBeacon iBeacon, Region region, final int action) {
        KATRequestAdvertPayload requestBeaconRegion = new KATRequestAdvertPayload(context);
        requestBeaconRegion.setAccuracy((float) iBeacon.getAccuracy());
        requestBeaconRegion.setActionType(action);
        requestBeaconRegion.setMajor(iBeacon.getMajor());
        requestBeaconRegion.setMinor(iBeacon.getMinor());
        requestBeaconRegion.setProximity(iBeacon.getProximity());
        requestBeaconRegion.setRegionId(region.getProximityUuid().toUpperCase());
        requestBeaconRegion.setRssi((float) iBeacon.getRssi());
        requestBeaconRegion.setState(isApplicationOpen ? 0 : 2);

        sendRequest(requestBeaconRegion);
    }

    private void sendRegionRequestToServer(Region region, final int action) {
        KATRequestAdvertPayload requestBeaconRegion = new KATRequestAdvertPayload(context);
        requestBeaconRegion.setActionType(action);
        requestBeaconRegion.setMajor(region.getMajor() != null ? region.getMajor() : 0);
        requestBeaconRegion.setMinor(region.getMinor() != null ? region.getMinor() : 0);
        requestBeaconRegion.setRegionId(region.getProximityUuid().toUpperCase());
        requestBeaconRegion.setState(isApplicationOpen ? 0 : 2);

        sendRequest(requestBeaconRegion);
    }

    private void sendRequest(KATRequestAdvertPayload payload) {
        KATRequest requestObject = new KATRequest(payload, KATEndPoints.baseURL(context) + KATEndPoints.COLLECT_EVENT_URL, null);

        KATPostReqAsync request = new KATPostReqAsync(apiToken, new KATRequestDone() {

            @Override
            public void KATRequestSuccess(KATResponse response) {
            }

            @Override
            public void KATRequestFailed(int responseCode, String errorMessage) {
            }
        });
        request.execute(requestObject);
    }
}
