/**
 * Radius Networks, Inc.
 * http://www.radiusnetworks.com
 *
 * @author David G. Young
 * <p/>
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * <p/>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p/>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package io.glimr.sdk.beacon;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothManager;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.util.Log;

import io.glimr.sdk.beacon.service.IBeaconService;
import io.glimr.sdk.beacon.service.RegionData;
import io.glimr.sdk.beacon.service.StartRMData;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * An class used to set up interaction with iBeacons from an <code>Activity</code> or
 * <code>Service</code>.
 * This class is used in conjunction with <code>IBeaconConsumer</code> interface, which provides a
 * callback
 * when the <code>IBeaconService</code> is ready to use.  Until this callback is made, ranging and
 * monitoring
 * of iBeacons is not possible.
 * <p/>
 * In the example below, an Activity implements the <code>IBeaconConsumer</code> interface, binds
 * to the service, then when it gets the callback saying the service is ready, it starts ranging.
 * <p/>
 * <pre><code>
 *  public class RangingActivity extends Activity implements IBeaconConsumer {
 *  	protected static final String TAG = "RangingActivity";
 *  	private IBeaconManager iBeaconManager = IBeaconManager.getInstanceForApplication(this);
 *         {@literal @}Override
 *  	protected void onCreate(Bundle savedInstanceState) {
 *  		super.onCreate(savedInstanceState);
 *  		setContentView(R.layout.activity_ranging);
 *  		iBeaconManager.bind(this);
 *        }
 *         {@literal @}Override
 *  	protected void onDestroy() {
 *  		super.onDestroy();
 *  		iBeaconManager.unBind(this);
 *        }
 *         {@literal @}Override
 *  	public void onIBeaconServiceConnect() {
 *  		iBeaconManager.setRangeNotifier(new RangeNotifier() {
 *                 {@literal @}Override
 *        	public void didRangeBeaconsInRegion(Collection<IBeacon> iBeacons, Region region) {
 *     			if (iBeacons.size() > 0) {
 * 	      			Log.i(TAG, "The first iBeacon I see is about "+iBeacons.iterator().next().getAccuracy()+"
 * meters away.");
 *                        }
 *                }
 *                });
 *
 *  		try {
 *  			iBeaconManager.startRangingBeaconsInRegion(new Region("myRangingUniqueId", null, null,
 * null));
 *                } catch (RemoteException e) {        }
 *        }
 *  }
 *  </code></pre>
 *
 * @author David G. Young
 */
public class IBeaconManager {

    private static final String TAG = "IBeaconManager";
    /**
     * set to true if you want to see debug messages associated with this library
     */
    public static boolean LOG_DEBUG = false;
    protected static IBeaconManager client = null;
    protected RangeNotifier rangeNotifier = null;
    protected RangeNotifier dataRequestNotifier = null;
    protected MonitorNotifier monitorNotifier = null;
    private Context context;
    private Map<IBeaconConsumer, ConsumerInfo> consumers
            = new HashMap<IBeaconConsumer, ConsumerInfo>();
    private Messenger serviceMessenger = null;
    private ArrayList<Region> monitoredRegions = new ArrayList<Region>();
    private ArrayList<Region> rangedRegions = new ArrayList<Region>();
    private boolean isServiceBound = false;
    private ServiceConnection iBeaconServiceConnection = new ServiceConnection() {
        // Called when the connection with the service is established
        public void onServiceConnected(ComponentName className, IBinder service) {
            serviceMessenger = new Messenger(service);
            isServiceBound = true;
            synchronized (consumers) {
                Iterator<IBeaconConsumer> consumerIterator = consumers.keySet().iterator();
                while (consumerIterator.hasNext()) {
                    IBeaconConsumer consumer = consumerIterator.next();
                    Boolean alreadyConnected = consumers.get(consumer).isConnected;
                    if (!alreadyConnected) {
                        consumer.onIBeaconServiceConnect();
                        ConsumerInfo consumerInfo = consumers.get(consumer);
                        consumerInfo.isConnected = true;
                        consumers.put(consumer, consumerInfo);
                    }
                }
            }
        }

        // Called when the connection with the service disconnects unexpectedly
        public void onServiceDisconnected(ComponentName className) {
            Log.e(TAG, "#Beacon onServiceDisconnected");
            isServiceBound = false;
        }
    };

    protected IBeaconManager(Context context) {
        this.context = context;
    }

    /**
     * An accessor for the singleton instance of this class.  A context must be provided, but if
     * you
     * need to use it from a non-Activity
     * or non-Service class, you can attach it to another singleton or a subclass of the Android
     * Application class.
     */
    protected static IBeaconManager getInstanceForApplication(Context context) {
        if (client == null) {
            if (IBeaconManager.LOG_DEBUG) {
                Log.d(TAG, "#Beacon IBeaconManager instance creation");
            }
            client = new IBeaconManager(context);
        }
        return client;
    }

    /**
     * Check if Bluetooth LE is supported by this Android device, and if so, make sure it is
     * enabled.
     * Throws a BleNotAvailableException if Bluetooth LE is not supported.  (Note: The Android
     * emulator will do this)
     *
     * @return false if it is supported and not enabled
     */
    public boolean checkAvailability() {
        if (!context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE)) {
            throw new BleNotAvailableException("Bluetooth LE not supported by this device");
        } else {
            if (((BluetoothManager) context.getSystemService(Context.BLUETOOTH_SERVICE))
                    .getAdapter().isEnabled()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Binds an Android <code>Activity</code> or <code>Service</code> to the
     * <code>IBeaconService</code>.  The
     * <code>Activity</code> or <code>Service</code> must implement the
     * <code>IBeaconConsuemr</code>
     * interface so
     * that it can get a callback when the service is ready to use.
     *
     * @param consumer the <code>Activity</code> or <code>Service</code> that will receive the
     *                 callback when the service is ready.
     */
    protected void bind(IBeaconConsumer consumer) {
        synchronized (consumers) {
            if (consumers.keySet().contains(consumer)) {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG, "This consumer is already bound");
                }
            } else {
                consumers.put(consumer, new ConsumerInfo());
                Intent intent = new Intent(consumer.getApplicationContext(), IBeaconService.class);
                consumer.bindService(intent, iBeaconServiceConnection, Context.BIND_AUTO_CREATE);
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG, "#Beacon consumer count is now:" + consumers.size());
                }
            }
        }
    }

    /**
     * Unbinds an Android <code>Activity</code> or <code>Service</code> to the
     * <code>IBeaconService</code>.  This should
     * typically be called in the onDestroy() method.
     *
     * @param consumer the <code>Activity</code> or <code>Service</code> that no longer needs to
     *                 use
     *                 the service.
     */
    protected void unBind(IBeaconConsumer consumer) {
        synchronized (consumers) {
            if (consumers.keySet().contains(consumer)) {
                consumer.unbindService(iBeaconServiceConnection);
                consumers.remove(consumer);
            } else {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG, "This consumer is not bound to: " + consumer);
                }
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG, "Bound consumers: ");
                }
                for (int i = 0; i < consumers.size(); i++) {
                    Log.i(TAG, " " + consumers.get(i));
                }
            }
        }
    }

    protected boolean getIsServiceBound() {
        return isServiceBound;
    }

    /**
     * Tells you if the passed iBeacon consumer is bound to the service
     */
    protected boolean isBound(IBeaconConsumer consumer) {
        synchronized (consumers) {
            return consumers.keySet().contains(consumer) && (serviceMessenger != null);
        }
    }

    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
    protected void startScan() throws RemoteException {
        if (serviceMessenger == null) {
            throw new RemoteException(
                    "The IBeaconManager is not bound to the service.  Call iBeaconManager.bind(IBeaconConsumer consumer) and wait for a callback to onIBeaconServiceConnect()");
        }
        Message msg = Message.obtain(null, IBeaconService.MSG_START_SCANNING, 0, 0);
        serviceMessenger.send(msg);
    }

    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
    protected void stopScan() throws RemoteException {
        if (serviceMessenger == null) {
            return;
        }
        Message msg = Message.obtain(null, IBeaconService.MSG_STOP_SCANNING, 0, 0);
        serviceMessenger.send(msg);
    }

    /**
     * Specifies a class that should be called each time the <code>IBeaconService</code> gets
     * ranging
     * data, which is nominally once per second when iBeacons are detected.
     * <p/>
     * IMPORTANT:  Only one RangeNotifier may be active for a given application.  If two different
     * activities or services set different RangeNotifier instances, the last one set will receive
     * all the notifications.
     *
     * @see RangeNotifier
     */
    protected void setRangeNotifier(RangeNotifier notifier) {
        rangeNotifier = notifier;
    }

    /**
     * Specifies a class that should be called each time the <code>IBeaconService</code> gets sees
     * or stops seeing a Region of iBeacons.
     * <p/>
     * IMPORTANT:  Only one MonitorNotifier may be active for a given application.  If two
     * different
     * activities or services set different RangeNotifier instances, the last one set will receive
     * all the notifications.
     *
     * @see MonitorNotifier
     * @see #startMonitoringBeaconsInRegion(Region region)
     * @see Region
     */
    protected void setMonitorNotifier(MonitorNotifier notifier) {
        monitorNotifier = notifier;
    }

    /**
     * Tells the <code>IBeaconService</code> to start looking for iBeacons that match the passed
     * <code>Region</code> object, and providing updates on the estimated distance very seconds
     * while
     * iBeacons in the Region are visible.  Note that the Region's unique identifier must be
     * retained to
     * later call the stopRangingBeaconsInRegion method.
     *
     * @see IBeaconManager#setRangeNotifier(RangeNotifier)
     * @see IBeaconManager#stopRangingBeaconsInRegion(Region region)
     * @see RangeNotifier
     * @see Region
     */
    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
    protected void startRangingBeaconsInRegion(Region region) throws RemoteException {
        if (serviceMessenger == null) {
            throw new RemoteException(
                    "The IBeaconManager is not bound to the service.  Call iBeaconManager.bind(IBeaconConsumer consumer) and wait for a callback to onIBeaconServiceConnect()");
        }
        Message msg = Message.obtain(null, IBeaconService.MSG_START_RANGING, 0, 0);
        StartRMData obj = new StartRMData(new RegionData(region), callbackPackageName(), 0, 0);
        msg.obj = obj;
        serviceMessenger.send(msg);
        synchronized (rangedRegions) {
            rangedRegions.add((Region) region.clone());
        }
    }

    /**
     * Tells the <code>IBeaconService</code> to stop looking for iBeacons that match the passed
     * <code>Region</code> object and providing distance information for them.
     *
     * @see #setMonitorNotifier(MonitorNotifier notifier)
     * @see #startMonitoringBeaconsInRegion(Region region)
     * @see MonitorNotifier
     * @see Region
     */
    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
    protected void stopRangingBeaconsInRegion(Region region) throws RemoteException {
        if (serviceMessenger == null) {
            throw new RemoteException(
                    "The IBeaconManager is not bound to the service.  Call iBeaconManager.bind(IBeaconConsumer consumer) and wait for a callback to onIBeaconServiceConnect()");
        }
        Message msg = Message.obtain(null, IBeaconService.MSG_STOP_RANGING, 0, 0);
        StartRMData obj = new StartRMData(new RegionData(region), callbackPackageName(), 0, 0);
        msg.obj = obj;
        serviceMessenger.send(msg);
        synchronized (rangedRegions) {
            Region regionToRemove = null;
            for (Region rangedRegion : rangedRegions) {
                if (region.getUniqueId().equals(rangedRegion.getProximityUuid())) {
                    regionToRemove = rangedRegion;
                }
            }
            rangedRegions.remove(regionToRemove);
        }
    }

    /**
     * Tells the <code>IBeaconService</code> to start looking for iBeacons that match the passed
     * <code>Region</code> object.  Note that the Region's unique identifier must be retained to
     * later call the stopMonitoringBeaconsInRegion method.
     *
     * @see IBeaconManager#setMonitorNotifier(MonitorNotifier)
     * @see IBeaconManager#stopMonitoringBeaconsInRegion(Region region)
     * @see MonitorNotifier
     * @see Region
     */
    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
    protected void startMonitoringBeaconsInRegion(Region region) throws RemoteException {
        if (serviceMessenger == null) {
            throw new RemoteException(
                    "The IBeaconManager is not bound to the service.  Call iBeaconManager.bind(IBeaconConsumer consumer) and wait for a callback to onIBeaconServiceConnect()");
        }
        Message msg = Message.obtain(null, IBeaconService.MSG_START_MONITORING, 0, 0);
        StartRMData obj = new StartRMData(new RegionData(region), callbackPackageName(), 0, 0);
        msg.obj = obj;
        serviceMessenger.send(msg);
        synchronized (monitoredRegions) {
            monitoredRegions.add((Region) region.clone());
        }
    }

    /**
     * Tells the <code>IBeaconService</code> to stop looking for iBeacons that match the passed
     * <code>Region</code> object.  Note that the Region's unique identifier is used to match it to
     * and existing monitored Region.
     *
     * @see IBeaconManager#setMonitorNotifier(MonitorNotifier)
     * @see IBeaconManager#startMonitoringBeaconsInRegion(Region region)
     * @see MonitorNotifier
     * @see Region
     */
    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
    protected void stopMonitoringBeaconsInRegion(Region region) throws RemoteException {
        if (serviceMessenger == null) {
            throw new RemoteException(
                    "The IBeaconManager is not bound to the service.  Call iBeaconManager.bind(IBeaconConsumer consumer) and wait for a callback to onIBeaconServiceConnect()");
        }
        Message msg = Message.obtain(null, IBeaconService.MSG_STOP_MONITORING, 0, 0);
        StartRMData obj = new StartRMData(new RegionData(region), callbackPackageName(), 0, 0);
        msg.obj = obj;
        serviceMessenger.send(msg);
        synchronized (monitoredRegions) {
            Region regionToRemove = null;
            for (Region monitoredRegion : monitoredRegions) {
                if (region.getUniqueId().equals(monitoredRegion.getProximityUuid())) {
                    regionToRemove = monitoredRegion;
                }
            }
            monitoredRegions.remove(regionToRemove);
        }
    }

    private String callbackPackageName() {
        return context.getPackageName();
    }

    /**
     * @return monitorNotifier
     * @see #monitorNotifier
     */
    protected MonitorNotifier getMonitoringNotifier() {
        return this.monitorNotifier;
    }

    /**
     * @return rangeNotifier
     * @see #rangeNotifier
     */
    protected RangeNotifier getRangingNotifier() {
        return this.rangeNotifier;
    }

    /**
     * @return the list of regions currently being monitored
     */
    protected Collection<Region> getMonitoredRegions() {
        ArrayList<Region> clonedMontoredRegions = new ArrayList<Region>();
        synchronized (this.monitoredRegions) {
            for (Region montioredRegion : this.monitoredRegions) {
                clonedMontoredRegions.add((Region) montioredRegion.clone());
            }
        }
        return clonedMontoredRegions;
    }

    /**
     * @return the list of regions currently being ranged
     */
    protected Collection<Region> getRangedRegions() {
        ArrayList<Region> clonedRangedRegions = new ArrayList<Region>();
        synchronized (this.rangedRegions) {
            for (Region rangedRegion : this.rangedRegions) {
                clonedRangedRegions.add((Region) rangedRegion.clone());
            }
        }
        return clonedRangedRegions;
    }

    protected RangeNotifier getDataRequestNotifier() {
        return this.dataRequestNotifier;
    }

    protected void setDataRequestNotifier(RangeNotifier notifier) {
        this.dataRequestNotifier = notifier;
    }

    private class ConsumerInfo {

        protected boolean isConnected = false;

        protected boolean isInBackground = false;
    }

}
