package io.glimr.sdk.geofence;

import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.util.Log;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.LocationServices;
import com.google.gson.internal.LinkedTreeMap;
import io.glimr.sdk.utils.KATSystemInformation;

import java.util.ArrayList;
import java.util.List;

public class KATGeofenceManager implements GoogleApiClient.OnConnectionFailedListener, GoogleApiClient.ConnectionCallbacks {

    public static final String TAG = "com.glimr.sdk";
    // Internal List of Geofence objects. In a real app, these might be provided by an API based on
    // locations within the user's proximity.
    List<Geofence> mGeofenceList;
    // Persistent storage for geofences.
    private KATGeofenceStore mGeofenceStorage;
    // Stores the PendingIntent used to request geofence monitoring.
    private PendingIntent mGeofenceRequestIntent;
    private GoogleApiClient mApiClient;
    private Context context;

    public KATGeofenceManager(Context context) {
        this.context = context;

        // Rather than displayng this activity, simply display a toast indicating that the geofence
        // service is being created. This should happen in less than a second.
        if (!isGooglePlayServicesAvailable()) {
            Log.i("GlimrSDK", "Google Play services unavailable.");
            return;
        }

        mApiClient = new GoogleApiClient.Builder(this.getApplicationContext())
                .addApi(LocationServices.API)
                .addConnectionCallbacks(this)
                .addOnConnectionFailedListener(this)
                .build();

        // Instantiate a new geofence storage area.
        mGeofenceStorage = new KATGeofenceStore(this.getApplicationContext());
    }


    public void updateGeofences(final ArrayList<LinkedTreeMap> circulars) {
        mGeofenceList = new ArrayList<Geofence>();

        if (circulars.size() > 0) {
            try {
                for (LinkedTreeMap map : circulars) {
                    KATGeofence mUIGeofence1 = new KATGeofence(
                            (String) map.get("identifier"),
                            (Double) map.get("latitude"),
                            (Double) map.get("longitude"),
                            new Float((Double) map.get("radius")),
                            Geofence.GEOFENCE_TRANSITION_ENTER | Geofence.GEOFENCE_TRANSITION_EXIT);
                    mGeofenceStorage.setGeofence((String) map.get("identifier"), mUIGeofence1);
                    mGeofenceList.add(mUIGeofence1.toGeofence());
                }

                mApiClient.connect();
            } catch (Exception e) {
                Log.e("GlimrSDK", "Exception while parsing regions.");
            }
        }
    }


    private Context getApplicationContext() {
        return context.getApplicationContext();
    }


    @Override
    public void onConnectionFailed(ConnectionResult connectionResult) {
        // If the error has a resolution, start a Google Play services activity to resolve it.
        if (connectionResult.hasResolution()) {
            Log.e("GlimrSDK", "Exception while resolving connection error.");
        } else {
            int errorCode = connectionResult.getErrorCode();
            Log.e("GlimrSDK", "Connection to Google Play services failed with error code " + errorCode);
        }
    }


    /**
     * Once the connection is available, send a request to add the Geofences.
     */
    @Override
    public void onConnected(Bundle connectionHint) {
        // Get the PendingIntent for the geofence monitoring request.
        // Send a request to add the current geofences.
        try {
            if (KATSystemInformation.hasLocationPermission(context)) {
                mGeofenceRequestIntent = getGeofenceTransitionPendingIntent();
                LocationServices.GeofencingApi.removeGeofences(mApiClient, mGeofenceRequestIntent);
                LocationServices.GeofencingApi.addGeofences(mApiClient, mGeofenceList, mGeofenceRequestIntent);
            }
        } catch (Exception e) {
            Log.e("GlimrSDK", "Exception while setting up geofences.");
        }
    }


    @Override
    public void onConnectionSuspended(int i) {
    }


    /**
     * Checks if Google Play services is available.
     *
     * @return true if it is.
     */
    private boolean isGooglePlayServicesAvailable() {
        int resultCode = GooglePlayServicesUtil.isGooglePlayServicesAvailable(this.context);
        if (ConnectionResult.SUCCESS == resultCode) {
            return true;
        } else {
            Log.e("GlimrSDK", "Google Play services is unavailable.");
            return false;
        }
    }


    /**
     * Create a PendingIntent that triggers GeofenceTransitionIntentService when a geofence
     * transition occurs.
     */
    private PendingIntent getGeofenceTransitionPendingIntent() {
        Intent intent = new Intent(this.getApplicationContext(), KATGeofenceIntentService.class);
        return PendingIntent.getService(this.getApplicationContext(), 0, intent,
                PendingIntent.FLAG_UPDATE_CURRENT);
    }

}
