package io.glimr.sdk.audience;

import android.content.Context;
import android.content.pm.PackageManager;
import android.location.Location;
import android.util.Log;

import io.glimr.sdk.engine.KATEvent;
import io.glimr.sdk.engine.KATManager;
import io.glimr.sdk.network.KATEndPoints;
import io.glimr.sdk.network.KATPostReqAsync;
import io.glimr.sdk.network.KATRequest;
import io.glimr.sdk.network.KATRequestDone;
import io.glimr.sdk.network.KATRequestPayload;
import io.glimr.sdk.network.KATResponse;
import io.glimr.sdk.utils.KATSharedPreferences;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static java.lang.Math.min;

public class KATAudienceManager {

    private String apiToken;
    private int tagCacheExpireSeconds;
    private Context context;

    public KATAudienceManager(Context context, String apiToken, int tagCacheExpireSeconds) {
        this.context = context;
        this.apiToken = apiToken;
        this.tagCacheExpireSeconds = min(tagCacheExpireSeconds, 300);
    }

    private static String hashOfConcludes(Map<String, String> map) {
        String s = "";
        if (map == null || map.isEmpty()) {
            return s;
        }
        for (Map.Entry<String, String> entry : map.entrySet()) {
            s += entry.getKey() + String.valueOf(entry.getValue());
        }

        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest.getInstance("MD5");
            digest.update(s.getBytes());
            byte messageDigest[] = digest.digest();

            // Create Hex String
            StringBuilder hexString = new StringBuilder();
            for (byte aMessageDigest : messageDigest) {
                hexString.append(Integer.toHexString(0xFF & aMessageDigest));
            }
            return hexString.toString();

        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }

        return s;
    }

    public void getAudiences(Location location, final KATEvent callback) {
        if (callback == null) {
            return;
        }

        final Date now = new Date();
        long diffInMs = now.getTime() - KATSharedPreferences.getLastAudienceUpdateDate(context).getTime();
        if (KATManager.disableCaching) {
            Log.d("GlimrSDK", "#Caching disabled, please enable for production - Glimr");
        }
        if (!KATManager.disableCaching && diffInMs > 0 && TimeUnit.MILLISECONDS.toSeconds(diffInMs) < this.tagCacheExpireSeconds) {
            @SuppressWarnings("unchecked")
            HashMap<String, ArrayList<String>> storedTags = KATSharedPreferences.getAudienceResponseMap(context);
            if (storedTags != null) {
                callback.availableAudiencesUpdated(storedTags);
                return;
            }
        }

        KATRequestPayload payload = new KATRequestPayload(this.context);
        if (location != null) {
            payload.lat = location.getLatitude();
            payload.lon = location.getLongitude();
            payload.speed = location.getSpeed();
            payload.course = location.getBearing();
        }

        KATRequest requestObject = new KATRequest(
                payload,
                KATEndPoints.baseURL(context) + KATEndPoints.AUDIENCE_URL,
                KATUserTags.class);

        KATPostReqAsync request = new KATPostReqAsync(apiToken, new KATRequestDone() {
            @Override
            public void KATRequestSuccess(KATResponse response) {
                if (response.responseObject != null
                        && ((KATUserTags) response.responseObject).mapping != null
                        && ((KATUserTags) response.responseObject).mapping.size() > 0) {
                    callback.availableAudiencesUpdated(
                            ((KATUserTags) response.responseObject).mapping);

                    KATSharedPreferences.setLastAudienceUpdateDate(context, now);
                    KATSharedPreferences.saveAudienceResponseMap(context, ((KATUserTags) response.responseObject).mapping);
                }
            }

            @Override
            public void KATRequestFailed(int responseCode, String errorMessage) {
                Log.d("GlimrSDK", errorMessage);
            }
        });
        request.execute(requestObject);
    }

    public void collect(HashMap concludes, Location location) {
        try {
            KATRequestConcludePayload basePayload = new KATRequestConcludePayload(this.context);
            String storedConcludesHash = KATSharedPreferences.getConcludesHash(this.context);
            String calculatedConcludesHash = hashOfConcludes(concludes);

            storedConcludesHash = storedConcludesHash == null ? "" : storedConcludesHash;
            calculatedConcludesHash = calculatedConcludesHash == null ? "" : calculatedConcludesHash;

            boolean hasChanged = calculatedConcludesHash.compareTo(storedConcludesHash) != 0;

            final Date now = new Date();
            long diffInMs = now.getTime() - KATSharedPreferences.getLastCollectionUpdateDate(context).getTime();
            if (KATManager.disableCaching) {
                Log.d("GlimrSDK", "#Caching disabled, please enable for production - Glimr");
            }

            if (hasChanged || (!KATManager.disableCaching && diffInMs > 0 && TimeUnit.MILLISECONDS.toSeconds(diffInMs) < 604800)) {
                basePayload.setConcludes(concludes);
                KATSharedPreferences.saveConcludesHash(this.context, calculatedConcludesHash);
                KATSharedPreferences.setLastCollectionUpdateDate(context, now);
            }

            basePayload.setUrls(KATSharedPreferences.getTrackedURLs(this.context));
            basePayload.setCollect(true);

            if (location != null) {
                basePayload.lat = location.getLatitude();
                basePayload.lon = location.getLongitude();
                basePayload.speed = location.getSpeed();
                basePayload.course = location.getBearing();
            }

            KATRequest requestObject = new KATRequest(
                    basePayload, KATEndPoints.baseURL(context) + KATEndPoints.AUDIENCE_URL,
                    null);

            KATPostReqAsync request = new KATPostReqAsync(apiToken, null);
            request.execute(requestObject);
        } catch (Exception e) {
            Log.e("Error", e.getMessage());
        }
    }

    public void trackURL(String url) {
        try {
            KATSharedPreferences.storeTrackedURL(context, url, new Date());
        } catch (Exception e) {
            Log.e("Error", e.getMessage());
        }
    }

    public boolean isPackageInstalled(String packagename, Context context) {
        PackageManager pm = context.getPackageManager();
        try {
            pm.getPackageInfo(packagename, PackageManager.GET_ACTIVITIES);
            return true;
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
    }
}
