package io.glimr.sdk.engine;

import android.content.Context;
import android.location.Location;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.text.TextUtils;
import android.util.Log;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.LocationServices;
import com.google.gson.internal.LinkedTreeMap;
import io.glimr.sdk.audience.KATAudienceManager;
import io.glimr.sdk.beacon.KATBeaconManager;
import io.glimr.sdk.beacon.service.KATBeaconReceiver;
import io.glimr.sdk.geofence.KATGeofenceManager;
import io.glimr.sdk.network.KATEndPoints;
import io.glimr.sdk.network.KATPostReqAsync;
import io.glimr.sdk.network.KATRequest;
import io.glimr.sdk.network.KATRequestDone;
import io.glimr.sdk.network.KATRequestPayload;
import io.glimr.sdk.network.KATResponse;
import io.glimr.sdk.utils.KATSharedPreferences;
import io.glimr.sdk.utils.KATSystemInformation;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;

public class KATManager implements GoogleApiClient.ConnectionCallbacks,
        GoogleApiClient.OnConnectionFailedListener {

    public static boolean disableCaching;
    private static KATGeofenceManager mKatGeofenceManager;
    private static KATAudienceManager mKatAudienceManager;
    private static GoogleApiClient mGoogleApiClient;
    private static Location mLastLocation;
    private static Location mMockLocation;
    private static KATManager instance;
    private boolean geotagCallRequested;
    private boolean monitoringCallRequested;
    private KATEvent callbackContext;
    private Context context;
    private KATBeaconManager mKatBeaconManager;
    private String apiToken;

    protected KATManager(final Context context, final String apiToken, final KATEvent callbackContext, final int scanInterval, final boolean forceIpOnly) {
        this.context = context;
        this.apiToken = apiToken;
        this.callbackContext = callbackContext;
        this.setForceIpOnly(forceIpOnly);

        Thread thread = new Thread(new Runnable(){
            @Override
            public void run(){
                KATSharedPreferences.saveEntry(context, apiToken);
                KATSharedPreferences.saveScanInterval(context, scanInterval);
                KATSharedPreferences.getSharedPreferencesUrls(context);
            }
        });
        thread.start();

        try {
            mKatBeaconManager = KATBeaconManager.getInstance(context);
            mKatBeaconManager.setApiToken(apiToken);
            mKatAudienceManager = new KATAudienceManager(context, callbackContext, apiToken);
            mKatGeofenceManager = new KATGeofenceManager(context);
        } catch (Exception e) {
            Log.e("GlimrSDK Error", e.getMessage());
        }
    }

    /**
     * Get the instance of the class
     */
    public static KATManager getInstance(Context context, String apiToken,
                                         KATEvent callbackContext) {
        if (instance == null || instance.callbackContext == null) {
            instance = new KATManager(context, apiToken, callbackContext, 90, false);
            KATSystemInformation.initSettings(context);
        }
        return instance;
    }

    public static KATManager getInstance(Context context, String apiToken,
                                         KATEvent callbackContext, int scanInterval) {
        if (instance == null || instance.callbackContext == null) {
            instance = new KATManager(context, apiToken, callbackContext, scanInterval, false);
            KATSystemInformation.initSettings(context);
        }
        return instance;
    }

    public static KATManager getInstance(Context context, String apiToken,
                                         KATEvent callbackContext, int scanInterval, boolean forceIpOnly) {
        if (instance == null || instance.callbackContext == null) {
            instance = new KATManager(context, apiToken, callbackContext, scanInterval, forceIpOnly);
            KATSystemInformation.initSettings(context);
        }
        return instance;
    }

    public static ArrayList<String> mapToArrayList(HashMap<String, ArrayList<String>> kv) {
        ArrayList<String> result = new ArrayList<String>();
        for (HashMap.Entry<String, ArrayList<String>> entry : kv.entrySet()) {
            result.addAll(entry.getValue());
        }
        return result;
    }

    public static String map2QueryString(HashMap<String, ArrayList<String>> kv) {
        ArrayList<String> parts = new ArrayList<String>();
        for (HashMap.Entry<String, ArrayList<String>> entry : kv.entrySet()) {
            String key = entry.getKey();
            ArrayList<String> values = entry.getValue();

            String part = "";
            for (String value : values) {
                part = key + "=" + Uri.encode(value);
            }

            if (!part.equals("")) {
                parts.add(part);
            }
        }

        return TextUtils.join("&", parts);
    }

    public void setApiToken(String apiToken) {
        this.apiToken = apiToken;
    }

    protected synchronized void buildGoogleApiClient() {
        mKatBeaconManager.isApplicationOpen = true;

        if (!KATSystemInformation.hasLocationPermission(context)) {
            return;
        }

        if (mGoogleApiClient == null) {
            mGoogleApiClient = new GoogleApiClient.Builder(this.context)
                    .addConnectionCallbacks(this)
                    .addOnConnectionFailedListener(this)
                    .addApi(LocationServices.API)
                    .build();
        }

        if (mGoogleApiClient.isConnected()) {
            if (mGoogleApiClient.hasConnectedApi(LocationServices.API)) {
                mLastLocation = updatedLocation();
            } else {
                mGoogleApiClient.reconnect();
            }
        } else {
            mGoogleApiClient.connect();
        }
    }

    private void initMonitoring(ArrayList<LinkedTreeMap> regionArray) {
        if (android.os.Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2) {
            return;
        }

        ArrayList<String> beacons = new ArrayList<String>();
        ArrayList<LinkedTreeMap> circular = new ArrayList<LinkedTreeMap>();

        for (LinkedTreeMap region : regionArray) {
            if ((Double) region.get("type") == 0.0) {
                beacons.add((String) region.get("identifier"));
            } else {
                circular.add(region);
            }
        }

        if (KATSystemInformation.canDoBle(context)) {
            if (beacons.size() > 0) {
                KATBeaconReceiver beaconAlarm = new KATBeaconReceiver();
                beaconAlarm.schedule(context);
            }
        } else {
            Log.d("GlimrSDK", "Can't do BLE tracking, needs > 4.4 + permissions");
        }

        if (KATSystemInformation.hasLocationPermission(context)) {
            mKatGeofenceManager.updateGeofences(circular);
        }
    }

    private void updateConfig() {
        final Date now = new Date();
        long diffInMs = now.getTime() - KATSharedPreferences.getLastConfigUpdateDate(context)
                .getTime();
        if (KATManager.disableCaching) {
            Log.d("GlimrSDK", "#Caching disabled, please enable for production - Glimr");
        }
        if (!KATManager.disableCaching && diffInMs > 0 && TimeUnit.MILLISECONDS.toSeconds(diffInMs) < 900) {
            ArrayList config = KATSharedPreferences.getConfigResponse(context);
            if (config != null) {
                initMonitoring(config);
                return;
            }
        }

        try {
            KATRequestPayload requestPayload = new KATRequestPayload(context);
            if (mLastLocation != null) {
                requestPayload.lat = mLastLocation.getLatitude();
                requestPayload.lon = mLastLocation.getLongitude();
                requestPayload.course = mLastLocation.getBearing();
                requestPayload.speed = mLastLocation.getSpeed();
            }

            KATRequest requestObject = new KATRequest(
                    requestPayload,
                    KATEndPoints.baseURL(context) + KATEndPoints.CONFIG_URL,
                    null);

            KATPostReqAsync request = new KATPostReqAsync(apiToken, new KATRequestDone() {
                @Override
                public void KATRequestSuccess(KATResponse response) {
                    try {
                        LinkedTreeMap<String, Object> responseMap
                                = (LinkedTreeMap<String, Object>) response.responseObject;
                        if (response.responseObject != null && responseMap.containsKey("regions")) {
                            initMonitoring((ArrayList<LinkedTreeMap>) responseMap.get("regions"));
                            KATSharedPreferences.setLastConfigUpdateDate(context, now);
                            KATSharedPreferences.saveConfigResponse(context,
                                    (ArrayList<LinkedTreeMap>) responseMap.get("regions"));
                        }

                        HashMap<String, Boolean> concludes = new HashMap<String, Boolean>();
                        if (response.responseObject != null && responseMap.containsKey("concludes")) {
                            for (String url : (ArrayList<String>) responseMap.get("concludes")) {
                                if (mKatAudienceManager.isPackageInstalled(url, context)) {
                                    concludes.put(url, Boolean.TRUE);
                                } else {
                                    concludes.put(url, Boolean.FALSE);
                                }
                            }
                        }
                        mKatAudienceManager.collect(concludes, mLastLocation);

                        if (response.responseObject != null && responseMap.containsKey("serviceURL")) {
                            String serviceURL = (String) responseMap.get("serviceURL");
                            if (serviceURL != null && serviceURL.startsWith("https://")) {
                                KATSharedPreferences.saveServiceURL(context, serviceURL);
                            }
                        }
                    } catch (Exception e) {
                        Log.e("GlimrSDK Error", e.getMessage());
                    }
                }

                @Override
                public void KATRequestFailed(int responseCode, String errorMessage) {
                    try {
                        KATSharedPreferences.saveServiceURL(context, KATEndPoints.fallbackURL());
                        ArrayList config = KATSharedPreferences.getConfigResponse(context);
                        if (config != null) {
                            initMonitoring(config);
                        }
                    } catch (Exception e) {
                        Log.e("GlimrSDK Error", e.getMessage());
                    }
                }
            });
            request.execute(requestObject);
        } catch (Exception e) {
            Log.e("GlimrSDK Error", e.getMessage());
        }
    }

    public void stopMonitoring() {
        if (mGoogleApiClient != null && mGoogleApiClient.isConnected()) {
            mGoogleApiClient.disconnect();
        }

        monitoringCallRequested = false;
        geotagCallRequested = false;
        mLastLocation = null;
        mKatBeaconManager.isApplicationOpen = false;
    }

    public void startMonitoring() {
        if (!KATSystemInformation.isGenerallySupported()) {
            Log.e("GlimrSDK Info", "This call requires >=4.2");
            return;
        }

        buildGoogleApiClient();

        if (KATSystemInformation.advertId != null
                && (mLastLocation != null || !KATSystemInformation.hasLocationPermission(context))) {
            executeStartMonitoring();
        } else {
            monitoringCallRequested = true;
        }
    }

    public void getAudiencesAndGeotags() {
        if (!KATSystemInformation.isGenerallySupported()) {
            Log.e("GlimrSDK Info", "This call requires >=4.2");
            return;
        }

        buildGoogleApiClient();

        if (KATSystemInformation.advertId != null
                && (mLastLocation != null || !KATSystemInformation.hasLocationPermission(context))) {
            executeAudiencesAndGeotags();
        } else {
            geotagCallRequested = true;
        }
    }

    public void trackURL(String url) {
        try {
            mKatAudienceManager.trackURL(url);
        } catch (Exception e) {
            Log.e("GlimrSDK Error", e.getMessage());
        }
    }

    private void executeStartMonitoring() {
        monitoringCallRequested = false;

        if (KATSystemInformation.advertId != null) {
            updateConfig();
            Log.d("GlimrSDK", "startMonitoring");
        } else {
            try {
                final Handler handler = new Handler();
                handler.postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        updateConfig();
                        Log.d("GlimrSDK", "startMonitoring delayed");
                    }
                }, KATSystemInformation.advertIdDelay());
            } catch (Exception e) {
                Log.e("GlimrSDK Error", e.getMessage());
            }
        }
    }

    private void executeAudiencesAndGeotags() {
        geotagCallRequested = false;

        if (KATSystemInformation.advertId != null) {
            mKatAudienceManager.getAudiences(mLastLocation);
            Log.d("GlimrSDK", "getAudiencesAndGeotags");
        } else {
            try {
                final Handler handler = new Handler();
                handler.postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        mKatAudienceManager.getAudiences(mLastLocation);
                        Log.d("GlimrSDK", "getAudiencesAndGeotags delayed");
                    }
                }, KATSystemInformation.advertIdDelay());
            } catch (Exception e) {
                Log.e("GlimrSDK Error", e.getMessage());
            }
        }
    }

    @Override
    public void onConnected(Bundle bundle) {
        Log.d("GlimrSDK", "ApiClient onConnected");

        mLastLocation = updatedLocation();

        if (geotagCallRequested) {
            executeAudiencesAndGeotags();
        }
        if (monitoringCallRequested) {
            executeStartMonitoring();
        }
    }

    @Override
    public void onConnectionSuspended(int i) {
        Log.e("GlimrSDK Error", "ApiClient onConnectionSuspended");

        if (geotagCallRequested) {
            executeAudiencesAndGeotags();
        }
        if (monitoringCallRequested) {
            executeStartMonitoring();
        }
    }

    @Override
    public void onConnectionFailed(ConnectionResult connectionResult) {
        Log.e("GlimrSDK Error", "ApiClient onConnectionFailed");

        if (geotagCallRequested) {
            executeAudiencesAndGeotags();
        }
        if (monitoringCallRequested) {
            executeStartMonitoring();
        }
    }

    public void setMockLocation(Location location) {
        mMockLocation = location;
    }

    public void setForceIpOnly(boolean forceIpOnly) {
        KATSystemInformation.setHasForcedIpOnly(forceIpOnly);
    }

    private Location updatedLocation() {
        if (KATSystemInformation.hasForcedIpOnly()) {
            return null;
        }
        if (mMockLocation != null) {
            return mMockLocation;
        }
        if (mGoogleApiClient != null) {
            return LocationServices.FusedLocationApi.getLastLocation(mGoogleApiClient);
        }
        return null;
    }
}
