/**
 * Radius Networks, Inc.
 * http://www.radiusnetworks.com
 *
 * @author David G. Young
 * <p/>
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * <p/>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p/>
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package io.glimr.sdk.beacon.service;

import android.app.Service;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothManager;
import android.content.Context;
import android.content.Intent;
import android.os.AsyncTask;
import android.os.Binder;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.util.Log;

import io.glimr.sdk.beacon.IBeacon;
import io.glimr.sdk.beacon.IBeaconManager;
import io.glimr.sdk.beacon.Region;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author dyoung
 */

public class IBeaconService extends Service {

    public static final String TAG = "IBeaconService";
    /**
     * Command to the service to display a message
     */
    public static final int MSG_START_RANGING = 2;
    public static final int MSG_STOP_RANGING = 3;
    public static final int MSG_START_MONITORING = 4;
    public static final int MSG_STOP_MONITORING = 5;
    public static final int MSG_SET_SCAN_PERIODS = 6;
    public static final int MSG_START_SCANNING = 7;
    public static final int MSG_STOP_SCANNING = 8;
    /**
     * Target we publish for clients to send messages to IncomingHandler.
     */
    final Messenger mMessenger = new Messenger(new IncomingHandler(this));
    private Map<Region, RangeState> rangedRegionState = new HashMap<Region, RangeState>();
    /*
     * The scan period is how long we wait between restarting the BLE advertisement scans
     * Each time we restart we only see the unique advertisements once (e.g. unique iBeacons)
     * So if we want updates, we have to restart.  iOS gets updates once per second, so ideally we
     * would restart scanning that often to get the same update rate.  The trouble is that when you 
     * restart scanning, it is not instantaneous, and you lose any iBeacon packets that were in the 
     * air during the restart.  So the more frequently you restart, the more packets you lose.  The
     * frequency is therefore a tradeoff.  Testing with 14 iBeacons, transmitting once per second,
     * here are the counts I got for various values of the SCAN_PERIOD:
     * 
     * Scan period     Avg iBeacons      % missed
     *    1s               6                 57
     *    2s               10                29
     *    3s               12                14
     *    5s               14                0
     *    
     * Also, because iBeacons transmit once per second, the scan period should not be an even multiple
     * of seconds, because then it may always miss a beacon that is synchronized with when it is stopping
     * scanning.
     * 
     */
    private Map<Region, MonitorState> monitoredRegionState = new HashMap<Region, MonitorState>();
    private BluetoothAdapter bluetoothAdapter;
    private boolean scanning;
    private boolean scanningPaused;
    private Date lastIBeaconDetectionTime = new Date();
    private HashSet<IBeacon> trackedBeacons;
    private Handler handler = new Handler();
    private int bindCount = 0;
    private List<IBeacon> simulatedScanData = null;
    private int ongoing_notification_id = 1;
    private long lastScanEndTime = 0l;
    // Device scan callback.
    private BluetoothAdapter.LeScanCallback leScanCallback =
            new BluetoothAdapter.LeScanCallback() {

                @Override
                public void onLeScan(final BluetoothDevice device, final int rssi,
                                     final byte[] scanRecord) {
                    new ScanProcessor().execute(new ScanData(device, rssi, scanRecord));

                }
            };

    /**
     * When binding to the service, we return an interface to our messenger
     * for sending messages to the service.
     */
    @Override
    public IBinder onBind(Intent intent) {
        bindCount++;
        return mMessenger.getBinder();
    }

    @Override
    public boolean onUnbind(Intent intent) {
        bindCount--;
        return false;
    }

    @Override
    public void onCreate() {
        getBluetoothAdapter();
    }

    @Override
    public void onDestroy() {
        if (IBeaconManager.LOG_DEBUG) {
            Log.e(TAG, "#Beacon onDestory called.  stopping scanning");
        }
        if (bluetoothAdapter != null && scanning && leScanCallback != null && bluetoothAdapter.getState() == BluetoothAdapter.STATE_ON) {
            bluetoothAdapter.stopLeScan(leScanCallback);
            lastScanEndTime = new Date().getTime();
        }
    }

    /* 
     * Returns true if the service is running, but all bound clients have indicated they are in the background
     */
    private boolean isInBackground() {
        if (IBeaconManager.LOG_DEBUG) {
            Log.d(TAG, "bound client count:" + bindCount);
        }
        return bindCount == 0;
    }

    /**
     * methods for clients
     */

    public void startRangingBeaconsInRegion(Region region, Callback callback) {
        synchronized (rangedRegionState) {
            if (rangedRegionState.containsKey(region)) {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.i(TAG, "Already ranging that region -- will replace existing region.");
                }
                rangedRegionState
                        .remove(region); // need to remove it, otherwise the old object will be retained because they are .equal
            }
            rangedRegionState.put(region, new RangeState(callback));
        }
        if (!scanning) {
            scanLeDevice(true);
        }
    }

    public void stopRangingBeaconsInRegion(Region region) {
        synchronized (rangedRegionState) {
            rangedRegionState.remove(region);
        }
        if (scanning && rangedRegionState.size() == 0 && monitoredRegionState.size() == 0) {
            scanLeDevice(false);
        }
    }

    public void startMonitoringBeaconsInRegion(Region region, Callback callback) {
        if (IBeaconManager.LOG_DEBUG) {
            Log.d(TAG, "startMonitoring called");
        }
        synchronized (monitoredRegionState) {
            if (monitoredRegionState.containsKey(region)) {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.i(TAG,
                            "Already monitoring that region -- will replace existing region monitor.");
                }
                monitoredRegionState
                        .remove(region); // need to remove it, otherwise the old object will be retained because they are .equal
            }
            monitoredRegionState.put(region, new MonitorState(callback));
        }
        if (IBeaconManager.LOG_DEBUG) {
            Log.d(TAG, "Currently monitoring " + monitoredRegionState.size() + " regions.");
        }
        if (!scanning) {
            scanLeDevice(true);
        }
    }

    public void stopMonitoringBeaconsInRegion(Region region) {
        if (IBeaconManager.LOG_DEBUG) {
            Log.d(TAG, "stopMonitoring called");
        }
        synchronized (monitoredRegionState) {
            monitoredRegionState.remove(region);
        }
        if (IBeaconManager.LOG_DEBUG) {
            Log.d(TAG, "Currently monitoring " + monitoredRegionState.size() + " regions.");
        }
        if (scanning && rangedRegionState.size() == 0 && monitoredRegionState.size() == 0) {
            scanLeDevice(false);
        }
    }

    private void scanLeDevice(final Boolean enable) {
        BluetoothAdapter adapter = getBluetoothAdapter();
        if (adapter == null || leScanCallback == null || adapter.getState() != BluetoothAdapter.STATE_ON) {
            if (IBeaconManager.LOG_DEBUG) {
                Log.e(TAG, "No bluetooth adapter.  iBeaconService cannot scan.");
            }
            if (simulatedScanData == null) {
                Log.w(TAG, "exiting");
                return;
            } else {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.w(TAG, "proceeding with simulated scan data");
                }
            }
        }
        if (enable) {
            trackedBeacons = new HashSet<IBeacon>();
            if (scanning == false || scanningPaused == true) {
                scanning = true;
                scanningPaused = false;
                try {
                    if (adapter != null && leScanCallback != null) {
                        if (adapter.isEnabled() && adapter.getState() == BluetoothAdapter.STATE_ON) {
                            adapter.startLeScan(leScanCallback);
                        } else {
                            if (IBeaconManager.LOG_DEBUG) {
                                Log.w(TAG, "Bluetooth is disabled.  Cannot scan for iBeacons.");
                            }
                        }
                    }
                } catch (Exception e) {
                    Log.e("TAG",
                            "Exception starting bluetooth scan.  Perhaps bluetooth is disabled or unavailable?");
                }
            } else {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG, "We are already scanning");
                }
            }

            if (IBeaconManager.LOG_DEBUG) {
                Log.d(TAG, "Scan started");
            }
        } else {
            if (IBeaconManager.LOG_DEBUG) {
                Log.d(TAG, "Scan stopped");
            }

            finishScanCycle();
        }
    }

    private void scheduleScanStop() {
        finishScanCycle();
    }

    private void finishScanCycle() {
        processExpiredMonitors();
        if (scanning == true) {
            if (!anyRangingOrMonitoringRegionsActive()) {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG,
                            "Not starting scan because no monitoring or ranging regions are defined.");
                }
            } else {
                processRangeData();
                BluetoothAdapter adapter = getBluetoothAdapter();
                if (adapter != null && leScanCallback != null && adapter.getState() == BluetoothAdapter.STATE_ON) {
                    if (adapter.isEnabled()) {
                        scanning = false;
                        adapter.stopLeScan(leScanCallback);
                        lastScanEndTime = new Date().getTime();
                    } else {
                        if (IBeaconManager.LOG_DEBUG) {
                            Log.w(TAG, "Bluetooth is disabled.  Cannot scan for iBeacons.");
                        }
                    }
                }

                scanningPaused = true;
            }
        }
    }

    private void processRangeData() {
        Iterator<Region> regionIterator = rangedRegionState.keySet().iterator();
        while (regionIterator.hasNext()) {
            Region region = regionIterator.next();
            RangeState rangeState = rangedRegionState.get(region);
            if (IBeaconManager.LOG_DEBUG) {
                Log.d(TAG, "Calling ranging callback with " + rangeState.getIBeacons().size()
                        + " iBeacons");
            }
            rangeState.getCallback().call(IBeaconService.this, "rangingData",
                    new RangingData(rangeState.getIBeacons(), region));
            rangeState.clearIBeacons();
        }

    }

    private void processExpiredMonitors() {
        Iterator<Region> monitoredRegionIterator = monitoredRegionState.keySet().iterator();
        while (monitoredRegionIterator.hasNext()) {
            Region region = monitoredRegionIterator.next();
            MonitorState state = monitoredRegionState.get(region);
            if (state.isNewlyOutside()) {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG, "found a monitor that expired: " + region);
                }
                state.getCallback().call(IBeaconService.this, "monitoringData",
                        new MonitoringData(state.isInside(), region));
            }
        }
    }

    private void processIBeaconFromScan(IBeacon iBeacon) {
        lastIBeaconDetectionTime = new Date();
        trackedBeacons.add(iBeacon);
        if (IBeaconManager.LOG_DEBUG) {
            Log.d(TAG,
                    "iBeacon detected :" + iBeacon.getProximityUuid() + " "
                            + iBeacon.getMajor() + " " + iBeacon.getMinor()
                            + " accuracy: " + iBeacon.getAccuracy()
                            + " proximity: " + iBeacon.getProximity());
        }

        List<Region> matchedRegions = null;
        synchronized (monitoredRegionState) {
            matchedRegions = matchingRegions(iBeacon,
                    monitoredRegionState.keySet());
        }
        Iterator<Region> matchedRegionIterator = matchedRegions.iterator();
        while (matchedRegionIterator.hasNext()) {
            Region region = matchedRegionIterator.next();
            MonitorState state = monitoredRegionState.get(region);
            if (state.markInside()) {
                state.getCallback().call(IBeaconService.this, "monitoringData",
                        new MonitoringData(state.isInside(), region));
            }
        }

        if (IBeaconManager.LOG_DEBUG) {
            Log.d(TAG, "looking for ranging region matches for this ibeacon");
        }
        synchronized (rangedRegionState) {
            matchedRegions = matchingRegions(iBeacon, rangedRegionState.keySet());
        }
        matchedRegionIterator = matchedRegions.iterator();
        while (matchedRegionIterator.hasNext()) {
            Region region = matchedRegionIterator.next();
            if (IBeaconManager.LOG_DEBUG) {
                Log.d(TAG, "matches ranging region: " + region);
            }
            RangeState rangeState = rangedRegionState.get(region);
            rangeState.addIBeacon(iBeacon);
        }
    }

    private List<Region> matchingRegions(IBeacon iBeacon, Collection<Region> regions) {
        List<Region> matched = new ArrayList<Region>();
        Iterator<Region> regionIterator = regions.iterator();
        while (regionIterator.hasNext()) {
            Region region = regionIterator.next();
            if (region.matchesIBeacon(iBeacon)) {
                matched.add(region);
            } else {
                if (IBeaconManager.LOG_DEBUG) {
                    Log.d(TAG, "This region does not match: " + region);
                }
            }

        }
        return matched;
    }

    /*
     Returns false if no ranging or monitoring regions have beeen requested.  This is useful in determining if we should scan at all.
     */
    private boolean anyRangingOrMonitoringRegionsActive() {
        return (rangedRegionState.size() + monitoredRegionState.size()) > 0;
    }

    private BluetoothAdapter getBluetoothAdapter() {
        if (bluetoothAdapter == null) {
            // Initializes Bluetooth adapter.
            final BluetoothManager bluetoothManager =
                    (BluetoothManager) this.getApplicationContext()
                            .getSystemService(Context.BLUETOOTH_SERVICE);
            bluetoothAdapter = bluetoothManager.getAdapter();
        }
        return bluetoothAdapter;
    }

    static class IncomingHandler extends Handler {

        private final WeakReference<IBeaconService> mService;

        IncomingHandler(IBeaconService service) {
            mService = new WeakReference<IBeaconService>(service);
        }

        @Override
        public void handleMessage(Message msg) {
            IBeaconService service = mService.get();
            StartRMData startRMData = (StartRMData) msg.obj;

            if (service != null) {
                switch (msg.what) {
                    case MSG_START_RANGING:
                        service.startRangingBeaconsInRegion(startRMData.getRegionData(),
                                new io.glimr.sdk.beacon.service.Callback(
                                        startRMData.getCallbackPackageName()));
                        break;
                    case MSG_STOP_RANGING:
                        service.stopRangingBeaconsInRegion(startRMData.getRegionData());
                        break;
                    case MSG_START_MONITORING:
                        service.startMonitoringBeaconsInRegion(startRMData.getRegionData(),
                                new io.glimr.sdk.beacon.service.Callback(
                                        startRMData.getCallbackPackageName()));
                        break;
                    case MSG_STOP_MONITORING:
                        service.stopMonitoringBeaconsInRegion(startRMData.getRegionData());
                        break;
                    case MSG_SET_SCAN_PERIODS:
                        break;
                    case MSG_START_SCANNING:
                        service.scanLeDevice(true);
                        break;
                    case MSG_STOP_SCANNING:
                        service.scanLeDevice(false);
                        break;
                    default:
                        super.handleMessage(msg);
                }
            }
        }
    }

    /**
     * Class used for the client Binder.  Because we know this service always
     * runs in the same process as its clients, we don't need to deal with IPC.
     */
    public class IBeaconBinder extends Binder {

        public IBeaconService getService() {
            if (IBeaconManager.LOG_DEBUG) {
                Log.i(TAG, "getService of IBeaconBinder called");
            }
            // Return this instance of LocalService so clients can call public methods
            return IBeaconService.this;
        }
    }

    private class ScanData {

        @SuppressWarnings("unused")
        public BluetoothDevice device;
        public int rssi;
        public byte[] scanRecord;

        public ScanData(BluetoothDevice device, int rssi, byte[] scanRecord) {
            this.device = device;
            this.rssi = rssi;
            this.scanRecord = scanRecord;
        }
    }

    private class ScanProcessor extends AsyncTask<ScanData, Void, Void> {

        @Override
        protected Void doInBackground(ScanData... params) {
            ScanData scanData = params[0];

            IBeacon iBeacon = IBeacon.fromScanData(scanData.scanRecord,
                    scanData.rssi);
            if (iBeacon != null) {
                processIBeaconFromScan(iBeacon);
            }
            return null;
        }

        @Override
        protected void onPostExecute(Void result) {
        }

        @Override
        protected void onPreExecute() {
        }

        @Override
        protected void onProgressUpdate(Void... values) {
        }
    }

}
