package io.glimr.sdk.audience;

import android.content.Context;
import android.content.pm.PackageManager;
import android.location.Location;
import android.util.Log;

import io.glimr.sdk.engine.KATEvent;
import io.glimr.sdk.engine.KATManager;
import io.glimr.sdk.network.KATEndPoints;
import io.glimr.sdk.network.KATPostReqAsync;
import io.glimr.sdk.network.KATRequest;
import io.glimr.sdk.network.KATRequestDone;
import io.glimr.sdk.network.KATRequestPayload;
import io.glimr.sdk.network.KATResponse;
import io.glimr.sdk.utils.KATSharedPreferences;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class KATAudienceManager {

    private String apiToken;
    private Context context;
    private KATEvent callbackContext;

    public KATAudienceManager(Context context, KATEvent callbackContext, String apiToken) {
        this.context = context;
        this.callbackContext = callbackContext;
        this.apiToken = apiToken;
    }

    private static String hashOfConcludes(Map<String, String> map) {
        String s = "";
        if (map == null || map.isEmpty()) {
            return s;
        }
        for (Map.Entry<String, String> entry : map.entrySet()) {
            s += entry.getKey() + String.valueOf(entry.getValue());
        }

        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest.getInstance("MD5");
            digest.update(s.getBytes());
            byte messageDigest[] = digest.digest();

            // Create Hex String
            StringBuilder hexString = new StringBuilder();
            for (byte aMessageDigest : messageDigest) {
                hexString.append(Integer.toHexString(0xFF & aMessageDigest));
            }
            return hexString.toString();

        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }

        return s;
    }

    public void getAudiences(Location location) {

        final Date now = new Date();
        long diffInMs = now.getTime() - KATSharedPreferences.getLastAudienceUpdateDate(context).getTime();
        if (KATManager.disableCaching) {
            Log.d("GlimrSDK", "#Caching disabled, please enable for production - Glimr");
        }
        if (!KATManager.disableCaching && diffInMs > 0 && TimeUnit.MILLISECONDS.toSeconds(diffInMs) < 90) {
            @SuppressWarnings("unchecked")
            HashMap<String, ArrayList<String>> storedTags = KATSharedPreferences.getAudienceResponseMap(context);
            if (storedTags != null) {
                callbackContext.availableAudiencesUpdated(storedTags);
                return;
            }
        }

        KATRequestPayload payload = new KATRequestPayload(this.context);
        if (location != null) {
            payload.lat = location.getLatitude();
            payload.lon = location.getLongitude();
            payload.speed = location.getSpeed();
            payload.course = location.getBearing();
        }

        KATRequest requestObject = new KATRequest(
                payload,
                KATEndPoints.baseURL(context) + KATEndPoints.AUDIENCE_URL,
                KATUserTags.class);

        KATPostReqAsync request = new KATPostReqAsync(apiToken, new KATRequestDone() {
            @Override
            public void KATRequestSuccess(KATResponse response) {
                if (response.responseObject != null
                        && ((KATUserTags) response.responseObject).mapping != null
                        && ((KATUserTags) response.responseObject).mapping.size() > 0) {
                    callbackContext.availableAudiencesUpdated(
                            ((KATUserTags) response.responseObject).mapping);

                    KATSharedPreferences.setLastAudienceUpdateDate(context, now);
                    KATSharedPreferences.saveAudienceResponseMap(context, ((KATUserTags) response.responseObject).mapping);
                }
            }

            @Override
            public void KATRequestFailed(int responseCode, String errorMessage) {
                Log.d("GlimrSDK", errorMessage);
            }
        });
        request.execute(requestObject);
    }

    public void collect(HashMap concludes, Location location) {
        try {
            KATRequestConcludePayload basePayload = new KATRequestConcludePayload(this.context);
            String storedConcludesHash = KATSharedPreferences.getConcludesHash(this.context);
            String calculatedConcludesHash = hashOfConcludes(concludes);

            storedConcludesHash = storedConcludesHash == null ? "" : storedConcludesHash;
            calculatedConcludesHash = calculatedConcludesHash == null ? "" : calculatedConcludesHash;

            boolean hasChanged = calculatedConcludesHash.compareTo(storedConcludesHash) != 0;
            if (hasChanged) {
                basePayload.setConcludes(concludes);
                KATSharedPreferences.saveConcludesHash(this.context, calculatedConcludesHash);
            }

            basePayload.setUrls(KATSharedPreferences.getTrackedURLs(this.context));
            basePayload.setCollect(true);

            if (location != null) {
                basePayload.lat = location.getLatitude();
                basePayload.lon = location.getLongitude();
                basePayload.speed = location.getSpeed();
                basePayload.course = location.getBearing();
            }

            KATRequest requestObject = new KATRequest(
                    basePayload, KATEndPoints.baseURL(context) + KATEndPoints.AUDIENCE_URL,
                    null);

            KATPostReqAsync request = new KATPostReqAsync(apiToken, null);
            request.execute(requestObject);
        } catch (Exception e) {
            Log.e("Error", e.getMessage());
        }
    }

    public void trackURL(String url) {
        try {
            KATSharedPreferences.storeTrackedURL(context, url, new Date());
        } catch (Exception e) {
            Log.e("Error", e.getMessage());
        }
    }

    public boolean isPackageInstalled(String packagename, Context context) {
        PackageManager pm = context.getPackageManager();
        try {
            pm.getPackageInfo(packagename, PackageManager.GET_ACTIVITIES);
            return true;
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
    }
}
