package io.glimr.sdk.geofence;

import android.app.IntentService;
import android.content.Intent;
import android.text.TextUtils;
import android.util.Log;

import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingEvent;
import io.glimr.sdk.engine.KATActionType;
import io.glimr.sdk.engine.KATRequestAdvertPayload;
import io.glimr.sdk.network.KATEndPoints;
import io.glimr.sdk.network.KATPostReqAsync;
import io.glimr.sdk.network.KATRequest;
import io.glimr.sdk.network.KATRequestDone;
import io.glimr.sdk.network.KATResponse;
import io.glimr.sdk.utils.KATSharedPreferences;
import io.glimr.sdk.utils.KATSystemInformation;

import java.util.List;

/**
 * This class receives geofence transition events from Location Services, in the
 * form of an Intent containing the transition type and geofence id(s) that triggered
 * the event.
 */
public class KATGeofenceIntentService extends IntentService {

    /**
     * Sets an identifier for this class' background thread
     */
    public KATGeofenceIntentService() {
        super("KATGeofenceIntentService");
    }

    /**
     * Handles incoming intents
     *
     * @param intent The Intent sent by Location Services. This Intent is provided
     *               to Location Services (inside a PendingIntent) when you call addGeofences()
     */
    @Override
    protected void onHandleIntent(Intent intent) {
        GeofencingEvent geoFenceEvent = GeofencingEvent.fromIntent(intent);

        if (geoFenceEvent.hasError()) {
            int errorCode = geoFenceEvent.getErrorCode();
            // Get the error message
            String errorMessage = "CODE " + errorCode;

            // Log the error
            Log.d("GlimrSDK", errorMessage);

        } else {
            int transitionType = geoFenceEvent.getGeofenceTransition();
            if (Geofence.GEOFENCE_TRANSITION_ENTER == transitionType
                    || Geofence.GEOFENCE_TRANSITION_EXIT == transitionType) {

                // String triggeredGeoFenceId = geoFenceEvent.getTriggeringGeofences().get(0).getRequestId();
                int actionType = transitionType == Geofence.GEOFENCE_TRANSITION_ENTER ? KATActionType.KATActionTypeEnter : KATActionType.KATActionTypeExit;

                List<Geofence> geofences = geoFenceEvent.getTriggeringGeofences();
                String[] geofenceIds = new String[geofences.size()];
                for (int index = 0; index < geofences.size(); index++) {
                    geofenceIds[index] = geofences.get(index).getRequestId();
                }

                String ids = TextUtils.join(",", geofenceIds);
                Log.d("GlimrSDK #Trigger", ids);

                sendRequest(geofenceIds[0], actionType);
            } else {
                // Always log as an error
                Log.d("GlimrSDK", "#Tansition " + transitionType);
            }
        }
    }

    private void sendRequest(final String identifier, final int action) {
        KATSystemInformation.initSettings(this);

        try {
            Thread.sleep(KATSystemInformation.advertIdDelay());
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

        KATRequestAdvertPayload payload = new KATRequestAdvertPayload(this);
        payload.setActionType(action);
        payload.setRegionId(identifier);
        payload.setState(2);

        KATRequest requestObject = new KATRequest(payload, KATEndPoints.baseURL(getApplicationContext()) + KATEndPoints.COLLECT_EVENT_URL, null);
        KATPostReqAsync request = new KATPostReqAsync(KATSharedPreferences.getEntry(this), new KATRequestDone() {
            @Override
            public void KATRequestSuccess(KATResponse response) {
            }

            @Override
            public void KATRequestFailed(int responseCode, String errorMessage) {
            }
        });
        request.execute(requestObject);
    }
}
