/*
 * Copyright (C) 2023 Berner Fachhochschule https://e-voting.bfh.ch
 *
 *  - This program is free software: you can redistribute it and/or modify                           -
 *  - it under the terms of the GNU Affero General Public License as published by                    -
 *  - the Free Software Foundation, either version 3 of the License, or                              -
 *  - (at your option) any later version.                                                            -
 *  -                                                                                                -
 *  - This program is distributed in the hope that it will be useful,                                -
 *  - but WITHOUT ANY WARRANTY; without even the implied warranty of                                 -
 *  - MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                                   -
 *  - GNU General Public License for more details.                                                   -
 *  -                                                                                                -
 *  - You should have received a copy of the GNU Affero General Public License                       -
 *  - along with this program. If not, see <http://www.gnu.org/licenses/>.                           -
 */
package ch.openchvote.printingauthority.writein.states;

import ch.openchvote.algorithms.AlgorithmException;
import ch.openchvote.framework.Message;
import ch.openchvote.framework.State;
import ch.openchvote.framework.context.EventMessages;
import ch.openchvote.framework.context.EventSetup;
import ch.openchvote.protocol.parameters.SystemParameters;
import ch.openchvote.printingauthority.PrintingAuthority;
import ch.openchvote.printingauthority.writein.EventData;
import ch.openchvote.printingauthority.writein.tasks.T1;
import ch.openchvote.protocol.message.MessageType;
import ch.openchvote.protocol.message.writein.MAP1;
import ch.openchvote.protocol.message.writein.MEP1;
import ch.openchvote.protocol.message.writein.MPV1;

public final class S1 extends State<PrintingAuthority, EventData> {

    public S1() {
        super(Type.START);
        this.registerMessageHandler(MessageType.MAP1, S1::handleMAP1);
        this.registerMessageHandler(MessageType.MEP1, S1::handleMEP1);
        this.registerMessageHandler(S1::handleINT);
    }

    private static void handleMAP1(PrintingAuthority printingAuthority, Message message, EventSetup eventSetup, EventData eventData) {

        // check and get message content
        var messageContent = printingAuthority.checkAndGetContent(MAP1.class, message, eventSetup);
        var EP = messageContent.get_EP();

        // update event data
        eventData.EP.set(EP);

        // send internal message
        printingAuthority.sendInternalMessage(eventSetup);
    }

    private static void handleMEP1(PrintingAuthority printingAuthority, Message message, EventSetup eventSetup, EventData eventData) {

        // select event data
        var EP = eventData.EP.get();

        // check and get message content
        var messageContent = printingAuthority.checkAndGetContent(MEP1.class, message, EP, eventSetup);
        var bold_d_j = messageContent.get_bold_d();

        // get sender's participant index
        var j = eventSetup.getParticipantIndex(message.getSenderId());

        // update event data
        eventData.bold_D.set(j, bold_d_j);

        // send internal message
        printingAuthority.sendInternalMessage(eventSetup);
    }

    private static void handleINT(PrintingAuthority printingAuthority, EventSetup eventSetup, EventData eventData, EventMessages eventMessages) {
        var params = new SystemParameters(eventSetup.getSecurityLevel(), eventSetup.getConfiguration());

        // check if all messages are available
        if (eventMessages.hasAllMessages(eventSetup, MessageType.MAP1) && eventMessages.hasAllMessages(eventSetup, MessageType.MEP1)) {
            try {
                // perform task
                T1.run(eventData, params);

                // select event data
                var N_E = eventData.EP.get().get_bold_d().getLength();
                var bold_ec = eventData.bold_ec.get();

                // send individual MPV1 messages to all voters
                for (int i = 1; i <= N_E; i++) {
                    var EC_i = bold_ec.getValue(i);
                    printingAuthority.sendMessage(i, new MPV1(EC_i), eventSetup);
                }

                // update state
                eventData.setCurrentState(S2.class);

            } catch (AlgorithmException exception) {
                // move to error state
                eventData.setCurrentState(E1.class);
            }
        }
    }

}
