/*
 * Decompiled with CFR 0.152.
 */
package jadx.gui.cache.manager;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import jadx.api.plugins.utils.CommonFileUtils;
import jadx.core.utils.GsonUtils;
import jadx.core.utils.Utils;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.core.utils.files.FileUtils;
import jadx.gui.cache.manager.CacheEntry;
import jadx.gui.settings.JadxProject;
import jadx.gui.settings.JadxSettings;
import jadx.gui.settings.data.ProjectData;
import jadx.gui.utils.files.JadxFiles;
import java.io.BufferedReader;
import java.io.Reader;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CacheManager {
    private static final Logger LOG = LoggerFactory.getLogger(CacheManager.class);
    private static final Gson GSON = GsonUtils.buildGson();
    private static final Type CACHES_TYPE = new TypeToken<List<CacheEntry>>(){}.getType();
    private final Map<String, CacheEntry> cacheMap;
    private final JadxSettings settings;

    public CacheManager(JadxSettings settings) {
        this.settings = settings;
        this.cacheMap = this.loadCaches();
    }

    public Path getCacheDir(JadxProject project, @Nullable String cacheDirStr) {
        if (cacheDirStr == null) {
            Path newProjectCacheDir = this.buildCacheDir(project);
            this.addEntry(this.projectToKey(project), newProjectCacheDir);
            return newProjectCacheDir;
        }
        Path cacheDir = this.resolveCacheDirStr(cacheDirStr, project.getProjectPath());
        return this.verifyEntry(project, cacheDir);
    }

    public void projectPathUpdate(JadxProject project, Path newPath) {
        if (Objects.equals(project.getProjectPath(), newPath)) {
            return;
        }
        String key = this.projectToKey(project);
        CacheEntry prevEntry = this.cacheMap.remove(key);
        if (prevEntry == null) {
            return;
        }
        CacheEntry newEntry = new CacheEntry();
        newEntry.setProject(CacheManager.pathToString(newPath));
        newEntry.setCache(prevEntry.getCache());
        this.addEntry(newEntry);
    }

    public List<CacheEntry> getCachesList() {
        ArrayList<CacheEntry> list = new ArrayList<CacheEntry>(this.cacheMap.values());
        Collections.sort(list);
        return list;
    }

    public synchronized void removeCacheEntry(CacheEntry entry) {
        try {
            this.cacheMap.remove(entry.getProject());
            this.saveCaches(this.cacheMap);
            FileUtils.deleteDirIfExists((Path)Paths.get(entry.getCache(), new String[0]));
        }
        catch (Exception e) {
            LOG.error("Failed to remove cache entry: " + entry.getCache(), (Throwable)e);
        }
    }

    private Path resolveCacheDirStr(String cacheDirStr, Path projectPath) {
        Path path = Paths.get(cacheDirStr, new String[0]);
        if (path.isAbsolute() || projectPath == null) {
            return path;
        }
        return projectPath.resolveSibling(path);
    }

    public String buildCacheDirStr(Path dir) {
        if (Objects.equals(this.settings.getCacheDir(), ".")) {
            return dir.getFileName().toString();
        }
        return CacheManager.pathToString(dir);
    }

    private Path buildCacheDir(JadxProject project) {
        String cacheDirValue = this.settings.getCacheDir();
        if (Objects.equals(cacheDirValue, ".")) {
            return CacheManager.buildLocalCacheDir(project);
        }
        Path cacheBaseDir = cacheDirValue == null ? JadxFiles.PROJECTS_CACHE_DIR : Paths.get(cacheDirValue, new String[0]);
        return cacheBaseDir.resolve(CacheManager.buildProjectUniqName(project));
    }

    private static Path buildLocalCacheDir(JadxProject project) {
        Path projectPath = project.getProjectPath();
        if (projectPath != null) {
            return projectPath.resolveSibling(projectPath.getFileName() + ".cache");
        }
        List<Path> files = project.getFilePaths();
        if (files.isEmpty()) {
            throw new JadxRuntimeException("Failed to build local cache dir");
        }
        Path path = files.stream().filter(p -> !p.getFileName().toString().endsWith(".jadx.kts")).findFirst().orElseGet(() -> (Path)files.get(0));
        String name = CommonFileUtils.removeFileExtension((String)path.getFileName().toString());
        return path.resolveSibling(name + ".jadx.cache");
    }

    private Path verifyEntry(JadxProject project, Path cacheDir) {
        boolean cacheExists = Files.exists(cacheDir, new LinkOption[0]);
        String key = this.projectToKey(project);
        CacheEntry entry = this.cacheMap.get(key);
        if (entry == null) {
            Path newCacheDir = cacheExists ? cacheDir : this.buildCacheDir(project);
            this.addEntry(key, newCacheDir);
            return newCacheDir;
        }
        if (entry.getCache().equals(CacheManager.pathToString(cacheDir)) && cacheExists) {
            return cacheDir;
        }
        FileUtils.deleteDirIfExists((Path)Paths.get(entry.getCache(), new String[0]));
        Path newCacheDir = cacheExists ? cacheDir : this.buildCacheDir(project);
        entry.setCache(CacheManager.pathToString(newCacheDir));
        entry.setTimestamp(System.currentTimeMillis());
        this.saveCaches(this.cacheMap);
        return newCacheDir;
    }

    private void addEntry(String projectKey, Path cacheDir) {
        CacheEntry entry = new CacheEntry();
        entry.setProject(projectKey);
        entry.setCache(CacheManager.pathToString(cacheDir));
        this.addEntry(entry);
    }

    private void addEntry(CacheEntry entry) {
        entry.setTimestamp(System.currentTimeMillis());
        this.cacheMap.put(entry.getProject(), entry);
        this.saveCaches(this.cacheMap);
    }

    private String projectToKey(JadxProject project) {
        Path projectPath = project.getProjectPath();
        if (projectPath != null) {
            return CacheManager.pathToString(projectPath);
        }
        return "tmp:" + CacheManager.buildProjectUniqName(project);
    }

    private static String buildProjectUniqName(JadxProject project) {
        return project.getName() + "-" + FileUtils.buildInputsHash(project.getFilePaths());
    }

    public static String pathToString(Path path) {
        try {
            return path.toAbsolutePath().normalize().toString();
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to expand path: " + path, (Throwable)e);
        }
    }

    private synchronized Map<String, CacheEntry> loadCaches() {
        List list = null;
        if (Files.exists(JadxFiles.CACHES_LIST, new LinkOption[0])) {
            try (BufferedReader reader = Files.newBufferedReader(JadxFiles.CACHES_LIST);){
                list = (List)GSON.fromJson((Reader)reader, CACHES_TYPE);
            }
            catch (Exception e) {
                LOG.warn("Failed to load caches list", (Throwable)e);
            }
        } else {
            return this.initFromRecentProjects();
        }
        if (Utils.isEmpty(list)) {
            return new HashMap<String, CacheEntry>();
        }
        HashMap<String, CacheEntry> map = new HashMap<String, CacheEntry>(list.size());
        for (CacheEntry entry : list) {
            map.put(entry.getProject(), entry);
        }
        return map;
    }

    private synchronized void saveCaches(Map<String, CacheEntry> map) {
        ArrayList<CacheEntry> list = new ArrayList<CacheEntry>(map.values());
        Collections.sort(list);
        String json = GSON.toJson(list, CACHES_TYPE);
        try {
            Files.writeString(JadxFiles.CACHES_LIST, (CharSequence)json, StandardCharsets.UTF_8, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to write caches file", (Throwable)e);
        }
    }

    private Map<String, CacheEntry> initFromRecentProjects() {
        try {
            HashMap<String, CacheEntry> map = new HashMap<String, CacheEntry>();
            long t = System.currentTimeMillis();
            for (Path project : this.settings.getRecentProjects()) {
                try {
                    Path cachePath;
                    ProjectData data = JadxProject.loadProjectData(project);
                    String cacheDir = data.getCacheDir();
                    if (cacheDir == null || !Files.isDirectory(cachePath = this.resolveCacheDirStr(cacheDir, project), new LinkOption[0])) continue;
                    String key = CacheManager.pathToString(project);
                    CacheEntry entry = new CacheEntry();
                    entry.setProject(key);
                    entry.setCache(CacheManager.pathToString(cachePath));
                    entry.setTimestamp(t++);
                    map.put(key, entry);
                }
                catch (Exception e) {
                    LOG.warn("Failed to load project file: {}", (Object)project, (Object)e);
                }
            }
            this.saveCaches(map);
            return map;
        }
        catch (Exception e) {
            LOG.warn("Failed to fill cache list from recent projects", (Throwable)e);
            return new HashMap<String, CacheEntry>();
        }
    }
}

