/*
 * Copyright (C) 2011 the original author or authors.
 * See the notice.md file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.iq80.leveldb.table;

import org.iq80.leveldb.XFilterPolicy;
import org.iq80.leveldb.util.Hash;
import org.iq80.leveldb.util.Slice;

import java.util.List;

/**
 * BloomFilter policy
 *
 * @author Honore Vasconcelos
 * @link https://github.com/google/leveldb/commit/85584d497e7b354853b72f450683d59fcf6b9c5c
 */
public final class BloomFilterPolicy implements org.iq80.leveldb.table.FilterPolicy, XFilterPolicy
{
    private final int bitsPerKey;
    private final int k;

    public BloomFilterPolicy(final int bitsPerKey)
    {
        this.bitsPerKey = bitsPerKey;
        int k = (int) (bitsPerKey * 0.69);
        if (k < 1) {
            k = 1;
        }
        else if (k > 30) {
            k = 30;
        }
        this.k = k;
    }

    @Override
    public String name()
    {
        return "leveldb.BuiltinBloomFilter2";
    }

    @Override
    public byte[] createFilter(List<Slice> keys)
    {
        // Compute bloom filter size (in both bits and bytes)
        int bits = keys.size() * bitsPerKey;

        // For small n, we can see a very high false positive rate.  Fix it
        // by enforcing a minimum bloom filter length.
        if (bits < 64) {
            bits = 64;
        }

        int bytes = (bits + 7) / 8;
        bits = bytes * 8;

        final byte[] array = new byte[bytes + 1];
        array[array.length - 1] = (byte) k; // Remember # of probes in filter

        for (Slice key : keys) {
            // Use double-hashing to generate a sequence of hash values.
            // See analysis in [Kirsch,Mitzenmacher 2006].
            int h = bloomHash(key);
            int delta = (h >>> 17) | (h << 15);  // Rotate right 17 bits
            for (int j = 0; j < k; j++) {
                int bitpos = (int) ((toLong(h)) % bits);
                final int i = bitpos / 8;
                array[i] |= (1 << (bitpos % 8));
                h += delta;
            }
        }
        return array;
    }

    private int bloomHash(Slice data)
    {
        return Hash.hash(data.getRawArray(), data.getRawOffset(), data.length(), 0xbc9f1d34); //avoid data copy
    }

    @Override
    public boolean keyMayMatch(Slice key, Slice bloomFilter1)
    {
        int len = bloomFilter1.length();
        byte[] data = bloomFilter1.getRawArray();
        int offset = bloomFilter1.getRawOffset();
        if (len < 2) {
            return false;
        }

        int bits = (len - 1) * 8;

        // Use the encoded k so that we can read filters generated by
        // bloom filters created using different parameters.
        int k = data[offset + len - 1];
        if (k > 30) {
            // Reserved for potentially new encodings for short bloom filters.
            // Consider it a match.
            return true;
        }

        int h = bloomHash(key);
        int delta = (h >>> 17) | (h << 15);  // Rotate right 17 bits
        for (int j = 0; j < k; j++) {
            int bitpos = (int) (toLong(h) % bits);
            if ((data[offset + (bitpos / 8)] & (1 << (bitpos % 8))) == 0) {
                return false;
            }
            h += delta;
        }
        return true;
    }

    /**
     * Convert an unsigned int into a long
     */
    private long toLong(int h)
    {
        return h & 0xffffffffL;
    }
}
