/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.jspecify.annotations.NullMarked;

import java.io.Serializable;
import java.util.List;
import java.util.Objects;

/**
 * Represents a CWE weakness associated with a CVE.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({"source", "type", "description"})
@NullMarked
public class Weakness implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = -6330752220797574809L;
    /**
     * (Required)
     */
    @JsonProperty("source")
    private String source;
    /**
     * (Required)
     */
    @JsonProperty("type")
    private String type;
    /**
     * (Required)
     */
    @JsonProperty("description")
    private List<LangString> description;

    /**
     * Constructs a new Weakness.
     *
     * @param source the source of the weakness information
     * @param type the type of weakness
     * @param description the weakness descriptions in multiple languages
     */
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    @JsonCreator
    public Weakness(@JsonProperty("source") String source, @JsonProperty("type") String type,
            @JsonProperty("description") List<LangString> description) {
        this.source = source;
        this.type = type;
        this.description = description;
    }

    /**
     * (Required)
     *
     * @return source
     */
    @JsonProperty("source")
    public String getSource() {
        return source;
    }

    /**
     * (Required)
     *
     * @return type
     */
    @JsonProperty("type")
    public String getType() {
        return type;
    }

    /**
     * (Required)
     *
     * @return description
     */
    @JsonProperty("description")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<LangString> getDescription() {
        return description;
    }

    @Override
    public String toString() {
        return "Weakness{" + "source='" + source + '\'' + ", type='" + type + '\'' + ", description=" + description
                + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        Weakness weakness = (Weakness) o;
        return Objects.equals(source, weakness.source) && Objects.equals(type, weakness.type)
                && Objects.equals(description, weakness.description);
    }

    @Override
    public int hashCode() {
        return Objects.hash(source, type, description);
    }
}
