/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.util.List;
import java.util.Objects;

/**
 * Represents a reference to external information about a CVE.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"url", "source", "tags"})
public class Reference implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = -224192309845772254L;
    /**
     * (Required)
     */
    @JsonProperty("url")
    private String url;
    @JsonProperty("source")
    private @Nullable String source;
    @JsonProperty("tags")
    private @Nullable List<String> tags;

    /**
     * Constructs a new Reference with the required URL.
     *
     * @param url the reference URL
     */
    public Reference(@JsonProperty("url") String url) {
        this.url = url;
    }

    /**
     * Constructs a new Reference with all properties.
     *
     * @param url the reference URL
     * @param source the source of the reference
     * @param tags additional tags describing the reference
     */
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public Reference(String url, @Nullable String source, @Nullable List<String> tags) {
        this(url);
        this.source = source;
        this.tags = tags;
    }

    /**
     * (Required)
     *
     * @return url
     */
    @JsonProperty("url")
    public String getUrl() {
        return url;
    }

    /**
     * @return source
     */
    @JsonProperty("source")
    public @Nullable String getSource() {
        return source;
    }

    /**
     * @return tags
     */
    @JsonProperty("tags")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public @Nullable List<String> getTags() {
        return tags;
    }

    @Override
    public String toString() {
        return "Reference{" + "url='" + url + '\'' + ", source='" + source + '\'' + ", tags=" + tags + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        Reference reference = (Reference) o;
        return Objects.equals(url, reference.url) && Objects.equals(source, reference.source)
                && Objects.equals(tags, reference.tags);
    }

    @Override
    public int hashCode() {
        return Objects.hash(url, source, tags);
    }
}
