/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * JSON Schema for Common Vulnerability Scoring System version 3.0
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"version", "vectorString", "attackVector", "attackComplexity", "privilegesRequired",
        "userInteraction", "scope", "confidentialityImpact", "integrityImpact", "availabilityImpact", "baseScore",
        "baseSeverity", "exploitCodeMaturity", "remediationLevel", "reportConfidence", "temporalScore",
        "temporalSeverity", "confidentialityRequirement", "integrityRequirement", "availabilityRequirement",
        "modifiedAttackVector", "modifiedAttackComplexity", "modifiedPrivilegesRequired", "modifiedUserInteraction",
        "modifiedScope", "modifiedConfidentialityImpact", "modifiedIntegrityImpact", "modifiedAvailabilityImpact",
        "environmentalScore", "environmentalSeverity"})
@NullMarked
public class CvssV3Data implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 8537782209754450697L;

    /**
     * Constructor taking only the required set of JSON properties as the minimal constructor for a CvssV3Data.
     *
     * @param version The version
     * @param vectorString The CVSSv3 vector string
     * @param baseScore The CVSSv3 Base score
     * @param baseSeverity The CVSSv3 Base Severity
     */
    @JsonCreator
    public CvssV3Data(@JsonProperty("version") Version version, @JsonProperty("vectorString") String vectorString,
            @JsonProperty("baseScore") Double baseScore, @JsonProperty("baseSeverity") SeverityType baseSeverity) {
        this.version = version;
        this.vectorString = vectorString;
        this.baseScore = baseScore;
        this.baseSeverity = baseSeverity;

    }

    /**
     * Full constructor for CvssV3Data.
     *
     * @param version the CVSS version
     * @param vectorString the CVSS vector string
     * @param attackVector the attack vector
     * @param attackComplexity the attack complexity
     * @param privilegesRequired the privileges required
     * @param userInteraction the user interaction requirement
     * @param scope the scope
     * @param confidentialityImpact the confidentiality impact
     * @param integrityImpact the integrity impact
     * @param availabilityImpact the availability impact
     * @param baseScore the base score
     * @param baseSeverity the base severity
     * @param exploitCodeMaturity the exploit code maturity metric
     * @param remediationLevel the remediation level
     * @param reportConfidence the report confidence
     * @param temporalScore the temporal score
     * @param temporalSeverity the temporal severity
     * @param confidentialityRequirement the confidentiality requirement
     * @param integrityRequirement the integrity requirement
     * @param availabilityRequirement the availability requirement
     * @param modifiedAttackVector the modified attack vector
     * @param modifiedAttackComplexity the modified attack complexity
     * @param modifiedPrivilegesRequired the modified privileges required
     * @param modifiedUserInteraction the modified user interaction
     * @param modifiedScope the modified scope
     * @param modifiedConfidentialityImpact the modified confidentiality impact
     * @param modifiedIntegrityImpact the modified integrity impact
     * @param modifiedAvailabilityImpact the modified availability impact
     * @param environmentalScore the environmental score
     * @param environmentalSeverity the environmental severity
     */
    public CvssV3Data(Version version, String vectorString, @Nullable AttackVectorType attackVector,
            @Nullable AttackComplexityType attackComplexity, @Nullable PrivilegesRequiredType privilegesRequired,
            @Nullable UserInteractionType userInteraction, @Nullable ScopeType scope,
            @Nullable CiaType confidentialityImpact, @Nullable CiaType integrityImpact,
            @Nullable CiaType availabilityImpact, Double baseScore, SeverityType baseSeverity,
            @Nullable ExploitCodeMaturityType exploitCodeMaturity, @Nullable RemediationLevelType remediationLevel,
            @Nullable ConfidenceType reportConfidence, @Nullable Double temporalScore,
            @Nullable SeverityType temporalSeverity, @Nullable CiaRequirementType confidentialityRequirement,
            @Nullable CiaRequirementType integrityRequirement, @Nullable CiaRequirementType availabilityRequirement,
            @Nullable ModifiedAttackVectorType modifiedAttackVector,
            @Nullable ModifiedAttackComplexityType modifiedAttackComplexity,
            @Nullable ModifiedPrivilegesRequiredType modifiedPrivilegesRequired,
            @Nullable ModifiedUserInteractionType modifiedUserInteraction, @Nullable ModifiedScopeType modifiedScope,
            @Nullable ModifiedCiaType modifiedConfidentialityImpact, @Nullable ModifiedCiaType modifiedIntegrityImpact,
            @Nullable ModifiedCiaType modifiedAvailabilityImpact, @Nullable Double environmentalScore,
            @Nullable SeverityType environmentalSeverity) {
        this(version, vectorString, baseScore, baseSeverity);
        this.attackVector = attackVector;
        this.attackComplexity = attackComplexity;
        this.privilegesRequired = privilegesRequired;
        this.userInteraction = userInteraction;
        this.scope = scope;
        this.confidentialityImpact = confidentialityImpact;
        this.integrityImpact = integrityImpact;
        this.availabilityImpact = availabilityImpact;
        this.exploitCodeMaturity = exploitCodeMaturity;
        this.remediationLevel = remediationLevel;
        this.reportConfidence = reportConfidence;
        this.temporalScore = temporalScore;
        this.temporalSeverity = temporalSeverity;
        this.confidentialityRequirement = confidentialityRequirement;
        this.integrityRequirement = integrityRequirement;
        this.availabilityRequirement = availabilityRequirement;
        this.modifiedAttackVector = modifiedAttackVector;
        this.modifiedAttackComplexity = modifiedAttackComplexity;
        this.modifiedPrivilegesRequired = modifiedPrivilegesRequired;
        this.modifiedUserInteraction = modifiedUserInteraction;
        this.modifiedScope = modifiedScope;
        this.modifiedConfidentialityImpact = modifiedConfidentialityImpact;
        this.modifiedIntegrityImpact = modifiedIntegrityImpact;
        this.modifiedAvailabilityImpact = modifiedAvailabilityImpact;
        this.environmentalScore = environmentalScore;
        this.environmentalSeverity = environmentalSeverity;
    }

    /**
     * CVSS Version (Required)
     */
    @JsonProperty("version")
    @JsonPropertyDescription("CVSS Version")
    private Version version;
    /**
     * (Required)
     */
    @JsonProperty("vectorString")
    private String vectorString;
    @JsonProperty("attackVector")
    private @Nullable AttackVectorType attackVector;
    @JsonProperty("attackComplexity")
    private @Nullable AttackComplexityType attackComplexity;
    @JsonProperty("privilegesRequired")
    private @Nullable PrivilegesRequiredType privilegesRequired;
    @JsonProperty("userInteraction")
    private @Nullable UserInteractionType userInteraction;
    @JsonProperty("scope")
    private @Nullable ScopeType scope;
    @JsonProperty("confidentialityImpact")
    private @Nullable CiaType confidentialityImpact;
    @JsonProperty("integrityImpact")
    private @Nullable CiaType integrityImpact;
    @JsonProperty("availabilityImpact")
    private @Nullable CiaType availabilityImpact;
    /**
     * (Required)
     */
    @JsonProperty("baseScore")
    private Double baseScore;
    /**
     * (Required)
     */
    @JsonProperty("baseSeverity")
    private SeverityType baseSeverity;
    @JsonProperty("exploitCodeMaturity")
    private @Nullable ExploitCodeMaturityType exploitCodeMaturity;
    @JsonProperty("remediationLevel")
    private @Nullable RemediationLevelType remediationLevel;
    @JsonProperty("reportConfidence")
    private @Nullable ConfidenceType reportConfidence;
    @JsonProperty("temporalScore")
    private @Nullable Double temporalScore;
    @JsonProperty("temporalSeverity")
    private @Nullable SeverityType temporalSeverity;
    @JsonProperty("confidentialityRequirement")
    private @Nullable CiaRequirementType confidentialityRequirement;
    @JsonProperty("integrityRequirement")
    private @Nullable CiaRequirementType integrityRequirement;
    @JsonProperty("availabilityRequirement")
    private @Nullable CiaRequirementType availabilityRequirement;
    @JsonProperty("modifiedAttackVector")
    private @Nullable ModifiedAttackVectorType modifiedAttackVector;
    @JsonProperty("modifiedAttackComplexity")
    private @Nullable ModifiedAttackComplexityType modifiedAttackComplexity;
    @JsonProperty("modifiedPrivilegesRequired")
    private @Nullable ModifiedPrivilegesRequiredType modifiedPrivilegesRequired;
    @JsonProperty("modifiedUserInteraction")
    private @Nullable ModifiedUserInteractionType modifiedUserInteraction;
    @JsonProperty("modifiedScope")
    private @Nullable ModifiedScopeType modifiedScope;
    @JsonProperty("modifiedConfidentialityImpact")
    private @Nullable ModifiedCiaType modifiedConfidentialityImpact;
    @JsonProperty("modifiedIntegrityImpact")
    private @Nullable ModifiedCiaType modifiedIntegrityImpact;
    @JsonProperty("modifiedAvailabilityImpact")
    private @Nullable ModifiedCiaType modifiedAvailabilityImpact;
    @JsonProperty("environmentalScore")
    private @Nullable Double environmentalScore;
    @JsonProperty("environmentalSeverity")
    private @Nullable SeverityType environmentalSeverity;

    /**
     * CVSS Version (Required)
     *
     * @return version
     */
    @JsonProperty("version")
    public Version getVersion() {
        return version;
    }

    /**
     * (Required)
     *
     * @return vectorString
     */
    @JsonProperty("vectorString")
    public String getVectorString() {
        return vectorString;
    }

    /**
     * @return attackVector
     */
    @JsonProperty("attackVector")
    public @Nullable AttackVectorType getAttackVector() {
        return attackVector;
    }

    /**
     * @return attackComplexity
     */
    @JsonProperty("attackComplexity")
    public @Nullable AttackComplexityType getAttackComplexity() {
        return attackComplexity;
    }

    /**
     * @return privilegesRequired
     */
    @JsonProperty("privilegesRequired")
    public @Nullable PrivilegesRequiredType getPrivilegesRequired() {
        return privilegesRequired;
    }

    /**
     * @return userInteraction
     */
    @JsonProperty("userInteraction")
    public @Nullable UserInteractionType getUserInteraction() {
        return userInteraction;
    }

    /**
     * @return scope
     */
    @JsonProperty("scope")
    public @Nullable ScopeType getScope() {
        return scope;
    }

    /**
     * @return confidentialityImpact
     */
    @JsonProperty("confidentialityImpact")
    public @Nullable CiaType getConfidentialityImpact() {
        return confidentialityImpact;
    }

    /**
     * @return integrityImpact
     */
    @JsonProperty("integrityImpact")
    public @Nullable CiaType getIntegrityImpact() {
        return integrityImpact;
    }

    /**
     * @return availabilityImpact
     */
    @JsonProperty("availabilityImpact")
    public @Nullable CiaType getAvailabilityImpact() {
        return availabilityImpact;
    }

    /**
     * (Required)
     *
     * @return baseScore
     */
    @JsonProperty("baseScore")
    public Double getBaseScore() {
        return baseScore;
    }

    /**
     * (Required)
     *
     * @return baseSeverity
     */
    @JsonProperty("baseSeverity")
    public SeverityType getBaseSeverity() {
        return baseSeverity;
    }

    /**
     * @return exploitCodeMaturity
     */
    @JsonProperty("exploitCodeMaturity")
    public @Nullable ExploitCodeMaturityType getExploitCodeMaturity() {
        return exploitCodeMaturity;
    }

    /**
     * @return remediationLevel
     */
    @JsonProperty("remediationLevel")
    public @Nullable RemediationLevelType getRemediationLevel() {
        return remediationLevel;
    }

    /**
     * @return reportConfidence
     */
    @JsonProperty("reportConfidence")
    public @Nullable ConfidenceType getReportConfidence() {
        return reportConfidence;
    }

    /**
     * @return temporalScore
     */
    @JsonProperty("temporalScore")
    public @Nullable Double getTemporalScore() {
        return temporalScore;
    }

    /**
     * @return temporalSeverity
     */
    @JsonProperty("temporalSeverity")
    public @Nullable SeverityType getTemporalSeverity() {
        return temporalSeverity;
    }

    /**
     * @return confidentialityRequirement
     */
    @JsonProperty("confidentialityRequirement")
    public @Nullable CiaRequirementType getConfidentialityRequirement() {
        return confidentialityRequirement;
    }

    /**
     * @return integrityRequirement
     */
    @JsonProperty("integrityRequirement")
    public @Nullable CiaRequirementType getIntegrityRequirement() {
        return integrityRequirement;
    }

    /**
     * @return availabilityRequirement
     */
    @JsonProperty("availabilityRequirement")
    public @Nullable CiaRequirementType getAvailabilityRequirement() {
        return availabilityRequirement;
    }

    /**
     * @return modifiedAttackVector
     */
    @JsonProperty("modifiedAttackVector")
    public @Nullable ModifiedAttackVectorType getModifiedAttackVector() {
        return modifiedAttackVector;
    }

    /**
     * @return modifiedAttackComplexity
     */
    @JsonProperty("modifiedAttackComplexity")
    public @Nullable ModifiedAttackComplexityType getModifiedAttackComplexity() {
        return modifiedAttackComplexity;
    }

    /**
     * @return modifiedPrivilegesRequired
     */
    @JsonProperty("modifiedPrivilegesRequired")
    public @Nullable ModifiedPrivilegesRequiredType getModifiedPrivilegesRequired() {
        return modifiedPrivilegesRequired;
    }

    /**
     * @return modifiedUserInteraction
     */
    @JsonProperty("modifiedUserInteraction")
    public @Nullable ModifiedUserInteractionType getModifiedUserInteraction() {
        return modifiedUserInteraction;
    }

    /**
     * @return modifiedScope
     */
    @JsonProperty("modifiedScope")
    public @Nullable ModifiedScopeType getModifiedScope() {
        return modifiedScope;
    }

    /**
     * @return modifiedConfidentialityImpact
     */
    @JsonProperty("modifiedConfidentialityImpact")
    public @Nullable ModifiedCiaType getModifiedConfidentialityImpact() {
        return modifiedConfidentialityImpact;
    }

    /**
     * @return modifiedIntegrityImpact
     */
    @JsonProperty("modifiedIntegrityImpact")
    public @Nullable ModifiedCiaType getModifiedIntegrityImpact() {
        return modifiedIntegrityImpact;
    }

    /**
     * @return modifiedAvailabilityImpact
     */
    @JsonProperty("modifiedAvailabilityImpact")
    public @Nullable ModifiedCiaType getModifiedAvailabilityImpact() {
        return modifiedAvailabilityImpact;
    }

    /**
     * @return environmentalScore
     */
    @JsonProperty("environmentalScore")
    public @Nullable Double getEnvironmentalScore() {
        return environmentalScore;
    }

    /**
     * @return environmentalSeverity
     */
    @JsonProperty("environmentalSeverity")
    public @Nullable SeverityType getEnvironmentalSeverity() {
        return environmentalSeverity;
    }

    @Override
    public String toString() {
        return String.format("CVSS:%s/AV:%s/AC:%s/PR:%s/UI:%s/S:%s/C:%s/I:%s/A:%s", version,
                attackVector == null ? "" : attackVector.value().substring(0, 1),
                attackComplexity == null ? "" : attackComplexity.value().substring(0, 1),
                privilegesRequired == null ? "" : privilegesRequired.value().substring(0, 1),
                userInteraction == null ? "" : userInteraction.value().substring(0, 1),
                scope == null ? "" : scope.value().substring(0, 1),
                confidentialityImpact == null ? "" : confidentialityImpact.value().substring(0, 1),
                integrityImpact == null ? "" : integrityImpact.value().substring(0, 1),
                availabilityImpact == null ? "" : availabilityImpact.value().substring(0, 1));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        CvssV3Data that = (CvssV3Data) o;
        return version == that.version && Objects.equals(vectorString, that.vectorString)
                && attackVector == that.attackVector && attackComplexity == that.attackComplexity
                && privilegesRequired == that.privilegesRequired && userInteraction == that.userInteraction
                && scope == that.scope && confidentialityImpact == that.confidentialityImpact
                && integrityImpact == that.integrityImpact && availabilityImpact == that.availabilityImpact
                && Objects.equals(baseScore, that.baseScore) && baseSeverity == that.baseSeverity
                && exploitCodeMaturity == that.exploitCodeMaturity && remediationLevel == that.remediationLevel
                && reportConfidence == that.reportConfidence && Objects.equals(temporalScore, that.temporalScore)
                && temporalSeverity == that.temporalSeverity
                && confidentialityRequirement == that.confidentialityRequirement
                && integrityRequirement == that.integrityRequirement
                && availabilityRequirement == that.availabilityRequirement
                && modifiedAttackVector == that.modifiedAttackVector
                && modifiedAttackComplexity == that.modifiedAttackComplexity
                && modifiedPrivilegesRequired == that.modifiedPrivilegesRequired
                && modifiedUserInteraction == that.modifiedUserInteraction && modifiedScope == that.modifiedScope
                && modifiedConfidentialityImpact == that.modifiedConfidentialityImpact
                && modifiedIntegrityImpact == that.modifiedIntegrityImpact
                && modifiedAvailabilityImpact == that.modifiedAvailabilityImpact
                && Objects.equals(environmentalScore, that.environmentalScore)
                && environmentalSeverity == that.environmentalSeverity;
    }

    @Override
    public int hashCode() {
        return Objects.hash(version, vectorString, attackVector, attackComplexity, privilegesRequired, userInteraction,
                scope, confidentialityImpact, integrityImpact, availabilityImpact, baseScore, baseSeverity,
                exploitCodeMaturity, remediationLevel, reportConfidence, temporalScore, temporalSeverity,
                confidentialityRequirement, integrityRequirement, availabilityRequirement, modifiedAttackVector,
                modifiedAttackComplexity, modifiedPrivilegesRequired, modifiedUserInteraction, modifiedScope,
                modifiedConfidentialityImpact, modifiedIntegrityImpact, modifiedAvailabilityImpact, environmentalScore,
                environmentalSeverity);
    }

    /**
     * CVSS v3 Attack Complexity metric values.
     */
    public enum AttackComplexityType {
        /** High attack complexity. */
        HIGH("HIGH"),
        /** Low attack complexity. */
        LOW("LOW");

        private static final Map<String, AttackComplexityType> CONSTANTS = new HashMap<>();

        static {
            for (AttackComplexityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackComplexityType(String value) {
            this.value = value;
        }

        /**
         * Creates an AttackComplexityType from its string value.
         *
         * @param value the string value
         * @return the AttackComplexityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AttackComplexityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (AttackComplexityType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            AttackComplexityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this attack complexity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Attack Vector metric values.
     */
    public enum AttackVectorType {
        /** Network access vector. */
        NETWORK("NETWORK"),
        /** Adjacent network access vector. */
        ADJACENT_NETWORK("ADJACENT_NETWORK"),
        /** Local access vector. */
        LOCAL("LOCAL"),
        /** Physical access vector. */
        PHYSICAL("PHYSICAL");

        private static final Map<String, AttackVectorType> CONSTANTS = new HashMap<>();

        static {
            for (AttackVectorType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackVectorType(String value) {
            this.value = value;
        }

        /**
         * Creates an AttackVectorType from its string value.
         *
         * @param value the string value
         * @return the AttackVectorType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AttackVectorType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (AttackVectorType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            AttackVectorType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this attack vector type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Confidentiality, Integrity, and Availability requirement values for environmental scoring.
     */
    public enum CiaRequirementType {
        /** Low requirement. */
        LOW("LOW"),
        /** Medium requirement. */
        MEDIUM("MEDIUM"),
        /** High requirement. */
        HIGH("HIGH"),
        /** Not defined requirement. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, CiaRequirementType> CONSTANTS = new HashMap<>();

        static {
            for (CiaRequirementType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaRequirementType(String value) {
            this.value = value;
        }

        /**
         * Creates a CiaRequirementType from its string value.
         *
         * @param value the string value
         * @return the CiaRequirementType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static CiaRequirementType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (CiaRequirementType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            CiaRequirementType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this CIA requirement type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Confidentiality, Integrity, and Availability impact values.
     */
    public enum CiaType {
        /** No impact. */
        NONE("NONE"),
        /** Low impact. */
        LOW("LOW"),
        /** High impact. */
        HIGH("HIGH");

        private static final Map<String, CiaType> CONSTANTS = new HashMap<>();

        static {
            for (CiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaType(String value) {
            this.value = value;
        }

        /**
         * Creates a CiaType from its string value.
         *
         * @param value the string value
         * @return the CiaType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static CiaType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CiaType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            CiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this CIA type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Report Confidence metric values for temporal scoring.
     */
    public enum ConfidenceType {
        /** Unknown confidence. */
        UNKNOWN("UNKNOWN"),
        /** Reasonable confidence. */
        REASONABLE("REASONABLE"),
        /** Confirmed confidence. */
        CONFIRMED("CONFIRMED"),
        /** Not defined confidence. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ConfidenceType> CONSTANTS = new HashMap<>();

        static {
            for (ConfidenceType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ConfidenceType(String value) {
            this.value = value;
        }

        /**
         * Creates a ConfidenceType from its string value.
         *
         * @param value the string value
         * @return the ConfidenceType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ConfidenceType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ConfidenceType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ConfidenceType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this confidence type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Exploit Code Maturity metric values for temporal scoring.
     */
    public enum ExploitCodeMaturityType {
        /** Unproven exploit exists. */
        UNPROVEN("UNPROVEN"),
        /** Proof-of-concept exploit exists. */
        PROOF_OF_CONCEPT("PROOF_OF_CONCEPT"),
        /** Functional exploit exists. */
        FUNCTIONAL("FUNCTIONAL"),
        /** High exploit maturity. */
        HIGH("HIGH"),
        /** Not defined exploit code maturity. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ExploitCodeMaturityType> CONSTANTS = new HashMap<>();

        static {
            for (ExploitCodeMaturityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ExploitCodeMaturityType(String value) {
            this.value = value;
        }

        /**
         * Creates an ExploitCodeMaturityType from its string value.
         *
         * @param value the string value
         * @return the ExploitCodeMaturityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ExploitCodeMaturityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ExploitCodeMaturityType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ExploitCodeMaturityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this exploit code maturity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Modified Attack Complexity metric values for environmental scoring.
     */
    public enum ModifiedAttackComplexityType {
        /** High attack complexity. */
        HIGH("HIGH"),
        /** Low attack complexity. */
        LOW("LOW"),
        /** Not defined attack complexity. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedAttackComplexityType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackComplexityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackComplexityType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedAttackComplexityType from its string value.
         *
         * @param value the string value
         * @return the ModifiedAttackComplexityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedAttackComplexityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackComplexityType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ModifiedAttackComplexityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified attack complexity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Modified Attack Vector metric values for environmental scoring.
     */
    public enum ModifiedAttackVectorType {
        /** Network access vector. */
        NETWORK("NETWORK"),
        /** Adjacent network access vector. */
        ADJACENT_NETWORK("ADJACENT_NETWORK"),
        /** Local access vector. */
        LOCAL("LOCAL"),
        /** Physical access vector. */
        PHYSICAL("PHYSICAL"),
        /** Not defined attack vector. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedAttackVectorType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackVectorType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackVectorType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedAttackVectorType from its string value.
         *
         * @param value the string value
         * @return the ModifiedAttackVectorType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedAttackVectorType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackVectorType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ModifiedAttackVectorType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified attack vector type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Modified Confidentiality, Integrity, and Availability impact values for environmental scoring.
     */
    public enum ModifiedCiaType {
        /** No impact. */
        NONE("NONE"),
        /** Low impact. */
        LOW("LOW"),
        /** High impact. */
        HIGH("HIGH"),
        /** Not defined impact. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedCiaType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedCiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedCiaType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedCiaType from its string value.
         *
         * @param value the string value
         * @return the ModifiedCiaType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedCiaType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedCiaType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ModifiedCiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified CIA type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Modified Privileges Required metric values for environmental scoring.
     */
    public enum ModifiedPrivilegesRequiredType {
        /** High privileges required. */
        HIGH("HIGH"),
        /** Low privileges required. */
        LOW("LOW"),
        /** No privileges required. */
        NONE("NONE"),
        /** Not defined privileges required. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedPrivilegesRequiredType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedPrivilegesRequiredType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedPrivilegesRequiredType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedPrivilegesRequiredType from its string value.
         *
         * @param value the string value
         * @return the ModifiedPrivilegesRequiredType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedPrivilegesRequiredType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedPrivilegesRequiredType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ModifiedPrivilegesRequiredType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified privileges required type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Modified Scope metric values for environmental scoring.
     */
    public enum ModifiedScopeType {
        /** Unchanged scope. */
        UNCHANGED("UNCHANGED"),
        /** Changed scope. */
        CHANGED("CHANGED"),
        /** Not defined scope. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedScopeType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedScopeType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedScopeType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedScopeType from its string value.
         *
         * @param value the string value
         * @return the ModifiedScopeType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedScopeType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedScopeType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ModifiedScopeType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified scope type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Modified User Interaction metric values for environmental scoring.
     */
    public enum ModifiedUserInteractionType {
        /** No user interaction required. */
        NONE("NONE"),
        /** User interaction required. */
        REQUIRED("REQUIRED"),
        /** Not defined user interaction. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedUserInteractionType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedUserInteractionType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedUserInteractionType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedUserInteractionType from its string value.
         *
         * @param value the string value
         * @return the ModifiedUserInteractionType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedUserInteractionType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedUserInteractionType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ModifiedUserInteractionType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified user interaction type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Privileges Required metric values.
     */
    public enum PrivilegesRequiredType {
        /** High privileges required. */
        HIGH("HIGH"),
        /** Low privileges required. */
        LOW("LOW"),
        /** No privileges required. */
        NONE("NONE");

        private static final Map<String, PrivilegesRequiredType> CONSTANTS = new HashMap<>();

        static {
            for (PrivilegesRequiredType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        PrivilegesRequiredType(String value) {
            this.value = value;
        }

        /**
         * Creates a PrivilegesRequiredType from its string value.
         *
         * @param value the string value
         * @return the PrivilegesRequiredType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static PrivilegesRequiredType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (PrivilegesRequiredType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            PrivilegesRequiredType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this privileges required type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Remediation Level metric values for temporal scoring.
     */
    public enum RemediationLevelType {
        /** Official fix is available. */
        OFFICIAL_FIX("OFFICIAL_FIX"),
        /** Temporary fix is available. */
        TEMPORARY_FIX("TEMPORARY_FIX"),
        /** Workaround is available. */
        WORKAROUND("WORKAROUND"),
        /** Remediation is unavailable. */
        UNAVAILABLE("UNAVAILABLE"),
        /** Not defined remediation level. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, RemediationLevelType> CONSTANTS = new HashMap<>();

        static {
            for (RemediationLevelType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        RemediationLevelType(String value) {
            this.value = value;
        }

        /**
         * Creates a RemediationLevelType from its string value.
         *
         * @param value the string value
         * @return the RemediationLevelType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static RemediationLevelType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (RemediationLevelType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            RemediationLevelType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this remediation level type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Scope metric values.
     */
    public enum ScopeType {
        /** Unchanged scope. */
        UNCHANGED("UNCHANGED"),
        /** Changed scope. */
        CHANGED("CHANGED");

        private static final Map<String, ScopeType> CONSTANTS = new HashMap<>();

        static {
            for (ScopeType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ScopeType(String value) {
            this.value = value;
        }

        /**
         * Creates a ScopeType from its string value.
         *
         * @param value the string value
         * @return the ScopeType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ScopeType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (ScopeType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ScopeType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this scope type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 Severity rating values.
     */
    public enum SeverityType {
        /** None severity. */
        NONE("NONE"),
        /** Low severity. */
        LOW("LOW"),
        /** Medium severity. */
        MEDIUM("MEDIUM"),
        /** High severity. */
        HIGH("HIGH"),
        /** Critical severity. */
        CRITICAL("CRITICAL");

        private static final Map<String, SeverityType> CONSTANTS = new HashMap<>();

        static {
            for (SeverityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        SeverityType(String value) {
            this.value = value;
        }

        /**
         * Creates a SeverityType from its string value.
         *
         * @param value the string value
         * @return the SeverityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static SeverityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (SeverityType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            SeverityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this severity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v3 User Interaction metric values.
     */
    public enum UserInteractionType {
        /** No user interaction required. */
        NONE("NONE"),
        /** User interaction required. */
        REQUIRED("REQUIRED");

        private static final Map<String, UserInteractionType> CONSTANTS = new HashMap<>();

        static {
            for (UserInteractionType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        UserInteractionType(String value) {
            this.value = value;
        }

        /**
         * Creates a UserInteractionType from its string value.
         *
         * @param value the string value
         * @return the UserInteractionType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static UserInteractionType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (UserInteractionType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            UserInteractionType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this user interaction type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS Version
     */
    public enum Version {
        /** CVSS version 3.0. */
        _3_0("3.0"),
        /** CVSS version 3.1. */
        _3_1("3.1");

        private static final Map<String, Version> CONSTANTS = new HashMap<>();

        static {
            for (Version c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        Version(String value) {
            this.value = value;
        }

        /**
         * Creates a Version from its string value.
         *
         * @param value the string value
         * @return the Version enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static Version fromValue(String value) {
            Version constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this version.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

}
