/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * JSON Schema for Common Vulnerability Scoring System version 2.0
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"version", "vectorString", "accessVector", "accessComplexity", "authentication",
        "confidentialityImpact", "integrityImpact", "availabilityImpact", "baseScore", "exploitability",
        "remediationLevel", "reportConfidence", "temporalScore", "collateralDamagePotential", "targetDistribution",
        "confidentialityRequirement", "integrityRequirement", "availabilityRequirement", "environmentalScore"})
@NullMarked
public class CvssV2Data implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = -3488320581980953116L;

    /**
     * Constructor taking only the required set of JSON properties as the minimal constructor for a CvssV2Data.
     *
     * @param version The version
     * @param vectorString The CVSSv2 vector string
     * @param baseScore The CVSSv2 Base score
     */
    @JsonCreator
    public CvssV2Data(@JsonProperty("version") Version version, @JsonProperty("vectorString") String vectorString,
            @JsonProperty("baseScore") Double baseScore) {
        this.version = version;
        this.vectorString = vectorString;
        this.baseScore = baseScore;
    }

    /**
     * Full constructor for CvssV2Data.
     *
     * @param version the CVSS version
     * @param vectorString the CVSS vector string
     * @param accessVector the access vector
     * @param accessComplexity the access complexity
     * @param authentication the authentication requirement
     * @param confidentialityImpact the confidentiality impact
     * @param integrityImpact the integrity impact
     * @param availabilityImpact the availability impact
     * @param baseScore the base score
     * @param baseSeverity the base severity
     * @param exploitability the exploitability metric
     * @param remediationLevel the remediation level
     * @param reportConfidence the report confidence
     * @param temporalScore the temporal score
     * @param collateralDamagePotential the collateral damage potential
     * @param targetDistribution the target distribution
     * @param confidentialityRequirement the confidentiality requirement
     * @param integrityRequirement the integrity requirement
     * @param availabilityRequirement the availability requirement
     * @param environmentalScore the environmental score
     */
    public CvssV2Data(Version version, String vectorString, @Nullable AccessVectorType accessVector,
            @Nullable AccessComplexityType accessComplexity, @Nullable AuthenticationType authentication,
            @Nullable CiaType confidentialityImpact, @Nullable CiaType integrityImpact,
            @Nullable CiaType availabilityImpact, Double baseScore, @Nullable String baseSeverity,
            @Nullable ExploitabilityType exploitability, @Nullable RemediationLevelType remediationLevel,
            @Nullable ReportConfidenceType reportConfidence, @Nullable Double temporalScore,
            @Nullable CollateralDamagePotentialType collateralDamagePotential,
            @Nullable TargetDistributionType targetDistribution,
            @Nullable CiaRequirementType confidentialityRequirement, @Nullable CiaRequirementType integrityRequirement,
            @Nullable CiaRequirementType availabilityRequirement, @Nullable Double environmentalScore) {
        this(version, vectorString, baseScore);
        this.accessVector = accessVector;
        this.accessComplexity = accessComplexity;
        this.authentication = authentication;
        this.confidentialityImpact = confidentialityImpact;
        this.integrityImpact = integrityImpact;
        this.availabilityImpact = availabilityImpact;
        this.baseSeverity = baseSeverity;
        this.exploitability = exploitability;
        this.remediationLevel = remediationLevel;
        this.reportConfidence = reportConfidence;
        this.temporalScore = temporalScore;
        this.collateralDamagePotential = collateralDamagePotential;
        this.targetDistribution = targetDistribution;
        this.confidentialityRequirement = confidentialityRequirement;
        this.integrityRequirement = integrityRequirement;
        this.availabilityRequirement = availabilityRequirement;
        this.environmentalScore = environmentalScore;
    }

    /**
     * CVSS Version (Required)
     */
    @JsonProperty("version")
    @JsonPropertyDescription("CVSS Version")
    private Version version;
    /**
     * (Required)
     */
    @JsonProperty("vectorString")
    private String vectorString;
    @JsonProperty("accessVector")
    private @Nullable AccessVectorType accessVector;
    @JsonProperty("accessComplexity")
    private @Nullable AccessComplexityType accessComplexity;
    @JsonProperty("authentication")
    private @Nullable AuthenticationType authentication;
    @JsonProperty("confidentialityImpact")
    private @Nullable CiaType confidentialityImpact;
    @JsonProperty("integrityImpact")
    private @Nullable CiaType integrityImpact;
    @JsonProperty("availabilityImpact")
    private @Nullable CiaType availabilityImpact;
    /**
     * (Required)
     */
    @JsonProperty("baseScore")
    private Double baseScore;
    @JsonProperty("baseSeverity")
    private @Nullable String baseSeverity;
    @JsonProperty("exploitability")
    private @Nullable ExploitabilityType exploitability;
    @JsonProperty("remediationLevel")
    private @Nullable RemediationLevelType remediationLevel;
    @JsonProperty("reportConfidence")
    private @Nullable ReportConfidenceType reportConfidence;
    @JsonProperty("temporalScore")
    private @Nullable Double temporalScore;
    @JsonProperty("collateralDamagePotential")
    private @Nullable CollateralDamagePotentialType collateralDamagePotential;
    @JsonProperty("targetDistribution")
    private @Nullable TargetDistributionType targetDistribution;
    @JsonProperty("confidentialityRequirement")
    private @Nullable CiaRequirementType confidentialityRequirement;
    @JsonProperty("integrityRequirement")
    private @Nullable CiaRequirementType integrityRequirement;
    @JsonProperty("availabilityRequirement")
    private @Nullable CiaRequirementType availabilityRequirement;
    @JsonProperty("environmentalScore")
    private @Nullable Double environmentalScore;

    /**
     * CVSS Version (Required)
     *
     * @return version
     */
    @JsonProperty("version")
    public Version getVersion() {
        return version;
    }

    /**
     * (Required)
     *
     * @return vectorString
     */
    @JsonProperty("vectorString")
    public String getVectorString() {
        return vectorString;
    }

    /**
     * @return accessVector
     */
    @JsonProperty("accessVector")
    public @Nullable AccessVectorType getAccessVector() {
        return accessVector;
    }

    /**
     * @return accessComplexity
     */
    @JsonProperty("accessComplexity")
    public @Nullable AccessComplexityType getAccessComplexity() {
        return accessComplexity;
    }

    /**
     * @return authentication
     */
    @JsonProperty("authentication")
    public @Nullable AuthenticationType getAuthentication() {
        return authentication;
    }

    /**
     * @return confidentialityImpact
     */
    @JsonProperty("confidentialityImpact")
    public @Nullable CiaType getConfidentialityImpact() {
        return confidentialityImpact;
    }

    /**
     * @return integrityImpact
     */
    @JsonProperty("integrityImpact")
    public @Nullable CiaType getIntegrityImpact() {
        return integrityImpact;
    }

    /**
     * @return availabilityImpact
     */
    @JsonProperty("availabilityImpact")
    public @Nullable CiaType getAvailabilityImpact() {
        return availabilityImpact;
    }

    /**
     * (Required)
     *
     * @return baseScore
     */
    @JsonProperty("baseScore")
    public Double getBaseScore() {
        return baseScore;
    }

    /**
     * @return baseSeverity
     */
    @JsonProperty("baseSeverity")
    public @Nullable String getBaseSeverity() {
        return baseSeverity;
    }

    /**
     * @return exploitability
     */
    @JsonProperty("exploitability")
    public @Nullable ExploitabilityType getExploitability() {
        return exploitability;
    }

    /**
     * @return remediationLevel
     */
    @JsonProperty("remediationLevel")
    public @Nullable RemediationLevelType getRemediationLevel() {
        return remediationLevel;
    }

    /**
     * @return reportConfidence
     */
    @JsonProperty("reportConfidence")
    public @Nullable ReportConfidenceType getReportConfidence() {
        return reportConfidence;
    }

    /**
     * @return temporalScore
     */
    @JsonProperty("temporalScore")
    public @Nullable Double getTemporalScore() {
        return temporalScore;
    }

    /**
     * @return collateralDamagePotential
     */
    @JsonProperty("collateralDamagePotential")
    public @Nullable CollateralDamagePotentialType getCollateralDamagePotential() {
        return collateralDamagePotential;
    }

    /**
     * @return targetDistribution
     */
    @JsonProperty("targetDistribution")
    public @Nullable TargetDistributionType getTargetDistribution() {
        return targetDistribution;
    }

    /**
     * @return confidentialityRequirement
     */
    @JsonProperty("confidentialityRequirement")
    public @Nullable CiaRequirementType getConfidentialityRequirement() {
        return confidentialityRequirement;
    }

    /**
     * @return integrityRequirement
     */
    @JsonProperty("integrityRequirement")
    public @Nullable CiaRequirementType getIntegrityRequirement() {
        return integrityRequirement;
    }

    /**
     * @return availabilityRequirement
     */
    @JsonProperty("availabilityRequirement")
    public @Nullable CiaRequirementType getAvailabilityRequirement() {
        return availabilityRequirement;
    }

    /**
     * @return environmentalScore
     */
    @JsonProperty("environmentalScore")
    public @Nullable Double getEnvironmentalScore() {
        return environmentalScore;
    }

    @Override
    public String toString() {
        return "CvssV2Data{" + "version=" + version + ", vectorString='" + vectorString + '\'' + ", accessVector="
                + accessVector + ", accessComplexity=" + accessComplexity + ", authentication=" + authentication
                + ", confidentialityImpact=" + confidentialityImpact + ", integrityImpact=" + integrityImpact
                + ", availabilityImpact=" + availabilityImpact + ", baseScore=" + baseScore + ", exploitability="
                + exploitability + ", remediationLevel=" + remediationLevel + ", reportConfidence=" + reportConfidence
                + ", temporalScore=" + temporalScore + ", collateralDamagePotential=" + collateralDamagePotential
                + ", targetDistribution=" + targetDistribution + ", confidentialityRequirement="
                + confidentialityRequirement + ", integrityRequirement=" + integrityRequirement
                + ", availabilityRequirement=" + availabilityRequirement + ", environmentalScore=" + environmentalScore
                + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        CvssV2Data cvssV2Data = (CvssV2Data) o;
        return version == cvssV2Data.version && Objects.equals(vectorString, cvssV2Data.vectorString)
                && accessVector == cvssV2Data.accessVector && accessComplexity == cvssV2Data.accessComplexity
                && authentication == cvssV2Data.authentication
                && confidentialityImpact == cvssV2Data.confidentialityImpact
                && integrityImpact == cvssV2Data.integrityImpact && availabilityImpact == cvssV2Data.availabilityImpact
                && Objects.equals(baseScore, cvssV2Data.baseScore) && exploitability == cvssV2Data.exploitability
                && remediationLevel == cvssV2Data.remediationLevel && reportConfidence == cvssV2Data.reportConfidence
                && Objects.equals(temporalScore, cvssV2Data.temporalScore)
                && collateralDamagePotential == cvssV2Data.collateralDamagePotential
                && targetDistribution == cvssV2Data.targetDistribution
                && confidentialityRequirement == cvssV2Data.confidentialityRequirement
                && integrityRequirement == cvssV2Data.integrityRequirement
                && availabilityRequirement == cvssV2Data.availabilityRequirement
                && Objects.equals(environmentalScore, cvssV2Data.environmentalScore);
    }

    @Override
    public int hashCode() {
        return Objects.hash(version, vectorString, accessVector, accessComplexity, authentication,
                confidentialityImpact, integrityImpact, availabilityImpact, baseScore, exploitability, remediationLevel,
                reportConfidence, temporalScore, collateralDamagePotential, targetDistribution,
                confidentialityRequirement, integrityRequirement, availabilityRequirement, environmentalScore);
    }

    /**
     * CVSS v2 Access Complexity metric values.
     */
    public enum AccessComplexityType {
        /** High access complexity. */
        HIGH("HIGH"),
        /** Medium access complexity. */
        MEDIUM("MEDIUM"),
        /** Low access complexity. */
        LOW("LOW");

        private static final Map<String, AccessComplexityType> CONSTANTS = new HashMap<>();

        static {
            for (AccessComplexityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AccessComplexityType(String value) {
            this.value = value;
        }

        /**
         * Creates an AccessComplexityType from its string value.
         *
         * @param value the string value
         * @return the AccessComplexityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AccessComplexityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CvssV2Data.AccessComplexityType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            AccessComplexityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this access complexity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Access Vector metric values.
     */
    public enum AccessVectorType {
        /** Network access vector. */
        NETWORK("NETWORK"),
        /** Adjacent network access vector. */
        ADJACENT_NETWORK("ADJACENT_NETWORK"),
        /** Local access vector. */
        LOCAL("LOCAL");

        private static final Map<String, AccessVectorType> CONSTANTS = new HashMap<>();

        static {
            for (AccessVectorType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AccessVectorType(String value) {
            this.value = value;
        }

        /**
         * Creates an AccessVectorType from its string value.
         *
         * @param value the string value
         * @return the AccessVectorType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AccessVectorType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CvssV2Data.AccessVectorType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            AccessVectorType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this access vector type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Authentication metric values.
     */
    public enum AuthenticationType {
        /** Multiple instances of authentication required. */
        MULTIPLE("MULTIPLE"),
        /** Single instance of authentication required. */
        SINGLE("SINGLE"),
        /** No authentication required. */
        NONE("NONE");

        private static final Map<String, AuthenticationType> CONSTANTS = new HashMap<>();

        static {
            for (AuthenticationType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AuthenticationType(String value) {
            this.value = value;
        }

        /**
         * Creates an AuthenticationType from its string value.
         *
         * @param value the string value
         * @return the AuthenticationType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AuthenticationType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CvssV2Data.AuthenticationType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            AuthenticationType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this authentication type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 CIA Requirement metric values for environmental scoring.
     */
    public enum CiaRequirementType {
        /** Low requirement. */
        LOW("LOW"),
        /** Medium requirement. */
        MEDIUM("MEDIUM"),
        /** High requirement. */
        HIGH("HIGH"),
        /** Not defined requirement. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, CiaRequirementType> CONSTANTS = new HashMap<>();

        static {
            for (CiaRequirementType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaRequirementType(String value) {
            this.value = value;
        }

        /**
         * Creates a CiaRequirementType from its string value.
         *
         * @param value the string value
         * @return the CiaRequirementType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static CiaRequirementType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (CvssV2Data.CiaRequirementType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            CiaRequirementType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this CIA requirement type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Confidentiality, Integrity, and Availability impact values.
     */
    public enum CiaType {
        /** No impact. */
        NONE("NONE"),
        /** Partial impact. */
        PARTIAL("PARTIAL"),
        /** Complete impact. */
        COMPLETE("COMPLETE");

        private static final Map<String, CiaType> CONSTANTS = new HashMap<>();

        static {
            for (CiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaType(String value) {
            this.value = value;
        }

        /**
         * Creates a CiaType from its string value.
         *
         * @param value the string value
         * @return the CiaType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static CiaType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CvssV2Data.CiaType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            CiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this CIA type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Collateral Damage Potential metric values for environmental scoring.
     */
    public enum CollateralDamagePotentialType {
        /** No collateral damage potential. */
        NONE("NONE"),
        /** Low collateral damage potential. */
        LOW("LOW"),
        /** Low-medium collateral damage potential. */
        LOW_MEDIUM("LOW_MEDIUM"),
        /** Medium-high collateral damage potential. */
        MEDIUM_HIGH("MEDIUM_HIGH"),
        /** High collateral damage potential. */
        HIGH("HIGH"),
        /** Not defined collateral damage potential. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, CollateralDamagePotentialType> CONSTANTS = new HashMap<>();

        static {
            for (CollateralDamagePotentialType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CollateralDamagePotentialType(String value) {
            this.value = value;
        }

        /**
         * Creates a CollateralDamagePotentialType from its string value.
         *
         * @param value the string value
         * @return the CollateralDamagePotentialType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static CollateralDamagePotentialType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CvssV2Data.CollateralDamagePotentialType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            CollateralDamagePotentialType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this collateral damage potential type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Exploitability metric values for temporal scoring.
     */
    public enum ExploitabilityType {
        /** Unproven exploit exists. */
        UNPROVEN("UNPROVEN"),
        /** Proof-of-concept exploit exists. */
        PROOF_OF_CONCEPT("PROOF_OF_CONCEPT"),
        /** Functional exploit exists. */
        FUNCTIONAL("FUNCTIONAL"),
        /** High exploitability. */
        HIGH("HIGH"),
        /** Not defined exploitability. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ExploitabilityType> CONSTANTS = new HashMap<>();

        static {
            for (ExploitabilityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ExploitabilityType(String value) {
            this.value = value;
        }

        /**
         * Creates an ExploitabilityType from its string value.
         *
         * @param value the string value
         * @return the ExploitabilityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ExploitabilityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CvssV2Data.ExploitabilityType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ExploitabilityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this exploitability type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Remediation Level metric values for temporal scoring.
     */
    public enum RemediationLevelType {
        /** Official fix is available. */
        OFFICIAL_FIX("OFFICIAL_FIX"),
        /** Temporary fix is available. */
        TEMPORARY_FIX("TEMPORARY_FIX"),
        /** Workaround is available. */
        WORKAROUND("WORKAROUND"),
        /** Remediation is unavailable. */
        UNAVAILABLE("UNAVAILABLE"),
        /** Not defined remediation level. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, RemediationLevelType> CONSTANTS = new HashMap<>();

        static {
            for (RemediationLevelType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        RemediationLevelType(String value) {
            this.value = value;
        }

        /**
         * Creates a RemediationLevelType from its string value.
         *
         * @param value the string value
         * @return the RemediationLevelType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static RemediationLevelType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (CvssV2Data.RemediationLevelType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            RemediationLevelType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this remediation level type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Report Confidence metric values for temporal scoring.
     */
    public enum ReportConfidenceType {
        /** Unconfirmed report. */
        UNCONFIRMED("UNCONFIRMED"),
        /** Uncorroborated report. */
        UNCORROBORATED("UNCORROBORATED"),
        /** Confirmed report. */
        CONFIRMED("CONFIRMED"),
        /** Not defined report confidence. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ReportConfidenceType> CONSTANTS = new HashMap<>();

        static {
            for (ReportConfidenceType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ReportConfidenceType(String value) {
            this.value = value;
        }

        /**
         * Creates a ReportConfidenceType from its string value.
         *
         * @param value the string value
         * @return the ReportConfidenceType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ReportConfidenceType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CvssV2Data.ReportConfidenceType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            ReportConfidenceType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this report confidence type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v2 Target Distribution metric values for environmental scoring.
     */
    public enum TargetDistributionType {
        /** No target distribution. */
        NONE("NONE"),
        /** Low target distribution. */
        LOW("LOW"),
        /** Medium target distribution. */
        MEDIUM("MEDIUM"),
        /** High target distribution. */
        HIGH("HIGH"),
        /** Not defined target distribution. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, TargetDistributionType> CONSTANTS = new HashMap<>();

        static {
            for (TargetDistributionType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        TargetDistributionType(String value) {
            this.value = value;
        }

        /**
         * Creates a TargetDistributionType from its string value.
         *
         * @param value the string value
         * @return the TargetDistributionType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static TargetDistributionType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (CvssV2Data.TargetDistributionType t : values()) {
                    if (t.value.startsWith(value)) {
                        return t;
                    }
                }
            }
            TargetDistributionType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this target distribution type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS Version
     */
    public enum Version {
        /** CVSS version 2.0. */
        _2_0("2.0");

        private static final Map<String, Version> CONSTANTS = new HashMap<>();

        static {
            for (Version c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        Version(String value) {
            this.value = value;
        }

        /**
         * Creates a Version from its string value.
         *
         * @param value the string value
         * @return the Version enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static Version fromValue(String value) {
            Version constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this version.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

}
