/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.util.Objects;

/**
 * CPE match string or range
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"vulnerable", "criteria", "versionStartExcluding", "versionStartIncluding", "versionEndExcluding",
        "versionEndIncluding", "matchCriteriaId"})
@NullMarked
public class CpeMatch implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = -7954886227420487016L;
    /**
     * (Required)
     */
    @JsonProperty("vulnerable")
    private Boolean vulnerable;
    /**
     * (Required)
     */
    @JsonProperty("criteria")
    private String criteria;
    /**
     * (Required)
     */
    @JsonProperty("matchCriteriaId")
    private String matchCriteriaId;
    @JsonProperty("versionStartExcluding")
    private @Nullable String versionStartExcluding;
    @JsonProperty("versionStartIncluding")
    private @Nullable String versionStartIncluding;
    @JsonProperty("versionEndExcluding")
    private @Nullable String versionEndExcluding;
    @JsonProperty("versionEndIncluding")
    private @Nullable String versionEndIncluding;

    /**
     * Constructor taking only the required set of JSON properties as the minimal constructor for a CpeMatch.
     *
     * @param vulnerable The vulnerable for the CpeMatch (required JSON property)
     * @param criteria The criteria for the CpeMatch (required JSON property)
     * @param matchCriteriaId The matchCriteriaId for the CpeMatch (required JSON property)
     */
    @JsonCreator
    public CpeMatch(@JsonProperty("vulnerable") Boolean vulnerable, @JsonProperty("criteria") String criteria,
            @JsonProperty("matchCriteriaId") String matchCriteriaId) {
        this.vulnerable = vulnerable;
        this.criteria = criteria;
        this.matchCriteriaId = matchCriteriaId;
    }

    /**
     * Full constructor for CPE match.
     *
     * @param vulnerable whether the match is for a vulnerable configuration
     * @param criteria the CPE 2.3 match string
     * @param matchCriteriaId the match criteria identifier
     * @param versionStartExcluding starting version excluded from the range
     * @param versionStartIncluding starting version included in the range
     * @param versionEndExcluding ending version excluded from the range
     * @param versionEndIncluding ending version included in the range
     */
    public CpeMatch(Boolean vulnerable, String criteria, String matchCriteriaId, @Nullable String versionStartExcluding,
            @Nullable String versionStartIncluding, @Nullable String versionEndExcluding,
            @Nullable String versionEndIncluding) {
        this(vulnerable, criteria, matchCriteriaId);
        this.versionStartExcluding = versionStartExcluding;
        this.versionStartIncluding = versionStartIncluding;
        this.versionEndExcluding = versionEndExcluding;
        this.versionEndIncluding = versionEndIncluding;
    }

    /**
     * (Required)
     *
     * @return vulnerable
     */
    @JsonProperty("vulnerable")
    public Boolean getVulnerable() {
        return vulnerable;
    }

    /**
     * (Required)
     *
     * @return criteria
     */
    @JsonProperty("criteria")
    public String getCriteria() {
        return criteria;
    }

    /**
     * (Required)
     *
     * @return matchCriteriaId
     */
    @JsonProperty("matchCriteriaId")
    public String getMatchCriteriaId() {
        return matchCriteriaId;
    }

    /**
     * @return versionStartExcluding
     */
    @JsonProperty("versionStartExcluding")
    public @Nullable String getVersionStartExcluding() {
        return versionStartExcluding;
    }

    /**
     * @return versionStartIncluding
     */
    @JsonProperty("versionStartIncluding")
    public @Nullable String getVersionStartIncluding() {
        return versionStartIncluding;
    }

    /**
     * @return versionStartIncluding
     */
    @JsonProperty("versionEndExcluding")
    public @Nullable String getVersionEndExcluding() {
        return versionEndExcluding;
    }

    /**
     * @return versionEndIncluding
     */
    @JsonProperty("versionEndIncluding")
    public @Nullable String getVersionEndIncluding() {
        return versionEndIncluding;
    }

    @Override
    public String toString() {
        return "CpeMatch{" + "vulnerable=" + vulnerable + ", criteria='" + criteria + '\'' + ", matchCriteriaId="
                + matchCriteriaId + ", versionStartExcluding='" + versionStartExcluding + '\''
                + ", versionStartIncluding='" + versionStartIncluding + '\'' + ", versionEndExcluding='"
                + versionEndExcluding + '\'' + ", versionEndIncluding='" + versionEndIncluding + '\'' + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        CpeMatch cpeMatch = (CpeMatch) o;
        return Objects.equals(vulnerable, cpeMatch.vulnerable) && Objects.equals(criteria, cpeMatch.criteria)
                && Objects.equals(matchCriteriaId, cpeMatch.matchCriteriaId)
                && Objects.equals(versionStartExcluding, cpeMatch.versionStartExcluding)
                && Objects.equals(versionStartIncluding, cpeMatch.versionStartIncluding)
                && Objects.equals(versionEndExcluding, cpeMatch.versionEndExcluding)
                && Objects.equals(versionEndIncluding, cpeMatch.versionEndIncluding);
    }

    @Override
    public int hashCode() {
        return Objects.hash(vulnerable, criteria, matchCriteriaId, versionStartExcluding, versionStartIncluding,
                versionEndExcluding, versionEndIncluding);
    }
}
