/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.epss;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;

import java.io.Serializable;

/**
 * Exploit Prediction Scoring System (EPSS) score.
 *
 * @see <a href="https://www.first.org/epss/">https://www.first.org/epss/</a>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"cve", "epss", "percentile"})
public class EpssItem implements Serializable {
    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 5043194930534860395L;
    /**
     * The CVE identifier.
     */
    @JsonProperty("cve")
    String cve;
    /**
     * The EPSS score.
     */
    @JsonProperty("epss")
    Double epss;
    /**
     * The percentile ranking.
     */
    @JsonProperty("percentile")
    Double percentile;

    /**
     * Constructs a new EpssItem.
     *
     * @param cve the CVE identifier
     * @param epss the EPSS score
     * @param percentile the percentile ranking
     */
    public EpssItem(String cve, Double epss, Double percentile) {
        this.cve = cve;
        this.epss = epss;
        this.percentile = percentile;
    }

    /**
     * Returns the CVE identifier.
     *
     * @return the CVE identifier
     */
    public String getCve() {
        return cve;
    }

    /**
     * Returns the EPSS score.
     *
     * @return the EPSS score
     */
    public Double getEpss() {
        return epss;
    }

    /**
     * Returns the percentile ranking.
     *
     * @return the percentile ranking
     */
    public Double getPercentile() {
        return percentile;
    }
}
