/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.apache.hc.client5.http.cache.HttpCacheCASOperation;
import org.apache.hc.client5.http.cache.HttpCacheEntry;
import org.apache.hc.client5.http.cache.HttpCacheStorage;
import org.apache.hc.client5.http.cache.HttpCacheStorageEntry;
import org.apache.hc.client5.http.cache.ResourceIOException;
import org.apache.hc.client5.http.impl.cache.HttpByteArrayCacheEntrySerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * Forced persistent cache storage. The implementation will cache the response to disk and consider it valid for a
 * configurable duration, the default is 20 hours.
 */
public class ForcedDiskCacheStorage implements HttpCacheStorage {
    private final Path storageDirectory;
    private final Duration timeToLive;
    private final HttpByteArrayCacheEntrySerializer serializer;

    /**
     * Default time-to-live duration for cache entries (20 hours).
     */
    public static final Duration DEFAULT_TTL = Duration.ofHours(20);
    private final static Logger LOG = LoggerFactory.getLogger(ForcedDiskCacheStorage.class);

    /**
     * Constructs a new ForcedDiskCacheStorage with the default time-to-live.
     *
     * @param storagePath the directory path where cache entries will be stored
     * @throws IOException if the storage directory cannot be created
     */
    @SuppressFBWarnings(value = "CT_CONSTRUCTOR_THROW", justification = "Constructor may legitimately throw exception")
    public ForcedDiskCacheStorage(String storagePath) throws IOException {
        this(storagePath, DEFAULT_TTL);
    }

    /**
     * Constructs a new ForcedDiskCacheStorage with a custom time-to-live.
     *
     * @param storagePath the directory path where cache entries will be stored
     * @param timeToLive the duration for which cache entries are considered valid
     * @throws IOException if the storage directory cannot be created
     */
    @SuppressFBWarnings(value = "CT_CONSTRUCTOR_THROW", justification = "Constructor may legitimately throw exception")
    public ForcedDiskCacheStorage(String storagePath, Duration timeToLive) throws IOException {
        this.storageDirectory = Paths.get(storagePath);
        this.timeToLive = timeToLive;
        this.serializer = new HttpByteArrayCacheEntrySerializer();
        Files.createDirectories(storageDirectory);
    }

    /**
     * Stores a cache entry to disk.
     *
     * @param key the cache key
     * @param entry the cache entry to store
     * @throws ResourceIOException if writing the cache entry fails
     */
    @Override
    public void putEntry(String key, HttpCacheEntry entry) throws ResourceIOException {
        Path filePath = storageDirectory.resolve(sanitizeKey(key));
        HttpCacheStorageEntry storageEntry = new HttpCacheStorageEntry(key, entry);
        byte[] serialized = serializer.serialize(storageEntry);
        try {
            Files.write(filePath, serialized);
        } catch (IOException e) {
            throw new ResourceIOException("Failed to write cache entry", e);
        }
    }

    /**
     * Retrieves a cache entry from disk if it exists and is still fresh.
     *
     * @param key the cache key
     * @return the cached entry if found and fresh, otherwise null
     * @throws ResourceIOException if reading the cache entry fails
     */
    @Override
    public HttpCacheEntry getEntry(String key) throws ResourceIOException {
        Path filePath = storageDirectory.resolve(sanitizeKey(key));
        if (!Files.exists(filePath)) {
            LOG.debug("Cache miss `{}`", filePath.getFileName());
            return null;
        }

        try {
            byte[] data = Files.readAllBytes(filePath);
            HttpCacheStorageEntry entry = serializer.deserialize(data);
            if (isFresh(entry.getContent())) {
                LOG.debug("Cache hit `{}`", filePath.getFileName());
                return entry.getContent();
            }
            LOG.debug("Cache entry stale `{}`", filePath.getFileName());
            removeEntry(key);
            return null;
        } catch (IOException e) {
            LOG.debug("Failed to read cache entry: " + e.getMessage());
            return null;
        }
    }

    /**
     * Removes a cache entry from disk.
     *
     * @param key the cache key
     * @throws ResourceIOException if deleting the cache entry fails
     */
    @Override
    public void removeEntry(String key) throws ResourceIOException {
        Path filePath = storageDirectory.resolve(sanitizeKey(key));
        try {
            Files.deleteIfExists(filePath);
        } catch (IOException e) {
            throw new ResourceIOException("Failed to delete cache entry", e);
        }
    }

    /**
     * Updates a cache entry using a compare-and-swap operation.
     *
     * @param key the cache key
     * @param casOperation the compare-and-swap operation to execute
     * @throws ResourceIOException if updating the cache entry fails
     */
    @Override
    public void updateEntry(String key, HttpCacheCASOperation casOperation) throws ResourceIOException {
        HttpCacheEntry existing = getEntry(key);
        HttpCacheEntry updated = casOperation.execute(existing);
        if (updated != null) {
            putEntry(key, updated);
        } else {
            removeEntry(key);
        }
    }

    /**
     * Retrieves multiple cache entries from disk.
     *
     * @param keys the collection of cache keys
     * @return a map of cache keys to entries for all found and fresh entries
     * @throws ResourceIOException if reading cache entries fails
     */
    @Override
    public Map<String, HttpCacheEntry> getEntries(Collection<String> keys) throws ResourceIOException {
        Map<String, HttpCacheEntry> entries = new HashMap<>();
        for (String key : keys) {
            HttpCacheEntry entry = getEntry(key);
            if (entry != null) {
                entries.put(key, entry);
            }
        }
        return entries;
    }

    /**
     * Checks if a cache entry is still fresh based on its age and the configured time-to-live.
     *
     * @param entry the cache entry to check
     * @return true if the entry is fresh, false otherwise
     */
    private boolean isFresh(HttpCacheEntry entry) {
        return entry.getRequestInstant().plus(timeToLive).isAfter(Instant.now());
    }

    /**
     * Sanitizes a cache key by replacing non-alphanumeric characters with underscores.
     *
     * @param key the cache key to sanitize
     * @return the sanitized cache key
     */
    private String sanitizeKey(String key) {
        return key.replaceAll("[^a-zA-Z0-9._-]", "_");
    }
}
