/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2022-2024 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import io.github.jeremylong.openvulnerability.client.HttpAsyncClientSupplier;
import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.client5.http.impl.async.HttpAsyncClients;
import org.apache.hc.client5.http.impl.routing.SystemDefaultRoutePlanner;
import org.apache.hc.core5.concurrent.FutureCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.ProxySelector;
import java.time.LocalTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * Rate limited client for use with a rate limited API. The client contains two mechanisms to control the rate:
 * <ol>
 * <li>RateMeter - limits the number of calls over a given time period</li>
 * <li>Delay - the minimum delay between calls</li>
 * </ol>
 * <p>
 * The two mechanisms may appear redundant - but each have its purpose in making the calls as fast as can be done while
 * being kind to the endpoint. If one is calling an endpoint, such as the NVD vulnerability API which is limited to 5
 * calls in 30 seconds without an API Key, to retrieve 4 page of data as quickly as possible you could set a smaller
 * delay and still keep the Rate Meter to limit to 5 calls per 30 seconds. However, if you are retrieving a large number
 * of pages you would want the delay to be slightly under the time period divided by the allowed number of calls (e.g.,
 * if we allowed 5 calls over 30 seconds we would use 30/5=6 seconds).
 */
class RateLimitedClient implements AutoCloseable {

    /**
     * Reference to the logger.
     */
    private final static Logger LOG = LoggerFactory.getLogger(SimpleFutureResponse.class);
    /**
     * The underlying Async Client.
     */
    private final CloseableHttpAsyncClient client;
    /**
     * Executor service for async implementation.
     */
    private final ExecutorService executor = Executors.newSingleThreadExecutor();
    /**
     * The epoch time of the last request.
     */
    private long lastRequest = 0;
    /**
     * The minimum delay in milliseconds between API calls.
     */
    private long delay;
    /**
     * Rate limiting meter.
     */
    private final RateMeter meter;

    /**
     * Construct a rate limited client without a delay or limiters.
     */
    RateLimitedClient() {
        this(0, 0, 0);
    }

    /**
     * Construct a rate limited client with a given delay and request window configuration. This allows callers to
     * configure 5 requests are allowed over a 30-second rolling window and we will delay at least 4 seconds between
     * calls to help more evenly distribute the calls across the request window.
     *
     * @param minimumDelay the number of milliseconds to wait between API calls
     * @param requestsCount the number of requests allowed during the rolling request window timespan
     * @param requestWindowMilliseconds the rolling request window size in milliseconds
     */
    RateLimitedClient(long minimumDelay, int requestsCount, long requestWindowMilliseconds) {
        this(minimumDelay,
                (requestsCount > 0 && requestWindowMilliseconds > 0)
                        ? new RateMeter(requestsCount, requestWindowMilliseconds)
                        : new RateMeter(100, 5));
    }

    /**
     * Construct a rate limited client with a given delay and request window configuration. This allows callers to
     * configure 5 requests are allowed over a 30-second rolling window and we will delay at least 4 seconds between
     * calls to help more evenly distribute the calls across the request window.
     *
     * @param minimumDelay the number of milliseconds to wait between API calls
     * @param meter the rate meter to limit the request rate
     */
    RateLimitedClient(long minimumDelay, RateMeter meter) {
        this(10, minimumDelay, meter, null);
    }

    /**
     * Construct a rate limited client with a given delay and request window configuration. This allows callers to
     * configure 5 requests are allowed over a 30-second rolling window and we will delay at least 4 seconds between
     * calls to help more evenly distribute the calls across the request window.
     *
     * @param maxRetries the maximum number of retry attemps
     * @param minimumDelay the number of milliseconds to wait between API calls
     * @param meter the rate meter to limit the request rate
     * @param httpClientSupplier supplier for custom HTTP clients; if {@code null} a default client will be used
     */
    RateLimitedClient(int maxRetries, long minimumDelay, RateMeter meter, HttpAsyncClientSupplier httpClientSupplier) {
        this.meter = meter;
        this.delay = minimumDelay;
        LOG.debug("rate limited call delay: {}", delay);
        if (httpClientSupplier == null) {
            NvdApiRetryStrategy retryStrategy = new NvdApiRetryStrategy(maxRetries, minimumDelay);
            SystemDefaultRoutePlanner planner = new SystemDefaultRoutePlanner(ProxySelector.getDefault());
            client = HttpAsyncClients.custom().setRoutePlanner(planner).setRetryStrategy(retryStrategy)
                    .useSystemProperties().build();
        } else {
            client = httpClientSupplier.get();
        }
        client.start();
    }

    @Override
    public void close() throws Exception {
        if (client != null) {
            client.close();
        }
    }

    /**
     * Returns the delay between API calls in milliseconds.
     *
     * @return the delay between API calls in milliseconds
     */
    public long getDelay() {
        return delay;
    }

    /**
     * Sets the minimum delay between API calls.
     *
     * @param milliseconds the delay duration
     */
    void setDelay(long milliseconds) {
        this.delay = milliseconds;
        LOG.debug("rate limited call delay: {}", delay);
    }

    /**
     * Ensures the minimum delay has passed since the last call and asynchronously calls the API.
     *
     * @param request the request
     * @return the future response
     */
    Future<RateLimitedCall> execute(SimpleHttpRequest request, int clientIndex, int startIndex) {
        return executor.submit(() -> delayedExecute(request, clientIndex, startIndex));
    }

    /**
     * Ensures the minimum delay has passed since the last call and calls the API.
     *
     * @param request the request
     * @return the future response
     * @throws ExecutionException thrown if there is an exception
     * @throws InterruptedException thrown if interrupted
     */
    private RateLimitedCall delayedExecute(SimpleHttpRequest request, int clientIndex, int startIndex)
            throws ExecutionException, InterruptedException {
        if (lastRequest > 0 && delay > 0) {
            long now = ZonedDateTime.now().toInstant().toEpochMilli();
            long wait = delay - (now - lastRequest);
            if (wait > 0) {
                LOG.debug("Rate Limited API call - waiting for {}ms", wait);
                Thread.sleep(wait);
            }
        }
        try (RateMeter.Ticket t = meter.getTicket()) {
            if (LOG.isDebugEnabled()) {
                LocalTime time = LocalTime.now();
                DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
                LOG.debug("Requested At: {}; URI: {}", time.format(formatter), request.getRequestUri());
            }
            Future<SimpleHttpResponse> f = client.execute(request, new SimpleFutureResponse());
            lastRequest = ZonedDateTime.now().toInstant().toEpochMilli();
            return new RateLimitedCall(f.get(), clientIndex, startIndex);
        } catch (Exception e) {
            if (e instanceof InterruptedException) {
                throw (InterruptedException) e;
            } else if (e instanceof ExecutionException) {
                throw (ExecutionException) e;
            }
            throw new NvdApiException(e);
        }
    }

    /**
     * Future response.
     */
    static class SimpleFutureResponse implements FutureCallback<SimpleHttpResponse> {
        /**
         * Reference to the logger.
         */
        private final Logger log = LoggerFactory.getLogger(SimpleFutureResponse.class);

        @Override
        public void completed(SimpleHttpResponse result) {
        }

        @Override
        public void failed(Exception ex) {
            log.debug("request failed", ex);
        }

        @Override
        public void cancelled() {
            // do nothing
        }
    }
}
