package com.gongbo.export.advise;


import com.gongbo.export.config.ExportConfig;
import com.gongbo.export.config.ResultHandler;
import com.gongbo.export.core.ExportContextHolder;
import com.gongbo.export.core.ExportHandlers;
import com.gongbo.export.core.ExportHelper;
import com.gongbo.export.core.handler.AfterExportHandler;
import com.gongbo.export.core.handler.ExportDataConvert;
import com.gongbo.export.core.handler.WriteHandler;
import com.gongbo.export.entity.ExportContext;
import com.gongbo.export.entity.ExportInfo;
import com.gongbo.export.entity.ExportParam;
import com.gongbo.export.exception.NotSupportExportException;
import com.gongbo.export.utils.ExportUtils;
import com.gongbo.export.utils.WebUtils;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.List;
import java.util.Objects;

@Aspect
@Slf4j
public class ExportAdvise {

    @Autowired
    private ResultHandler resultHandler;

    @Autowired
    private ExportConfig exportConfig;

    @Pointcut("@annotation(com.gongbo.export.annotations.AutoExport) || @annotation(com.gongbo.export.annotations.AutoExports)")
    public void doExport() {
    }

    /**
     * @param joinPoint
     * @return
     * @throws Throwable
     */
    @Around("doExport()")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
        //从Http请求中获取ExportParam
        ExportParam exportParam = ExportParam.parse(Objects.requireNonNull(WebUtils.getRequest()));

        if (exportParam == null) {
            return joinPoint.proceed();
        }
        //获取HttpServletResponse对象
        HttpServletResponse response = Objects.requireNonNull(WebUtils.getResponse());

        //获取代理方法
        Method targetMethod = getTargetMethod(joinPoint);

        try {
            //执行导出
            return doExport(exportParam, joinPoint, targetMethod, response);
        } catch (Exception e) {
            log.error("导出错误", e);
            return buildErrorResponse(e);
        }
    }

    /**
     * 执行导出
     *
     * @param exportParam
     * @param joinPoint
     * @param targetMethod
     * @param response
     * @return
     * @throws Throwable
     */
    private Object doExport(ExportParam exportParam, ProceedingJoinPoint joinPoint, Method targetMethod, HttpServletResponse response) throws Throwable {
        //构建导出上下文
        ExportContext exportContext = ExportHelper.buildExportContext(exportParam, targetMethod, exportConfig, resultHandler);

        try {
            ExportContextHolder.setContext(exportContext);

            if (exportContext.getExportParam().isInfo()) {
                return buildExportInfoResponse(exportContext);
            } else if (exportContext.getExportParam().isExcel()) {
                //执行请求，获取请求返回值
                Object result = joinPoint.proceed();
                if (!(resultHandler.check(result))) {
                    throw new NotSupportExportException(MessageFormat.format("请求接口方法返回类型必须是{0}类型", exportConfig.getResponseClassName()));
                } else {
                    return responseExcel(exportContext, result, response);
                }
            } else {
                throw new IllegalArgumentException();
            }
        } finally {
            ExportContextHolder.clear();
        }
    }

    /**
     * 生成ExportInfo响应信息
     *
     * @param exportContext
     * @return
     */
    private Object buildExportInfoResponse(ExportContext exportContext) {
        ExportInfo exportInfo = ExportInfo.buildExportInfo(exportContext);

        return resultHandler.success(exportInfo);
    }

    /**
     * 生成错误响应信息
     *
     * @param e
     * @return
     */
    private Object buildErrorResponse(Exception e) {
        return resultHandler.error(e);
    }

    /**
     * 响应文件
     *
     * @param exportContext
     * @param result
     * @param response
     * @return
     * @throws IOException
     */
    private static Object responseExcel(ExportContext exportContext, Object result, HttpServletResponse response) throws IOException {
        //数据转换
        ExportDataConvert exportDataConvert = ExportHandlers.of(exportContext.getAutoExport().dataConvert());
        List<?> data = exportDataConvert.convert(exportContext, result);

        //执行导出
        WriteHandler writeHandler = ExportHandlers.of(exportContext.getAutoExport().writeHandler());
        //导出
        writeHandler.write(exportContext, data, ExportUtils.getExportOutputStream(exportContext, response));

        AfterExportHandler afterExportHandler = ExportHandlers.of(exportContext.getAutoExport().afterExportHandler());
        //导出文件后执行
        afterExportHandler.afterExport(exportContext);

        return exportContext.isOutputFile() ? result : null;
    }

    /**
     * 获取当前请求执行方法
     *
     * @param proceedingJoinPoint
     * @return
     */
    private static Method getTargetMethod(ProceedingJoinPoint proceedingJoinPoint) {
        return ((MethodSignature) proceedingJoinPoint.getSignature()).getMethod();
    }

}
