package com.iflytek.aiui.player.core

import android.content.Context
import android.media.AudioManager
import android.os.Handler
import android.os.SystemClock
import com.iflytek.aiui.player.common.request.ErrorCallback
import com.iflytek.aiui.player.common.request.SuccessRetCallback
import com.iflytek.aiui.player.common.data.MetaItem
import com.iflytek.aiui.player.common.data.Config
import com.iflytek.aiui.player.common.default.DefaultMediaPlayer
import com.iflytek.aiui.player.common.logger.HttpLogger
import com.iflytek.aiui.player.common.logger.Logger
import com.iflytek.aiui.player.common.player.*
import com.iflytek.aiui.player.common.rpc.*
import com.iflytek.aiui.player.common.rpc.connection.DataConnection
import com.iflytek.aiui.player.common.rpc.connection.impl.WebSocketServerConnection
import com.iflytek.aiui.player.common.request.Request
import com.iflytek.aiui.player.common.storage.Storage
import com.iflytek.aiui.player.common.utils.UniqueIDUtil
import com.iflytek.aiui.player.common.utils.replaceBlank
import org.json.JSONArray
import org.json.JSONObject
import org.json.JSONTokener
import java.util.concurrent.CountDownLatch
import java.util.concurrent.TimeUnit
import kotlin.concurrent.thread
import kotlin.reflect.full.primaryConstructor

/**
 * 播放器状态
 */
enum class PlayState {
    /**
     * 未初始化状态，需要调用initialize
     */
    IDLE,

    /**
     * 正在初始化
     */
    INITIALIZING,

    /**
     * 就绪状态
     */
    READY,

    /**
     * 资源装在加载
     */
    LOADING,

    /**
     * 正在播放状态
     */
    PLAYING,

    /**
     * 暂停状态
     */
    PAUSED,

    /**
     * 列表播放完成状态，可调用resume或play，重新进入PLAYING
     */
    COMPLETE,

    /**
     * 错误状态，需调用reset，重新进入READY状态
     */
    ERROR;
}


/**
 * 播放状态监听器
 */
interface PlayerListener {
    /**
     * 播放器已就绪，调用AIUIPlayer的initialize后异步回调此接口通知已就绪
     */
    fun onPlayerReady()

    /**
     * 播放器状态回调，在播放状态变化时回调
     *
     * @param state 播放状态
     */
    fun onStateChange(state: PlayState)

    /**
     * 播放内容变化回调，直接调用next，previous或播放结束自动切换到下一首
     *
     * @param item 当前播放项
     */
    fun onMediaChange(item: MetaItem)

    /**
     * 错误回调
     *
     * @param error 错误码
     * @param info 错误描述信息
     */
    fun onError(error: Int, info: String)

    /**
     * 播放器已销毁
     */
    fun onPlayerRelease()
}


/**
 * AIUI播放器
 *
 * 用来解析播放AIUI平台返回可播放的信源内容列表，
 * 内部根据每项内容的source字段分发到具体的播放器
 *
 * @param context Android Context
 *
 * @constructor 创建AIUIPlayer
 */

class AIUIPlayer(private val context: Context, private val config: String) {
    private var mServerConnection: DataConnection = WebSocketServerConnection(4096)
    private var mRPCServer = RPC(mServerConnection, object : RPCListener {
        override fun onRequest(rpc: RPC, data: String) {
            val request = Request.decodeRequest(JSONObject(data))
            request?.let {
                for (subPlayer in mPlayers) {
                    val canHandle = subPlayer.onRequest(request,
                            object : SuccessRetCallback<Any> {
                                override fun invoke(p1: Any) = rpc.response(request, p1)
                            },
                            object : ErrorCallback {
                                override fun invoke(p1: Int, p2: String) = rpc.response(request, p1, p2)
                            })

                    if (canHandle) break
                }
            }
        }
    })
    var appId: String = ""
    var appKey: String = ""

    init {
        val configStrs = config.split(",")
        configStrs.forEach {
            val param = it.split("=")
            if (param.size == 2) {
                var key = replaceBlank(param[0])
                var value = replaceBlank(param[1])
                when (key) {
                    "appId" -> appId = value
                    "appKey" -> appKey = value
                }
            }
        }

    }

    private var mMainHandler = Handler(context.mainLooper)
    private var mStorage = Storage(context)
    private var mPlayers: MutableList<MetaAbstractPlayer> = mutableListOf()
    private var mActivePlayer: MetaAbstractPlayer? = null
    private val mListeners = mutableListOf<PlayerListener>()

    //默认初始化状态
    private var mState: PlayState = PlayState.IDLE

    //播放列表内容
    private var mData = mutableListOf<MetaItem>()
    private var mIndex = 0
    private var positiveDirection = true

    //初始化标识，避免重复初始化
    private var mInitialized = false

    //Ready标记保存，Listener的注册可能在Ready之后，保证状态回调正常
    private var mReadyCount = 0
    private var mAutoSkipError = true
    private var mLogger: Logger? = null
    private val mAudioFocus = AudioFocus(context, AudioManager.OnAudioFocusChangeListener { focusChange ->
        when (focusChange) {
            AudioManager.AUDIOFOCUS_LOSS_TRANSIENT, AudioManager.AUDIOFOCUS_LOSS -> {
                pause()
            }
        }
    })


    /** 当前播放项内容
     */
    val currentPlay: MetaItem?
        get() {
            return if (mIndex !in 0 until mData.size) {
                null
            } else {
                mData[mIndex]
            }
        }

    /** 当前播放状态
     */
    val currentState
        get() = mState

    /** 当前播放项播放长度
     */
    val duration: Long
        get() = mActivePlayer?.getDuration() ?: 0


    /**
     * 当前播放项播放进度
     */
    fun getCurrentPosition(): Long {
        return mActivePlayer?.getCurrentPos() ?: 0
    }

    /**
     * 初始化，在构造完成后立即调用
     *
     * （不放在构造函数中，是为测试用例分离需要)
     */
    fun initialize() {
        // 避免重复初始化
        if (mInitialized) return

        mLogger = HttpLogger(appId, appKey, UniqueIDUtil.getMscUniqueId(context))
        mPlayers = mutableListOf(MetaMediaPlayer(context, config, mRPCServer, mStorage, mLogger!!))
        //反射初始化可能配置的子播放器
        val prefix = "com.iflytek.aiui.player.players"
        val players = listOf("$prefix.KuwoLSMusicPlayer", "$prefix.MetaKGPlayer", "$prefix.MetaQTPlayer", "$prefix.MetaMiGuPlayer", "$prefix.MetaKWPlayer")
        players.forEach {
            try {
                val playerClass = Class.forName(it).kotlin
                val player = playerClass.primaryConstructor?.call(context, config, mRPCServer, mStorage, mLogger)
                player?.let { mPlayers.add(player as MetaAbstractPlayer) }
            } catch (exception: ClassNotFoundException) {
                //player dependency not configure just ignore
            }
        }
        mPlayers.add(DefaultMediaPlayer(context, config, mRPCServer, mStorage, mLogger!!))
        mServerConnection.start()
        onStateChange(PlayState.INITIALIZING)
        //初始化各子播放器
        mPlayers.forEach {
            it.initialize()
            it.setDebug(isDebug)
            it.addListener(object : MetaListener {
                override fun onError(error: Int, description: String) {
                    if (mAutoSkipError) {
                        if (positiveDirection) {
                            if (!next()) {
                                onStateError(error, description)
                            }
                        } else {
                            if (!previous()) {
                                onStateError(error, description)
                            }
                        }
                    } else {
                        onStateError(error, description)
                    }
                }

                override fun onReady() {
                    if (++mReadyCount == mPlayers.size) {
                        onStateChange(PlayState.READY)
                        runMain {
                            mListeners.forEach { listener -> listener.onPlayerReady() }
                        }
                    }
                }

                override fun onStateChange(state: MetaState) {
                    when (state) {
                        MetaState.LOADING -> {
                            onStateChange(PlayState.LOADING)
                        }

                        MetaState.PLAYING -> {
                            if (mState == PlayState.LOADING) {
                                onStateChange(PlayState.PLAYING)
                            }

                            if (mAudioFocus.requestAudioFocusIfNeed() != AudioManager.AUDIOFOCUS_REQUEST_GRANTED) {
                                pause()
                            }
                        }


                        MetaState.COMPLETE -> {
                            if (!next()) {
                                onComplete()
                                mAudioFocus.abandonAudioFocus()
                            }
                        }

                        else -> {
                        }

                    }
                }

                override fun onRelease() {
                    if (--mReadyCount == 0) {
                        runMain {
                            mListeners.forEach { listener -> listener.onPlayerRelease() }
                        }
                        onStateChange(PlayState.IDLE)
                    }
                }
            })
        }

        mInitialized = true
    }

    private var isDebug = false
    fun setDebug(isDebug: Boolean) {
        this.isDebug = isDebug
        mPlayers.forEach {
            it.setDebug(isDebug)
        }
    }

    /**
     * 信源列表是否用可播放项
     * 需要在PlayerReady后调用
     * @param data 信源内容列表
     * @param service 技能名称
     *
     * @return 列表是否有可播放项
     *
     */
    fun anyAvailablePlay(data: JSONArray, service: String, sid: String = ""): Boolean {
        for (i in 0 until data.length()) {
            if (mPlayers.any {
                        it.canDispose(MetaItem(data.optJSONObject(i), service, sid))
                    }) {
                return true
            }
        }

        return false
    }

    /**
     * 播放信源内容列表
     * 需要在PlayerReady后调用
     * @param data 信源内容列表
     * @param service 技能名称
     * @param sid AIUI SDK 语音交互得到结果中的sid字段，此字段可传空。
     * @param autoSkipError 是否跳过错误播放
     * @return 列表是否播放成功
     *
     */
    fun play(data: JSONArray, service: String, sid: String = "", autoSkipError: Boolean = true): Boolean {
        if (mState != PlayState.READY || !anyAvailablePlay(data, service)) return false

        mActivePlayer?.pause()
        mData = mutableListOf()
        for (i in 0 until data.length()) {
            mData.add(MetaItem(data.optJSONObject(i), service, sid))
        }
        mIndex = -1
        mAutoSkipError = autoSkipError
        return playToNextAvailable()
    }

    /**
     * 下一个
     *
     * @return 是否操作成功
     */
    fun next(): Boolean {
        positiveDirection = true
        return playToNextAvailable()
    }

    /**
     * 上一个
     *
     * @return 是否操作成功
     */
    fun previous(): Boolean {
        positiveDirection = false
        return playToNextAvailable(false)
    }

    /**
     * 暂停播放。暂停之后还可以继续播放
     */
    fun pause() {
        when (mState) {
            PlayState.PLAYING, PlayState.LOADING -> {
                onStateChange(PlayState.PAUSED)
                mActivePlayer?.pause()
            }
            else -> {
            }
        }
    }

    /**
     * 停止播放。停止之后，列表清空，状态恢复到READY
     */
    fun stop() {
        reset()
    }

    /**
     * 继续播放
     */
    fun resume() {
        when (mState) {
            PlayState.PAUSED -> {
                onStateChange(PlayState.PLAYING)
                mActivePlayer?.resume()
            }

            PlayState.COMPLETE -> {
                mIndex++
                playToNextAvailable(false)
            }

            else -> {
            }
        }
    }


    /**
     *  播放进度进度调整
     *
     *  @param msec 目标播放进度
     */
    fun seekTo(msec: Long) {
        mActivePlayer?.seekTo(msec)
    }

    /**
     * 播放器停止，列表清空，状态恢复到STOPPED
     */
    fun reset() {
        onStateChange(PlayState.READY)
        mActivePlayer?.pause()

        mData.clear()
        mIndex = -1
    }

    /**
     * 添加播放监听器
     *
     * @param listener 播放监听器
     */
    fun addListener(listener: PlayerListener) {
        mListeners.add(listener)
        if (mState != PlayState.IDLE && mState != PlayState.INITIALIZING) {
            runMain {
                listener.onPlayerReady()
                listener.onStateChange(PlayState.READY)
            }
        }
    }

    /**
     * 监听器移除
     *
     * @param listener 播放监听器
     */
    fun removeListener(listener: PlayerListener) {
        mListeners.remove(listener)
    }

    /**
     * 播放器销毁
     *
     * END状态
     */
    fun release() {
        if (!mInitialized) return
        mInitialized = false
        reset()
        mPlayers.forEach { it.release() }
    }


    /**
     * 请求执行
     *
     * @param request 请求
     * @param success 请求执行成功回调
     * @param error 请求执行失败回调
     */
    fun request(request: Request, success: SuccessRetCallback<Any>, error: ErrorCallback) {
        for (subPlayer in mPlayers) {
            if (subPlayer.onRequest(request, success, error)) break
        }
    }


    private fun runMain(action: () -> Unit) {
        mMainHandler.post {
            action()
        }
    }

    private fun onComplete() {
        mIndex = mData.size - 1
        mActivePlayer?.pause()
        onStateChange(PlayState.COMPLETE)
    }


    private fun onStateChange(state: PlayState) {
        mState = state
        runMain {
            mListeners.forEach {
                it.onStateChange(state)
            }
        }
    }

    private fun onStateError(error: Int, description: String) {
        onStateChange(PlayState.ERROR)
        runMain {
            mListeners.forEach {
                it.onError(error, description)
            }
        }
    }

    private fun onItemChange(item: MetaItem) {
        runMain {
            mListeners.forEach {
                it.onMediaChange(item)
            }
        }
    }

    private fun playToNextAvailable(positive: Boolean = true): Boolean {
        var range: IntProgression = mIndex + 1 until mData.size
        if (!positive) {
            range = mIndex - 1 downTo 0
        }
        for (index in range) {
            if (index !in 0 until mData.size) continue
            //寻找能处理此item的player
            val availablePlayer = mPlayers.find {
                it.canDispose(mData[index])
            }

            if (availablePlayer != null) {

                mActivePlayer?.pause()
                mActivePlayer?.mActive = false

                mIndex = index
                mActivePlayer = availablePlayer
                mActivePlayer?.mActive = true
                onItemChange(mData[mIndex])
                mActivePlayer?.play(mData[index])
                return true
            }
        }

        return false
    }


    /**
     * 获取音频播放的url
     * @param jsonArrayData：数据。如果是AIUI-SDK返回数据，可以将返回结果中result直接传进来。如果非AIUI-SDK返回的数据，必须要保证JSONObject里面包含itemId。
     * @param service：技能。如果是AIUI-SDK返回的数据，直接将返回数据中service的值传进来即可。如果非AIUI-SDK返回的数据，这里可以传 musicPro
     */
    @Deprecated("由于音乐播放的url有时效限制，目前是1个小时过期，所以批量获取可播放的url会存在无法播放的情况。此方法后期不维护")
    fun getAudioUrls(jsonArrayData: JSONArray, service: String, success: SuccessRetCallback<JSONArray>, error: ErrorCallback) {
        if (mPlayers.isEmpty()) {
            mListeners.forEach {
                error.invoke(-1, "please init AIUI Player")
            }
            return
        }
        thread {
            val json = jsonArrayData.toString()
            val tempData = JSONArray(JSONTokener(json))
            val needGetUrls = hashMapOf<String, MetaItem>()
            for (i in 0 until tempData.length()) {
                val jobj = tempData.optJSONObject(i)
                jobj?.put("expiretime", "")
                jobj?.put("audiopath", "")
                jobj?.put("resultCode", -1)//resultCode获取url的结果。默认是失败，当成功了，将它置为0
                val metaItem = MetaItem(tempData.optJSONObject(i), service, "")
                needGetUrls[jobj.optString("itemid")] = metaItem
            }
            val countDownLatch = CountDownLatch(needGetUrls.size)
            needGetUrls.values.forEach { metaItem ->
                val availablePlayer = mPlayers.find {
                    it.canDispose(metaItem)
                }
                availablePlayer?.getAudioUrls(metaItem,
                        { jsonobj ->
                            countDownLatch.countDown()
                            var itemid = jsonobj.optString("itemid")
                            var expiretime = jsonobj.optString("expiretime")
                            var audiopath = jsonobj.optString("audiopath")

                            var jobj = needGetUrls[itemid]?.info
                            jobj?.put("expiretime", expiretime)
                            jobj?.put("audiopath", audiopath)
                            jobj?.put("resultCode", 0)
                        },
                        { code, msg ->
                            countDownLatch.countDown()
                        })
            }
            countDownLatch.await(5, TimeUnit.SECONDS)
            runMain {
                success.invoke(tempData)
            }

        }
    }
}

