package com.rajat.pdfviewer.compose

import android.graphics.Bitmap
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.rajat.pdfviewer.PdfRendererCore
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

@Composable
fun PdfDocumentViewer(
    renderer: PdfRendererCore,
    modifier: Modifier = Modifier,
    showPageNumber: Boolean = true,
) {
    val context = LocalContext.current
    val pageCount = remember { renderer.getPageCount() }
    val screenWidth = context.resources.displayMetrics.widthPixels
    val screenHeight = (screenWidth * 1.41f).toInt() // A4 aspect ratio
    val listState = rememberLazyListState()
    val scope = rememberCoroutineScope()

    // 🔄 Prefetch logic
    LaunchedEffect(listState.firstVisibleItemIndex) {
        renderer.prefetchPagesAround(
            currentPage = listState.firstVisibleItemIndex,
            width = screenWidth,
            height = screenHeight,
            direction = 1 // forward-only
        )
    }

    LazyColumn(modifier = modifier, state = listState) {
        itemsIndexed(List(pageCount) { it }) { _, page ->
            var bitmap by remember(page) { mutableStateOf<Bitmap?>(null) }

            LaunchedEffect(page) {
                if (bitmap == null) {
                    scope.launch(Dispatchers.IO) {
                        val result = renderer.renderPageAsync(page, screenWidth, screenHeight)
                        bitmap = result
                    }
                }
            }

            Box(contentAlignment = Alignment.TopEnd) {
                if (bitmap != null) {
                    Image(
                        bitmap = bitmap!!.asImageBitmap(),
                        contentDescription = "PDF Page $page",
                        modifier = Modifier.fillMaxWidth()
                    )
                } else {
                    Box(
                        modifier = Modifier.fillMaxWidth(),
                        contentAlignment = Alignment.Center
                    ) {
                        CircularProgressIndicator(modifier = Modifier.align(Alignment.Center))
                    }
                }

                if (showPageNumber) {
                    Text(
                        text = "${page + 1}",
                        fontSize = 12.sp,
                        color = MaterialTheme.colorScheme.onBackground,
                        modifier = Modifier.align(Alignment.TopEnd)
                    )
                }
            }
        }
    }
}
