/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.dsl.internal;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.fabric8.kubernetes.api.model.HasMetadata;
import io.fabric8.kubernetes.api.model.KubernetesResourceList;
import io.fabric8.kubernetes.api.model.Status;
import io.fabric8.kubernetes.api.model.WatchEvent;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.base.BaseOperation;
import io.fabric8.kubernetes.client.dsl.base.OperationSupport;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;
import okhttp3.ws.WebSocket;
import okhttp3.ws.WebSocketCall;
import okhttp3.ws.WebSocketListener;
import okio.Buffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WatchConnectionManager<T extends HasMetadata, L extends KubernetesResourceList<T>>
implements Watch {
    private static final Logger logger = LoggerFactory.getLogger(WatchConnectionManager.class);
    private static final ObjectMapper mapper = new ObjectMapper();
    private final AtomicBoolean forceClosed = new AtomicBoolean();
    private final AtomicReference<String> resourceVersion;
    private final BaseOperation<T, L, ?, ?> baseOperation;
    private final Watcher<T> watcher;
    private final int reconnectLimit;
    private final int reconnectInterval;
    private static final int maxIntervalExponent = 5;
    private final long websocketTimeout;
    private final long websocketPingInterval;
    private final AtomicInteger currentReconnectAttempt = new AtomicInteger(0);
    private final AtomicReference<WebSocket> webSocketRef = new AtomicReference();
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactory(){

        @Override
        public Thread newThread(Runnable r) {
            Thread ret = new Thread(r, "Executor for Watch " + System.identityHashCode(WatchConnectionManager.this));
            ret.setDaemon(true);
            return ret;
        }
    });
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final AtomicBoolean reconnectPending = new AtomicBoolean(false);
    private final ArrayBlockingQueue<Object> queue = new ArrayBlockingQueue(1);
    private final URL requestUrl;
    private WebSocketCall webSocketCall;
    private ScheduledFuture<?> pingFuture;
    private OkHttpClient clonedClient;

    public WatchConnectionManager(OkHttpClient client, BaseOperation<T, L, ?, ?> baseOperation, String version, Watcher<T> watcher, int reconnectInterval, int reconnectLimit, long websocketTimeout, long websocketPingInterval) throws MalformedURLException {
        if (version == null) {
            L currentList = baseOperation.list();
            this.resourceVersion = new AtomicReference<String>(currentList.getMetadata().getResourceVersion());
        } else {
            this.resourceVersion = new AtomicReference<String>(version);
        }
        this.baseOperation = baseOperation;
        this.watcher = watcher;
        this.reconnectInterval = reconnectInterval;
        this.reconnectLimit = reconnectLimit;
        this.websocketTimeout = websocketTimeout;
        this.websocketPingInterval = websocketPingInterval;
        this.clonedClient = client.newBuilder().readTimeout(this.websocketTimeout, TimeUnit.MILLISECONDS).build();
        this.requestUrl = baseOperation.getNamespacedUrl();
        this.runWatch();
    }

    private final void runWatch() {
        logger.debug("Connecting websocket ... {}", (Object)this);
        HttpUrl.Builder httpUrlBuilder = HttpUrl.get((URL)this.requestUrl).newBuilder();
        String labelQueryParam = this.baseOperation.getLabelQueryParam();
        if (Utils.isNotNullOrEmpty(labelQueryParam)) {
            httpUrlBuilder.addQueryParameter("labelSelector", labelQueryParam);
        }
        String fieldQueryString = this.baseOperation.getFieldQueryParam();
        String name = this.baseOperation.getName();
        if (name != null && name.length() > 0) {
            if (fieldQueryString.length() > 0) {
                fieldQueryString = fieldQueryString + ",";
            }
            fieldQueryString = fieldQueryString + "metadata.name=" + name;
        }
        if (Utils.isNotNullOrEmpty(fieldQueryString)) {
            httpUrlBuilder.addQueryParameter("fieldSelector", fieldQueryString);
        }
        httpUrlBuilder.addQueryParameter("resourceVersion", this.resourceVersion.get()).addQueryParameter("watch", "true");
        Request request = new Request.Builder().get().url(httpUrlBuilder.build()).addHeader("Origin", this.requestUrl.getProtocol() + "://" + this.requestUrl.getHost() + ":" + this.requestUrl.getPort()).build();
        this.webSocketCall = WebSocketCall.create((OkHttpClient)this.clonedClient, (Request)request);
        this.webSocketCall.enqueue(new WebSocketListener(){

            public void onOpen(WebSocket webSocket, Response response) {
                if (response != null && response.body() != null) {
                    response.body().close();
                }
                logger.info("WebSocket successfully opened");
                WatchConnectionManager.this.webSocketRef.set(webSocket);
                WatchConnectionManager.this.currentReconnectAttempt.set(0);
                WatchConnectionManager.this.started.set(true);
                WatchConnectionManager.this.queue.clear();
                WatchConnectionManager.this.queue.add(true);
                WatchConnectionManager.this.pingFuture = WatchConnectionManager.this.executor.scheduleAtFixedRate(new NamedRunnable("websocket ping"){

                    @Override
                    public void execute() {
                        WebSocket ws = (WebSocket)WatchConnectionManager.this.webSocketRef.get();
                        if (ws == null) {
                            return;
                        }
                        try {
                            ws.sendPing(new Buffer().writeUtf8("Alive?"));
                        }
                        catch (IOException e) {
                            logger.debug("Failed to send ping", (Throwable)e);
                            WatchConnectionManager.this.closeWebSocket(ws);
                            this.onFailure(e, null);
                        }
                        catch (IllegalStateException e) {
                            logger.error("Tried to send ping in already closed websocket");
                        }
                    }
                }, WatchConnectionManager.this.websocketPingInterval, WatchConnectionManager.this.websocketPingInterval, TimeUnit.MILLISECONDS);
            }

            public void onFailure(IOException e, Response response) {
                if (WatchConnectionManager.this.forceClosed.get()) {
                    logger.warn("Ignoring onFailure for already closed/closing websocket", (Throwable)e);
                    if (response != null && response.body() != null) {
                        response.body().close();
                    }
                    return;
                }
                if (response != null) {
                    Status status = OperationSupport.createStatus(response);
                    if (response.body() != null) {
                        response.body().close();
                    }
                    logger.warn("Exec Failure: HTTP {}, Status: {} - {}", new Object[]{response.code(), status.getCode(), status.getMessage(), e});
                    if (!WatchConnectionManager.this.started.get()) {
                        WatchConnectionManager.this.queue.add(new KubernetesClientException(status));
                        return;
                    }
                } else {
                    logger.warn("Exec Failure", (Throwable)e);
                    if (!WatchConnectionManager.this.started.get()) {
                        WatchConnectionManager.this.queue.add(new KubernetesClientException("Failed to start websocket", e));
                        return;
                    }
                }
                if (WatchConnectionManager.this.currentReconnectAttempt.get() >= WatchConnectionManager.this.reconnectLimit && WatchConnectionManager.this.reconnectLimit >= 0) {
                    WatchConnectionManager.this.watcher.onClose(new KubernetesClientException("Connection failure", e));
                    return;
                }
                WatchConnectionManager.this.scheduleReconnect();
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onMessage(ResponseBody message) throws IOException {
                String messageSource = null;
                try {
                    messageSource = message.string();
                    WatchEvent event = (WatchEvent)mapper.readValue(messageSource, WatchEvent.class);
                    if (event.getObject() instanceof HasMetadata) {
                        String newResourceVersion;
                        HasMetadata obj = (HasMetadata)event.getObject();
                        String currentResourceVersion = (String)WatchConnectionManager.this.resourceVersion.get();
                        if (currentResourceVersion.compareTo(newResourceVersion = obj.getMetadata().getResourceVersion()) < 0) {
                            WatchConnectionManager.this.resourceVersion.compareAndSet(currentResourceVersion, newResourceVersion);
                        }
                        Watcher.Action action = Watcher.Action.valueOf(event.getType());
                        WatchConnectionManager.this.watcher.eventReceived(action, obj);
                    } else if (event.getObject() instanceof Status) {
                        Status status = (Status)event.getObject();
                        if (status.getCode() == 410) {
                            WatchConnectionManager.this.webSocketRef.set(null);
                            WatchConnectionManager.this.close();
                            WatchConnectionManager.this.watcher.onClose(new KubernetesClientException(status));
                            return;
                        }
                        logger.error("Error received: {}", (Object)status.toString());
                    } else {
                        logger.error("Unknown message received: {}", (Object)messageSource);
                    }
                }
                catch (IOException e) {
                    logger.error("Could not deserialize watch event: {}", (Object)messageSource, (Object)e);
                }
                catch (ClassCastException e) {
                    logger.error("Received wrong type of object for watch", (Throwable)e);
                }
                catch (IllegalArgumentException e) {
                    logger.error("Invalid event type", (Throwable)e);
                }
                finally {
                    message.close();
                }
            }

            public void onPong(Buffer buffer) {
                try {
                    if (!buffer.equals((Object)new Buffer().writeUtf8("Alive?"))) {
                        logger.error("Failed to verify pong");
                        WatchConnectionManager.this.closeWebSocket((WebSocket)WatchConnectionManager.this.webSocketRef.get());
                        this.onClose(4000, "WebSocket pong verification error");
                    }
                }
                finally {
                    buffer.close();
                }
            }

            public void onClose(int code, String reason) {
                logger.info("WebSocket close received. code: {}, reason: {}", (Object)code, (Object)reason);
                if (WatchConnectionManager.this.forceClosed.get()) {
                    logger.warn("Ignoring onClose for already closed/closing websocket");
                    return;
                }
                if (WatchConnectionManager.this.currentReconnectAttempt.get() >= WatchConnectionManager.this.reconnectLimit && WatchConnectionManager.this.reconnectLimit >= 0) {
                    WatchConnectionManager.this.watcher.onClose(new KubernetesClientException("Connection unexpectedly closed"));
                    return;
                }
                WatchConnectionManager.this.scheduleReconnect();
            }
        });
    }

    private void scheduleReconnect() {
        logger.debug("Submitting reconnect task to the executor");
        this.executor.submit(new NamedRunnable("scheduleReconnect"){

            @Override
            public void execute() {
                if (!WatchConnectionManager.this.reconnectPending.compareAndSet(false, true)) {
                    logger.debug("Reconnect already scheduled");
                    return;
                }
                WatchConnectionManager.this.stopWsPing();
                WatchConnectionManager.this.webSocketRef.set(null);
                try {
                    logger.debug("Scheduling reconnect task");
                    WatchConnectionManager.this.executor.schedule(new NamedRunnable("reconnectAttempt"){

                        @Override
                        public void execute() {
                            try {
                                WatchConnectionManager.this.runWatch();
                                WatchConnectionManager.this.reconnectPending.set(false);
                            }
                            catch (Exception e) {
                                logger.error("Exception in reconnect", (Throwable)e);
                                WatchConnectionManager.this.webSocketRef.set(null);
                                WatchConnectionManager.this.close();
                                WatchConnectionManager.this.watcher.onClose(new KubernetesClientException("Unhandled exception in reconnect attempt", e));
                            }
                        }
                    }, WatchConnectionManager.this.nextReconnectInterval(), TimeUnit.MILLISECONDS);
                }
                catch (RejectedExecutionException e) {
                    WatchConnectionManager.this.reconnectPending.set(false);
                }
            }
        });
    }

    public void waitUntilReady() {
        Utils.waitUntilReady(this.queue, 10L, TimeUnit.SECONDS);
    }

    @Override
    public void close() {
        logger.debug("Force closing the watch {}", (Object)this);
        this.forceClosed.set(true);
        this.stopWsPing();
        this.closeWebSocket(this.webSocketRef.getAndSet(null));
        if (!this.executor.isShutdown()) {
            try {
                this.executor.shutdown();
                if (!this.executor.awaitTermination(1L, TimeUnit.SECONDS)) {
                    logger.warn("Executor didn't terminate in time after shutdown in close(), killing it in: {}", (Object)this);
                    this.executor.shutdownNow();
                }
            }
            catch (Throwable t) {
                throw KubernetesClientException.launderThrowable(t);
            }
        }
    }

    private void stopWsPing() {
        if (this.pingFuture != null && !this.pingFuture.isCancelled() && !this.pingFuture.isDone()) {
            this.pingFuture.cancel(true);
        }
        this.pingFuture = null;
    }

    private void closeWebSocket(WebSocket ws) {
        if (ws != null) {
            logger.debug("Closing websocket {}", (Object)ws);
            try {
                ws.close(0, null);
            }
            catch (IOException e) {
                if ("closed".equals(e.getMessage())) {
                    logger.debug("Socket already closed");
                } else {
                    logger.warn("Failed to write websocket close message: {} {}", e.getClass(), (Object)e.getMessage());
                }
            }
            catch (IllegalStateException e) {
                logger.error("Called close on already closed websocket: {} {}", e.getClass(), (Object)e.getMessage());
            }
        }
    }

    private long nextReconnectInterval() {
        int exponentOfTwo = this.currentReconnectAttempt.getAndIncrement();
        if (exponentOfTwo > 5) {
            exponentOfTwo = 5;
        }
        long ret = this.reconnectInterval * (1 << exponentOfTwo);
        logger.info("Current reconnect backoff is " + ret + " milliseconds (T" + exponentOfTwo + ")");
        return ret;
    }

    private static abstract class NamedRunnable
    implements Runnable {
        private final String name;

        public NamedRunnable(String name) {
            this.name = Objects.requireNonNull(name);
        }

        private void tryToSetName(String value) {
            try {
                Thread.currentThread().setName(value);
            }
            catch (SecurityException securityException) {
                // empty catch block
            }
        }

        @Override
        public final void run() {
            String oldName = Thread.currentThread().getName();
            this.tryToSetName(this.name + "|" + oldName);
            try {
                this.execute();
            }
            finally {
                this.tryToSetName(oldName);
            }
        }

        protected abstract void execute();
    }
}

