/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.utils;

import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.ConfigBuilder;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.http.HttpClient;
import io.fabric8.kubernetes.client.utils.BackwardsCompatibilityInterceptor;
import io.fabric8.kubernetes.client.utils.HttpClientUtils;
import io.fabric8.kubernetes.client.utils.ImpersonatorInterceptor;
import io.fabric8.kubernetes.client.utils.TokenRefreshInterceptor;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.List;
import java.util.ServiceLoader;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.AssertionsForClassTypes;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;
import org.mockito.ArgumentMatchers;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import org.mockito.stubbing.Answer;

class HttpClientUtilsTest {
    HttpClientUtilsTest() {
    }

    @Test
    void toProxyTypeTestUnknown() {
        org.junit.jupiter.api.Assertions.assertThrows(MalformedURLException.class, () -> HttpClientUtils.toProxyType((String)"unknown"));
    }

    @Test
    void toProxyTypeTestNull() {
        org.junit.jupiter.api.Assertions.assertThrows(MalformedURLException.class, () -> HttpClientUtils.toProxyType(null));
    }

    @Test
    void toProxyTypeTestHttps() throws MalformedURLException {
        org.junit.jupiter.api.Assertions.assertEquals((Object)HttpClient.ProxyType.HTTP, (Object)HttpClientUtils.toProxyType((String)"https"));
    }

    @Test
    void testConfigureSocksProxy() throws Exception {
        Config config = ((ConfigBuilder)((ConfigBuilder)new ConfigBuilder().withMasterUrl("http://localhost")).withHttpProxy("socks5://192.168.0.1:8080")).build();
        HttpClient.Builder builder = (HttpClient.Builder)Mockito.mock(HttpClient.Builder.class, (Answer)Mockito.RETURNS_SELF);
        HttpClientUtils.configureProxy((Config)config, (HttpClient.Builder)builder);
        ((HttpClient.Builder)Mockito.verify((Object)builder)).proxyType(HttpClient.ProxyType.SOCKS5);
        ((HttpClient.Builder)Mockito.verify((Object)builder)).proxyAddress(new InetSocketAddress("192.168.0.1", 8080));
    }

    @Test
    void testConfigureProxyAuth() throws Exception {
        Config config = ((ConfigBuilder)((ConfigBuilder)new ConfigBuilder().withMasterUrl("http://localhost")).withHttpProxy("http://user:password@192.168.0.1:8080")).build();
        HttpClient.Builder builder = (HttpClient.Builder)Mockito.mock(HttpClient.Builder.class, (Answer)Mockito.RETURNS_SELF);
        HttpClientUtils.configureProxy((Config)config, (HttpClient.Builder)builder);
        ((HttpClient.Builder)Mockito.verify((Object)builder)).proxyType(HttpClient.ProxyType.HTTP);
        ((HttpClient.Builder)Mockito.verify((Object)builder)).proxyAuthorization("Basic dXNlcjpwYXNzd29yZA==");
    }

    @Test
    void testCreateApplicableInterceptors() {
        Config config = new ConfigBuilder().build();
        Collection interceptorList = HttpClientUtils.createApplicableInterceptors((Config)config, null).values();
        ((AbstractCollectionAssert)((AbstractCollectionAssert)((AbstractCollectionAssert)Assertions.assertThat(interceptorList).isNotNull()).hasSize(3)).hasAtLeastOneElementOfType(ImpersonatorInterceptor.class)).hasAtLeastOneElementOfType(TokenRefreshInterceptor.class);
    }

    @Test
    void testCreateApplicableInterceptorsWithBackwardsCompatibilityDisabled() {
        Config config = new ConfigBuilder().build();
        System.setProperty("kubernetes.backwardsCompatibilityInterceptor.disable", "true");
        try {
            Collection interceptorList = HttpClientUtils.createApplicableInterceptors((Config)config, null).values();
            ((AbstractCollectionAssert)((AbstractCollectionAssert)((AbstractCollectionAssert)((AbstractCollectionAssert)Assertions.assertThat(interceptorList).isNotNull()).hasSize(3)).noneMatch(i -> i instanceof BackwardsCompatibilityInterceptor)).hasAtLeastOneElementOfType(ImpersonatorInterceptor.class)).hasAtLeastOneElementOfType(TokenRefreshInterceptor.class);
        }
        finally {
            System.clearProperty("kubernetes.backwardsCompatibilityInterceptor.disable");
        }
    }

    @ParameterizedTest
    @MethodSource(value={"basicCredentialsInput"})
    void testBasicCredentials(String username, String password, String authentication) {
        String result = HttpClientUtils.basicCredentials((String)username, (String)password);
        Assertions.assertThat((String)result).isEqualTo(authentication);
        String decoded = new String(Base64.getDecoder().decode(result.substring("Basic ".length())), StandardCharsets.UTF_8);
        Assertions.assertThat((String)decoded).isEqualTo(username + ":" + password);
    }

    static Stream<Arguments> basicCredentialsInput() {
        return Stream.of(Arguments.arguments((Object[])new Object[]{"username", "password", "Basic dXNlcm5hbWU6cGFzc3dvcmQ="}), Arguments.arguments((Object[])new Object[]{"username", "\u00dea\u00df\u00dfword\u00a3", "Basic dXNlcm5hbWU6w55hw5/Dn3dvcmTCow=="}));
    }

    @Nested
    @DisplayName(value="getHttpClientFactory")
    class GetHttpClientFactory {
        private List<HttpClient.Factory> factories;
        private MockedStatic<ServiceLoader> mockedStaticServiceLoader;

        GetHttpClientFactory() {
        }

        @BeforeEach
        void setUp() {
            this.factories = new ArrayList<HttpClient.Factory>();
            this.mockedStaticServiceLoader = Mockito.mockStatic(ServiceLoader.class);
            ServiceLoader mockServiceLoader = (ServiceLoader)Mockito.mock(ServiceLoader.class);
            ((ServiceLoader)Mockito.doCallRealMethod().when((Object)mockServiceLoader)).forEach((Consumer)Mockito.any());
            Mockito.when(mockServiceLoader.iterator()).thenAnswer(i -> new ArrayList<HttpClient.Factory>(this.factories).iterator());
            Mockito.when(ServiceLoader.load((Class)ArgumentMatchers.eq(HttpClient.Factory.class), (ClassLoader)ArgumentMatchers.any())).thenReturn((Object)mockServiceLoader);
            this.mockedStaticServiceLoader.when(() -> ServiceLoader.load((Class)ArgumentMatchers.eq(HttpClient.Factory.class))).thenReturn((Object)mockServiceLoader);
        }

        @AfterEach
        void tearDown() {
            this.factories.clear();
            this.mockedStaticServiceLoader.close();
        }

        @Test
        @DisplayName(value="With no implementations, should throw exception")
        void withNoImplementations() {
            KubernetesClientException result = (KubernetesClientException)org.junit.jupiter.api.Assertions.assertThrows(KubernetesClientException.class, HttpClientUtils::getHttpClientFactory);
            org.junit.jupiter.api.Assertions.assertEquals((Object)"No httpclient implementations found on the context classloader, please ensure your classpath includes an implementation jar", (Object)result.getMessage());
        }

        @Test
        @DisplayName(value="With default implementation, should return default")
        void withDefault() {
            this.factories.add(new DefaultHttpClientFactory());
            org.junit.jupiter.api.Assertions.assertEquals(DefaultHttpClientFactory.class, HttpClientUtils.getHttpClientFactory().getClass());
        }

        @Test
        @DisplayName(value="With default and other implementation, should return other")
        void withDefaultAndOther() {
            this.factories.add(new ScoredHttpClientFactory("other", 0));
            this.factories.add(new DefaultHttpClientFactory());
            org.junit.jupiter.api.Assertions.assertEquals(ScoredHttpClientFactory.class, HttpClientUtils.getHttpClientFactory().getClass());
        }

        @Test
        @DisplayName(value="With default, other, and priority implementations; should return Priority")
        void withDefaultAndPriorityAndOther() {
            this.factories.add(new ScoredHttpClientFactory("other", 0));
            this.factories.add(new ScoredHttpClientFactory("priority", Integer.MAX_VALUE));
            this.factories.add(new DefaultHttpClientFactory());
            HttpClient.Factory result = HttpClientUtils.getHttpClientFactory();
            org.junit.jupiter.api.Assertions.assertEquals(ScoredHttpClientFactory.class, result.getClass());
            org.junit.jupiter.api.Assertions.assertEquals((int)Integer.MAX_VALUE, (int)result.priority());
            org.junit.jupiter.api.Assertions.assertEquals((Object)"priority", (Object)((ScoredHttpClientFactory)result).id);
        }

        @Test
        @DisplayName(value="With multiple implementations and several with max priority, should return first of max priority")
        void withMultipleAndCollision() {
            this.factories.add(new DefaultHttpClientFactory());
            this.factories.add(new ScoredHttpClientFactory("other", 0));
            this.factories.add(new ScoredHttpClientFactory("priority-1", Integer.MAX_VALUE));
            this.factories.add(new ScoredHttpClientFactory("priority-2", Integer.MAX_VALUE));
            this.factories.add(new DefaultHttpClientFactory());
            HttpClient.Factory result = HttpClientUtils.getHttpClientFactory();
            org.junit.jupiter.api.Assertions.assertEquals(ScoredHttpClientFactory.class, result.getClass());
            org.junit.jupiter.api.Assertions.assertEquals((int)Integer.MAX_VALUE, (int)result.priority());
            org.junit.jupiter.api.Assertions.assertEquals((Object)"priority-1", (Object)((ScoredHttpClientFactory)result).id);
        }

        private final class DefaultHttpClientFactory
        implements HttpClient.Factory {
            private DefaultHttpClientFactory() {
            }

            public HttpClient.Builder newBuilder() {
                return null;
            }

            public int priority() {
                return -1;
            }
        }

        private final class ScoredHttpClientFactory
        implements HttpClient.Factory {
            private final String id;
            private final int priority;

            public ScoredHttpClientFactory(String id, int priority) {
                this.id = id;
                this.priority = priority;
            }

            public HttpClient.Builder newBuilder() {
                return null;
            }

            public int priority() {
                return this.priority;
            }
        }
    }

    @Nested
    @DisplayName(value="getProxyUrl")
    @TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
    class GetProxyUrl {
        private ConfigBuilder configBuilder;

        GetProxyUrl() {
        }

        @BeforeEach
        void setUp() {
            this.configBuilder = new ConfigBuilder();
        }

        @DisplayName(value="noProxy matching master hostname")
        @ParameterizedTest(name="{index}: Master hostname ''{0}'' matched by No Proxy ''{1}'' ")
        @MethodSource(value={"masterHostnameDoesMatchNoProxyInput"})
        void masterHostnameDoesMatchNoProxy(String masterHostname, String[] noProxy) throws MalformedURLException {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)HttpClientUtils.isHostMatchedByNoProxy((String)masterHostname, (String[])noProxy));
        }

        Stream<Arguments> masterHostnameDoesMatchNoProxyInput() {
            return Stream.of(Arguments.arguments((Object[])new Object[]{"192.168.1.100", new String[]{"192.168.1.0/24"}}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[]{"master.example.com"}}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[]{".example.com"}}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[]{"circleci-internal-outer-build-agent", "one.com", "other.com", ".com"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"192.168.1.110"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"http://192.168.1.110"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"https://192.168.1.110"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"192.168.1.0/24"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"http://192.168.1.0/24"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"192.0.0.0/8"}}), Arguments.arguments((Object[])new Object[]{"2620:52:0:9c:0:0:0:1", new String[]{"2620:52:0:9c::/64"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"http://192.0.0.0/8"}}));
        }

        @DisplayName(value="noProxy not matching master hostname")
        @ParameterizedTest(name="{index}: Master hostname ''{0}'' not matched by No Proxy ''{1}'' ")
        @MethodSource(value={"masterHostnameDoesNotMatchNoProxyInput"})
        void masterHostnameDoesNotMatchNoProxy(String masterHostname, String[] noProxy) throws MalformedURLException, URISyntaxException {
            org.junit.jupiter.api.Assertions.assertFalse((boolean)HttpClientUtils.isHostMatchedByNoProxy((String)masterHostname, (String[])noProxy));
        }

        Stream<Arguments> masterHostnameDoesNotMatchNoProxyInput() {
            return Stream.of(Arguments.arguments((Object[])new Object[]{"192.168.2.100", new String[]{"192.168.1.0/24"}}), Arguments.arguments((Object[])new Object[]{"master.example.com", null}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[0]}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[]{"master1.example.com"}}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[]{".example1.com"}}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[]{"circleci-internal-outer-build-agent"}}), Arguments.arguments((Object[])new Object[]{"master.example.com", new String[]{"one.com", "other.com", "master.example.", ".co"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"192.168.1.111"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"http://192.168.1.111"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"https://192.168.1.111"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"192.168.10.0/24"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"http://192.168.1.0/32"}}), Arguments.arguments((Object[])new Object[]{"192.168.1.110", new String[]{"http://192.168.1.0/33"}}));
        }

        @DisplayName(value="With httpProxy and invalid noProxy should throw Exception")
        @ParameterizedTest(name="{index}: ''{0}'' is invalid")
        @ValueSource(strings={"*.my.domain.com", "!!!.com", "()-?\u00bf?", "http://proxy url"})
        void invalidNoProxyUrlThrowsException(String noProxy) {
            ((AbstractThrowableAssert)AssertionsForClassTypes.assertThatThrownBy(() -> HttpClientUtils.isHostMatchedByNoProxy((String)"http://localhost", (String[])new String[]{noProxy})).isInstanceOf(MalformedURLException.class)).hasMessage("NO_PROXY URL contains invalid entry: '" + noProxy + "'");
        }

        @Test
        void whenHttpsProxyUrlWithNoPort_shouldThrowException() {
            Config config = ((ConfigBuilder)((ConfigBuilder)this.configBuilder.withMasterUrl("http://localhost")).withHttpProxy("http://192.168.0.1")).build();
            Assertions.assertThatIllegalArgumentException().isThrownBy(() -> HttpClientUtils.getProxyUri((URL)new URL("http://localhost"), (Config)config)).withMessage("Failure in creating proxy URL. Proxy port is required!");
        }

        @Test
        void withNoHttpProxyProvidedReturnsNull() throws MalformedURLException, URISyntaxException {
            Config config = ((ConfigBuilder)this.configBuilder.withNoProxy(new String[]{"other.url"})).build();
            URI url = HttpClientUtils.getProxyUri((URL)new URL("http://localhost"), (Config)config);
            Assertions.assertThat((URI)url).isNull();
        }

        @Test
        void whenIncompleteHttpProxyUrlProvided_shouldInferProtocol() throws MalformedURLException, URISyntaxException {
            Config config = ((ConfigBuilder)this.configBuilder.withHttpProxy("example.com:8080")).build();
            URI url = HttpClientUtils.getProxyUri((URL)new URL("http://localhost"), (Config)config);
            Assertions.assertThat((URI)url).hasScheme("http").hasHost("example.com").hasPort(8080);
        }
    }
}

