/*
 * Decompiled with CFR 0.152.
 */
package io.digdag.core.storage;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.io.BaseEncoding;
import com.google.inject.Inject;
import io.digdag.client.config.Config;
import io.digdag.core.repository.ArchiveType;
import io.digdag.core.repository.ProjectStore;
import io.digdag.core.repository.ResourceNotFoundException;
import io.digdag.core.repository.StoredRevision;
import io.digdag.core.storage.StorageManager;
import io.digdag.spi.DirectDownloadHandle;
import io.digdag.spi.Storage;
import io.digdag.spi.StorageFileNotFoundException;
import io.digdag.spi.StorageObject;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Locale;
import java.util.UUID;

public class ArchiveManager {
    private final StorageManager storageManager;
    private final LoadingCache<ArchiveType, Storage> storageCache;
    private final ArchiveType uploadArchiveType;
    private final Config systemConfig;
    private final String pathPrefix;
    private final boolean directDownloadEnabled;
    private static final DateTimeFormatter DATE_TIME_SUFFIX_FORMAT = DateTimeFormatter.ofPattern("yyyyMM'T'ddHHmmss'Z'").withZone(ZoneId.of("UTC"));

    @Inject
    public ArchiveManager(StorageManager storageManager, Config systemConfig) {
        this.storageManager = storageManager;
        this.storageCache = CacheBuilder.newBuilder().maximumSize(2L).build((CacheLoader)new CacheLoader<ArchiveType, Storage>(){

            public Storage load(ArchiveType type) {
                return ArchiveManager.this.openStorage(type);
            }
        });
        this.systemConfig = systemConfig;
        this.uploadArchiveType = (ArchiveType)systemConfig.get("archive.type", ArchiveType.class, (Object)ArchiveType.DB);
        this.pathPrefix = this.getArchivePathPrefix(systemConfig, this.uploadArchiveType);
        this.directDownloadEnabled = (Boolean)systemConfig.get("archive." + this.uploadArchiveType + ".direct_download", Boolean.class, (Object)false);
    }

    Storage openStorage(ArchiveType type) {
        return this.storageManager.create(type.getName(), this.systemConfig, "archive.");
    }

    private String getArchivePathPrefix(Config systemConfig, ArchiveType type) {
        String pathPrefix = (String)systemConfig.get("archive." + type + ".path", String.class, (Object)"");
        if (pathPrefix.startsWith("/")) {
            pathPrefix = pathPrefix.substring(1);
        }
        if (!pathPrefix.endsWith("/") && !pathPrefix.isEmpty()) {
            pathPrefix = pathPrefix + "/";
        }
        return pathPrefix;
    }

    public Location newArchiveLocation(int siteId, String projectName, String revisionName, long contentLength) {
        if (this.uploadArchiveType.equals(ArchiveType.DB)) {
            return new Location(this.uploadArchiveType, null);
        }
        return new Location(this.uploadArchiveType, this.formatFilePath(siteId, projectName, revisionName));
    }

    public Storage getStorage(ArchiveType type) {
        return (Storage)this.storageCache.getUnchecked((Object)type);
    }

    private String formatFilePath(int siteId, String projectName, String revisionName) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)projectName) ? 1 : 0) != 0, (Object)"projectName");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)revisionName) ? 1 : 0) != 0, (Object)"revisionName");
        String encodedProjectName = BaseEncoding.base64Url().omitPadding().encode(projectName.getBytes(StandardCharsets.UTF_8));
        return String.format(Locale.ENGLISH, "%s%d/%s/%s.%s.tar.gz", this.pathPrefix, siteId, encodedProjectName, revisionName, UUID.randomUUID().toString());
    }

    public Optional<StorageObject> openArchive(ProjectStore ps, int projectId, String revisionName) throws ResourceNotFoundException, StorageFileNotFoundException {
        StoredRevision rev = this.findRevision(ps, projectId, revisionName);
        ArchiveType type = rev.getArchiveType();
        if (type.equals(ArchiveType.NONE)) {
            return Optional.absent();
        }
        if (type.equals(ArchiveType.DB)) {
            byte[] data = ps.getRevisionArchiveData(rev.getId());
            return Optional.of((Object)new StorageObject((InputStream)new ByteArrayInputStream(data), (long)data.length));
        }
        return Optional.of((Object)this.getStorage(type).open((String)rev.getArchivePath().or((Object)"")));
    }

    public Optional<StoredArchive> getArchive(ProjectStore ps, int projectId, String revisionName) throws ResourceNotFoundException {
        final StoredRevision rev = this.findRevision(ps, projectId, revisionName);
        ArchiveType type = rev.getArchiveType();
        if (type.equals(ArchiveType.NONE)) {
            return Optional.absent();
        }
        if (type.equals(ArchiveType.DB)) {
            final byte[] data = ps.getRevisionArchiveData(rev.getId());
            return Optional.of((Object)new StoredArchive(){

                @Override
                public Optional<byte[]> getByteArray() {
                    return Optional.of((Object)data);
                }

                @Override
                public Optional<DirectDownloadHandle> getDirectDownloadHandle() {
                    return Optional.absent();
                }

                @Override
                public StorageObject open() {
                    return new StorageObject((InputStream)new ByteArrayInputStream(data), (long)data.length);
                }
            });
        }
        final Storage storage = this.getStorage(type);
        return Optional.of((Object)new StoredArchive(){

            @Override
            public Optional<byte[]> getByteArray() {
                return Optional.absent();
            }

            @Override
            public Optional<DirectDownloadHandle> getDirectDownloadHandle() {
                if (!ArchiveManager.this.directDownloadEnabled) {
                    return Optional.absent();
                }
                return storage.getDirectDownloadHandle((String)rev.getArchivePath().or((Object)""));
            }

            @Override
            public StorageObject open() throws StorageFileNotFoundException {
                return storage.open((String)rev.getArchivePath().or((Object)""));
            }
        });
    }

    private StoredRevision findRevision(ProjectStore ps, int projectId, String revisionName) throws ResourceNotFoundException {
        if (revisionName == null) {
            return ps.getLatestRevision(projectId);
        }
        return ps.getRevisionByName(projectId, revisionName);
    }

    public static interface StoredArchive {
        public Optional<byte[]> getByteArray();

        public Optional<DirectDownloadHandle> getDirectDownloadHandle();

        public StorageObject open() throws StorageFileNotFoundException;
    }

    public static class Location {
        private final ArchiveType archiveType;
        private final String path;

        public Location(ArchiveType archiveType, String path) {
            this.archiveType = archiveType;
            this.path = path;
        }

        public ArchiveType getArchiveType() {
            return this.archiveType;
        }

        public String getPath() {
            return this.path;
        }
    }

    public static interface Upload {
        public byte[] get() throws IOException, InterruptedException;
    }
}

