/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.relational.history;

import io.debezium.annotation.ThreadSafe;
import io.debezium.config.Configuration;
import io.debezium.config.Field;
import io.debezium.document.DocumentReader;
import io.debezium.document.DocumentWriter;
import io.debezium.relational.history.AbstractDatabaseHistory;
import io.debezium.relational.history.DatabaseHistoryException;
import io.debezium.relational.history.DatabaseHistoryListener;
import io.debezium.relational.history.HistoryRecord;
import io.debezium.relational.history.HistoryRecordComparator;
import io.debezium.util.Collect;
import io.debezium.util.FunctionalReadWriteLock;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.apache.kafka.connect.errors.ConnectException;
import org.slf4j.Logger;

@ThreadSafe
public final class FileDatabaseHistory
extends AbstractDatabaseHistory {
    public static final Field FILE_PATH = Field.create("database.history.file.filename").withDescription("The path to the file that will be used to record the database history").required();
    public static Collection<Field> ALL_FIELDS = Collect.arrayListOf(FILE_PATH, new Field[0]);
    private static final Charset UTF8 = StandardCharsets.UTF_8;
    private final FunctionalReadWriteLock lock = FunctionalReadWriteLock.reentrant();
    private final DocumentWriter writer = DocumentWriter.defaultWriter();
    private final DocumentReader reader = DocumentReader.defaultReader();
    private final AtomicBoolean running = new AtomicBoolean();
    private Path path;

    @Override
    public void configure(Configuration config, HistoryRecordComparator comparator, DatabaseHistoryListener listener, boolean useCatalogBeforeSchema) {
        if (!config.validateAndRecord(ALL_FIELDS, arg_0 -> ((Logger)this.logger).error(arg_0))) {
            throw new ConnectException("Error configuring an instance of " + this.getClass().getSimpleName() + "; check the logs for details");
        }
        config.validateAndRecord(ALL_FIELDS, arg_0 -> ((Logger)this.logger).error(arg_0));
        if (this.running.get()) {
            throw new IllegalStateException("Database history file already initialized to " + this.path);
        }
        super.configure(config, comparator, listener, useCatalogBeforeSchema);
        this.path = Paths.get(config.getString(FILE_PATH), new String[0]);
    }

    @Override
    public void start() {
        super.start();
        this.lock.write(() -> {
            if (this.running.compareAndSet(false, true)) {
                Path path = this.path;
                if (path == null) {
                    throw new IllegalStateException("FileDatabaseHistory must be configured before it is started");
                }
                try {
                    if (!this.storageExists()) {
                        if (path.getParent() != null) {
                            Files.createDirectories(path.getParent(), new FileAttribute[0]);
                        }
                        try {
                            Files.createFile(path, new FileAttribute[0]);
                        }
                        catch (FileAlreadyExistsException fileAlreadyExistsException) {}
                    }
                }
                catch (IOException e) {
                    throw new DatabaseHistoryException("Unable to create history file at " + path + ": " + e.getMessage(), e);
                }
            }
        });
    }

    @Override
    protected void storeRecord(HistoryRecord record) throws DatabaseHistoryException {
        if (record == null) {
            return;
        }
        this.lock.write(() -> {
            if (!this.running.get()) {
                throw new IllegalStateException("The history has been stopped and will not accept more records");
            }
            try {
                String line = this.writer.write(record.document());
                try (BufferedWriter historyWriter = Files.newBufferedWriter(this.path, StandardOpenOption.APPEND);){
                    try {
                        historyWriter.append(line);
                        historyWriter.newLine();
                    }
                    catch (IOException e) {
                        this.logger.error("Failed to add record to history at {}: {}", new Object[]{this.path, record, e});
                        if (historyWriter != null) {
                            historyWriter.close();
                        }
                        return;
                    }
                }
                catch (IOException e) {
                    throw new DatabaseHistoryException("Unable to create writer for history file " + this.path + ": " + e.getMessage(), e);
                }
            }
            catch (IOException e) {
                this.logger.error("Failed to convert record to string: {}", (Object)record, (Object)e);
            }
        });
    }

    @Override
    public void stop() {
        this.running.set(false);
        super.stop();
    }

    @Override
    protected synchronized void recoverRecords(Consumer<HistoryRecord> records) {
        this.lock.write(() -> {
            try {
                if (this.exists()) {
                    for (String line : Files.readAllLines(this.path, UTF8)) {
                        if (line == null || line.isEmpty()) continue;
                        records.accept(new HistoryRecord(this.reader.read(line)));
                    }
                }
            }
            catch (IOException e) {
                this.logger.error("Failed to add recover records from history at {}", (Object)this.path, (Object)e);
            }
        });
    }

    @Override
    public boolean storageExists() {
        return Files.exists(this.path, new LinkOption[0]);
    }

    @Override
    public boolean exists() {
        return this.storageExists();
    }

    public String toString() {
        return "file " + (this.path != null ? this.path : "(unstarted)");
    }
}

