/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.oracle;

import io.debezium.DebeziumException;
import io.debezium.config.Field;
import io.debezium.connector.oracle.OracleConnectorConfig;
import io.debezium.connector.oracle.OracleDatabaseVersion;
import io.debezium.connector.oracle.Scn;
import io.debezium.jdbc.JdbcConfiguration;
import io.debezium.jdbc.JdbcConnection;
import io.debezium.relational.Column;
import io.debezium.relational.ColumnEditor;
import io.debezium.relational.TableId;
import io.debezium.relational.Tables;
import io.debezium.util.Strings;
import java.sql.Clob;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.sql.SQLRecoverableException;
import java.sql.Statement;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.RetriableException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OracleConnection
extends JdbcConnection {
    private static final Logger LOGGER = LoggerFactory.getLogger(OracleConnection.class);
    private static final int ORACLE_UNSET_SCALE = -127;
    private static final Pattern SYS_NC_PATTERN = Pattern.compile("^SYS_NC(?:_OID|_ROWINFO|[0-9][0-9][0-9][0-9][0-9])\\$$");
    private static final Pattern ADT_INDEX_NAMES_PATTERN = Pattern.compile("^\".*\"\\.\".*\".*");
    private static final Pattern MROW_PATTERN = Pattern.compile("^M_ROW\\$\\$");
    private static final Field URL = Field.create((String)"url", (String)"Raw JDBC url");
    private final OracleDatabaseVersion databaseVersion = this.resolveOracleDatabaseVersion();
    private static final String QUOTED_CHARACTER = "\"";

    public OracleConnection(JdbcConfiguration config) {
        this(config, true);
    }

    public OracleConnection(JdbcConfiguration config, JdbcConnection.ConnectionFactory connectionFactory) {
        this(config, connectionFactory, true);
    }

    public OracleConnection(JdbcConfiguration config, JdbcConnection.ConnectionFactory connectionFactory, boolean showVersion) {
        super(config, connectionFactory, QUOTED_CHARACTER, QUOTED_CHARACTER);
        if (showVersion) {
            LOGGER.info("Database Version: {}", (Object)this.databaseVersion.getBanner());
        }
    }

    public OracleConnection(JdbcConfiguration config, boolean showVersion) {
        super(config, OracleConnection.resolveConnectionFactory(config), QUOTED_CHARACTER, QUOTED_CHARACTER);
        if (showVersion) {
            LOGGER.info("Database Version: {}", (Object)this.databaseVersion.getBanner());
        }
    }

    public void setSessionToPdb(String pdbName) {
        Statement statement = null;
        try {
            statement = this.connection().createStatement();
            statement.execute("alter session set container=" + pdbName);
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOGGER.error("Couldn't close statement", (Throwable)e);
                }
            }
        }
    }

    public void resetSessionToCdb() {
        Statement statement = null;
        try {
            statement = this.connection().createStatement();
            statement.execute("alter session set container=cdb$root");
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (statement != null) {
                try {
                    statement.close();
                }
                catch (SQLException e) {
                    LOGGER.error("Couldn't close statement", (Throwable)e);
                }
            }
        }
    }

    public OracleDatabaseVersion getOracleVersion() {
        return this.databaseVersion;
    }

    private OracleDatabaseVersion resolveOracleDatabaseVersion() {
        String versionStr;
        try {
            try {
                versionStr = (String)this.queryAndMap("SELECT BANNER_FULL FROM V$VERSION WHERE BANNER_FULL LIKE 'Oracle Database%'", rs -> {
                    if (rs.next()) {
                        return rs.getString(1);
                    }
                    return null;
                });
            }
            catch (SQLException e) {
                if (e.getMessage().contains("ORA-00904: \"BANNER_FULL\"")) {
                    LOGGER.debug("BANNER_FULL column not in V$VERSION, using BANNER column as fallback");
                    versionStr = null;
                }
                throw e;
            }
            if (versionStr == null) {
                versionStr = (String)this.queryAndMap("SELECT BANNER FROM V$VERSION WHERE BANNER LIKE 'Oracle Database%'", rs -> {
                    if (rs.next()) {
                        return rs.getString(1);
                    }
                    return null;
                });
            }
        }
        catch (SQLException e) {
            if (e instanceof SQLRecoverableException) {
                throw new RetriableException("Failed to resolve Oracle database version", (Throwable)e);
            }
            throw new RuntimeException("Failed to resolve Oracle database version", e);
        }
        if (versionStr == null) {
            throw new RuntimeException("Failed to resolve Oracle database version");
        }
        return OracleDatabaseVersion.parse(versionStr);
    }

    public Set<TableId> readTableNames(String databaseCatalog, String schemaNamePattern, String tableNamePattern, String[] tableTypes) throws SQLException {
        Set tableIds = super.readTableNames(null, schemaNamePattern, tableNamePattern, tableTypes);
        return tableIds.stream().map(t -> new TableId(databaseCatalog, t.schema(), t.table())).collect(Collectors.toSet());
    }

    protected Set<TableId> getAllTableIds(String catalogName) throws SQLException {
        String query = "select owner, table_name from all_tables where table_name NOT LIKE 'MDRT_%' and table_name NOT LIKE 'MDRS_%' and table_name NOT LIKE 'MDXT_%' and (table_name NOT LIKE 'SYS_IOT_OVER_%' and IOT_NAME IS NULL) and nested = 'NO'and table_name not in (select PARENT_TABLE_NAME from ALL_NESTED_TABLES)";
        HashSet<TableId> tableIds = new HashSet<TableId>();
        this.query("select owner, table_name from all_tables where table_name NOT LIKE 'MDRT_%' and table_name NOT LIKE 'MDRS_%' and table_name NOT LIKE 'MDXT_%' and (table_name NOT LIKE 'SYS_IOT_OVER_%' and IOT_NAME IS NULL) and nested = 'NO'and table_name not in (select PARENT_TABLE_NAME from ALL_NESTED_TABLES)", rs -> {
            while (rs.next()) {
                tableIds.add(new TableId(catalogName, rs.getString(1), rs.getString(2)));
            }
            LOGGER.trace("TableIds are: {}", (Object)tableIds);
        });
        return tableIds;
    }

    protected String resolveCatalogName(String catalogName) {
        String pdbName = this.config().getString("pdb.name");
        return (!Strings.isNullOrEmpty((String)pdbName) ? pdbName : this.config().getString("dbname")).toUpperCase();
    }

    public List<String> readTableUniqueIndices(DatabaseMetaData metadata, TableId id) throws SQLException {
        return super.readTableUniqueIndices(metadata, id.toDoubleQuoted());
    }

    public Optional<Instant> getCurrentTimestamp() throws SQLException {
        return (Optional)this.queryAndMap("SELECT CURRENT_TIMESTAMP FROM DUAL", rs -> rs.next() ? Optional.of(rs.getTimestamp(1).toInstant()) : Optional.empty());
    }

    protected boolean isTableUniqueIndexIncluded(String indexName, String columnName) {
        if (columnName != null) {
            return !SYS_NC_PATTERN.matcher(columnName).matches() && !ADT_INDEX_NAMES_PATTERN.matcher(columnName).matches() && !MROW_PATTERN.matcher(columnName).matches();
        }
        return false;
    }

    public Scn getCurrentScn() throws SQLException {
        return (Scn)this.queryAndMap("SELECT CURRENT_SCN FROM V$DATABASE", rs -> {
            if (rs.next()) {
                return Scn.valueOf(rs.getString(1));
            }
            throw new IllegalStateException("Could not get SCN");
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getTableMetadataDdl(TableId tableId) throws SQLException, NonRelationalTableException {
        String string;
        try {
            String tableType = "SELECT COUNT(1) FROM ALL_ALL_TABLES WHERE OWNER='" + tableId.schema() + "' AND TABLE_NAME='" + tableId.table() + "' AND TABLE_TYPE IS NULL";
            if ((Integer)this.queryAndMap(tableType, rs -> rs.next() ? rs.getInt(1) : 0) == 0) {
                throw new NonRelationalTableException("Table " + tableId + " is not a relational table");
            }
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'STORAGE', false); end;"});
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'SEGMENT_ATTRIBUTES', false); end;"});
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'SQLTERMINATOR', true); end;"});
            string = (String)this.queryAndMap("SELECT dbms_metadata.get_ddl('TABLE','" + tableId.table() + "','" + tableId.schema() + "') FROM DUAL", rs -> {
                if (!rs.next()) {
                    throw new DebeziumException("Could not get DDL metadata for table: " + tableId);
                }
                Object res = rs.getObject(1);
                return ((Clob)res).getSubString(1L, (int)((Clob)res).length());
            });
        }
        catch (Throwable throwable) {
            this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'DEFAULT'); end;"});
            throw throwable;
        }
        this.executeWithoutCommitting(new String[]{"begin dbms_metadata.set_transform_param(DBMS_METADATA.SESSION_TRANSFORM, 'DEFAULT'); end;"});
        return string;
    }

    public Long getSessionStatisticByName(String name) throws SQLException {
        return (Long)this.queryAndMap("SELECT VALUE FROM v$statname n, v$mystat m WHERE n.name='" + name + "' AND n.statistic#=m.statistic#", rs -> rs.next() ? rs.getLong(1) : 0L);
    }

    public boolean isTableExists(String tableName) throws SQLException {
        return (Boolean)this.queryAndMap("SELECT COUNT(1) FROM USER_TABLES WHERE TABLE_NAME = '" + tableName + "'", rs -> rs.next() && rs.getLong(1) > 0L);
    }

    public boolean isTableExists(TableId tableId) throws SQLException {
        return (Boolean)this.queryAndMap("SELECT COUNT(1) FROM ALL_TABLES WHERE OWNER = '" + tableId.schema() + "' AND TABLE_NAME = '" + tableId.table() + "'", rs -> rs.next() && rs.getLong(1) > 0L);
    }

    public boolean isTableEmpty(String tableName) throws SQLException {
        return this.getRowCount(tableName) == 0L;
    }

    public long getRowCount(String tableName) throws SQLException {
        return (Long)this.queryAndMap("SELECT COUNT(1) FROM " + tableName, rs -> {
            if (rs.next()) {
                return rs.getLong(1);
            }
            return 0L;
        });
    }

    public <T> T singleOptionalValue(String query, JdbcConnection.ResultSetExtractor<T> extractor) throws SQLException {
        return (T)this.queryAndMap(query, rs -> rs.next() ? extractor.apply(rs) : null);
    }

    public String buildSelectWithRowLimits(TableId tableId, int limit, String projection, Optional<String> condition, Optional<String> additionalCondition, String orderBy) {
        TableId table = new TableId(null, tableId.schema(), tableId.table());
        StringBuilder sql = new StringBuilder("SELECT ");
        sql.append(projection).append(" FROM ");
        sql.append(this.quotedTableIdString(table));
        if (condition.isPresent()) {
            sql.append(" WHERE ").append(condition.get());
            if (additionalCondition.isPresent()) {
                sql.append(" AND ");
                sql.append(additionalCondition.get());
            }
        } else if (additionalCondition.isPresent()) {
            sql.append(" WHERE ");
            sql.append(additionalCondition.get());
        }
        if (this.getOracleVersion().getMajor() < 12) {
            sql.insert(0, " SELECT * FROM (").append(" ORDER BY ").append(orderBy).append(")").append(" WHERE ROWNUM <=").append(limit);
        } else {
            sql.append(" ORDER BY ").append(orderBy).append(" FETCH NEXT ").append(limit).append(" ROWS ONLY");
        }
        return sql.toString();
    }

    public static String connectionString(JdbcConfiguration config) {
        return config.getString(URL) != null ? config.getString(URL) : OracleConnectorConfig.ConnectorAdapter.parse(config.getString("connection.adapter")).getConnectionUrl();
    }

    private static JdbcConnection.ConnectionFactory resolveConnectionFactory(JdbcConfiguration config) {
        return JdbcConnection.patternBasedFactory((String)OracleConnection.connectionString(config), (Field[])new Field[0]);
    }

    protected boolean isArchiveLogMode() {
        try {
            String mode = (String)this.queryAndMap("SELECT LOG_MODE FROM V$DATABASE", rs -> rs.next() ? rs.getString(1) : "");
            LOGGER.debug("LOG_MODE={}", (Object)mode);
            return "ARCHIVELOG".equalsIgnoreCase(mode);
        }
        catch (SQLException e) {
            throw new DebeziumException("Unexpected error while connecting to Oracle and looking at LOG_MODE mode: ", (Throwable)e);
        }
    }

    public Optional<Instant> getScnToTimestamp(Scn scn) throws SQLException {
        try {
            return (Optional)this.queryAndMap("SELECT scn_to_timestamp('" + scn + "') FROM DUAL", rs -> rs.next() ? Optional.of(rs.getTimestamp(1).toInstant()) : Optional.empty());
        }
        catch (SQLException e) {
            if (e.getMessage().startsWith("ORA-08181")) {
                return Optional.empty();
            }
            throw e;
        }
    }

    public Scn getScnAdjustedByTime(Scn scn, Duration adjustment) throws SQLException {
        try {
            String result = (String)this.prepareQueryAndMap("SELECT timestamp_to_scn(scn_to_timestamp(?) - (? / 86400000)) FROM DUAL", st -> {
                st.setString(1, scn.toString());
                st.setLong(2, adjustment.toMillis());
            }, this.singleResultMapper(rs -> rs.getString(1), "Failed to get adjusted SCN from: " + scn));
            return Scn.valueOf(result);
        }
        catch (SQLException e) {
            if (e.getErrorCode() == 8181 || e.getErrorCode() == 8180) {
                return Scn.NULL;
            }
            throw e;
        }
    }

    public boolean isArchiveLogDestinationValid(String archiveDestinationName) throws SQLException {
        return (Boolean)this.prepareQueryAndMap("SELECT STATUS, TYPE FROM V$ARCHIVE_DEST_STATUS WHERE DEST_NAME=?", st -> st.setString(1, archiveDestinationName), rs -> {
            if (!rs.next()) {
                throw new DebeziumException(String.format("Archive log destination name '%s' is unknown to Oracle", archiveDestinationName));
            }
            return "VALID".equals(rs.getString("STATUS")) && "LOCAL".equals(rs.getString("TYPE"));
        });
    }

    public boolean isOnlyOneArchiveLogDestinationValid() throws SQLException {
        return (Boolean)this.queryAndMap("SELECT COUNT(1) FROM V$ARCHIVE_DEST_STATUS WHERE STATUS='VALID' AND TYPE='LOCAL'", rs -> {
            if (!rs.next()) {
                throw new DebeziumException("Unable to resolve number of archive log destinations");
            }
            return rs.getLong(1) == 1L;
        });
    }

    protected ColumnEditor overrideColumn(ColumnEditor column) {
        if (93 == column.jdbcType()) {
            column.length(column.scale().orElse(-1).intValue()).scale(null);
        } else if (2 == column.jdbcType()) {
            column.scale().filter(s -> s == -127).ifPresent(s -> column.scale(null));
        }
        return column;
    }

    protected Map<TableId, List<Column>> getColumnsDetails(String databaseCatalog, String schemaNamePattern, String tableName, Tables.TableFilter tableFilter, Tables.ColumnNameFilter columnFilter, DatabaseMetaData metadata, Set<TableId> viewIds) throws SQLException {
        if (tableName != null && tableName.contains("/")) {
            tableName = tableName.replace("/", "//");
        }
        return super.getColumnsDetails(databaseCatalog, schemaNamePattern, tableName, tableFilter, columnFilter, metadata, viewIds);
    }

    public Map<String, Object> reselectColumns(TableId tableId, List<String> columns, List<String> keyColumns, List<Object> keyValues, Struct source) throws SQLException {
        TableId oracleTableId = new TableId(null, tableId.schema(), tableId.table());
        String commitScn = source.getString("commit_scn");
        if (Strings.isNullOrEmpty((String)commitScn)) {
            return this.optionallyDoInContainer(() -> super.reselectColumns(oracleTableId, columns, keyColumns, keyValues, source));
        }
        String query = String.format("SELECT %s FROM (SELECT * FROM %s AS OF SCN ?) WHERE %s", columns.stream().map(arg_0 -> ((OracleConnection)this).quotedColumnIdString(arg_0)).collect(Collectors.joining(",")), this.quotedTableIdString(oracleTableId), keyColumns.stream().map(key -> key + "=?").collect(Collectors.joining(" AND ")));
        ArrayList<Object> bindValues = new ArrayList<Object>(keyValues.size() + 1);
        bindValues.add(commitScn);
        bindValues.addAll(keyValues);
        return this.optionallyDoInContainer(() -> {
            try {
                return this.reselectColumns(query, oracleTableId, columns, bindValues);
            }
            catch (SQLException e) {
                if (e.getErrorCode() == 1555 || e.getMessage().startsWith("ORA-01555")) {
                    LOGGER.warn("Failed to re-select row for table {} and key columns {} with values {}. Trying to perform re-selection without flashback.", new Object[]{tableId, keyColumns, keyValues});
                    return super.reselectColumns(oracleTableId, columns, keyColumns, keyValues, source);
                }
                throw e;
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T optionallyDoInContainer(ContainerWork<T> work) throws SQLException {
        boolean swapped = false;
        try {
            String pdbName = this.config().getString("pdb.name");
            if (!Strings.isNullOrEmpty((String)pdbName)) {
                this.setSessionToPdb(pdbName);
                swapped = true;
            }
            T t = work.execute();
            return t;
        }
        finally {
            if (swapped) {
                this.resetSessionToCdb();
            }
        }
    }

    public static class NonRelationalTableException
    extends DebeziumException {
        public NonRelationalTableException(String message) {
            super(message);
        }
    }

    @FunctionalInterface
    static interface ContainerWork<T> {
        public T execute() throws SQLException;
    }
}

