package coil.request

import android.graphics.Bitmap
import androidx.annotation.WorkerThread
import coil.ImageLoader
import coil.size.Dimension
import coil.size.Scale
import coil.size.Size
import coil.target.Target
import coil.target.ViewTarget
import coil.transform.Transformation
import coil.util.HardwareBitmapService
import coil.util.Logger
import coil.util.SystemCallbacks
import coil.util.VALID_TRANSFORMATION_CONFIGS
import coil.util.allowInexactSize
import coil.util.isHardware
import kotlinx.coroutines.Job

/** Handles operations that act on [ImageRequest]s. */
internal class RequestService(
    private val imageLoader: ImageLoader,
    private val systemCallbacks: SystemCallbacks,
    logger: Logger?
) {

    private val hardwareBitmapService = HardwareBitmapService(logger)

    /**
     * Wrap [initialRequest] to automatically dispose and/or restart the [ImageRequest]
     * based on its lifecycle.
     */
    fun requestDelegate(initialRequest: ImageRequest, job: Job): RequestDelegate {
        val lifecycle = initialRequest.lifecycle
        return when (val target = initialRequest.target) {
            is ViewTarget<*> ->
                ViewTargetRequestDelegate(imageLoader, initialRequest, target, lifecycle, job)
            else -> BaseRequestDelegate(lifecycle, job)
        }
    }

    fun errorResult(request: ImageRequest, throwable: Throwable): ErrorResult {
        return ErrorResult(
            drawable = if (throwable is NullRequestDataException) {
                request.fallback ?: request.error
            } else {
                request.error
            },
            request = request,
            throwable = throwable
        )
    }

    /**
     * Return the request options. The function is called from the main thread and must be fast.
     */
    fun options(request: ImageRequest, size: Size): Options {
        // Fall back to ARGB_8888 if the requested bitmap config does not pass the checks.
        val isValidConfig = isConfigValidForTransformations(request) &&
            isConfigValidForHardwareAllocation(request, size)
        val config = if (isValidConfig) request.bitmapConfig else Bitmap.Config.ARGB_8888

        // Use `Scale.FIT` if either dimension is undefined.
        val scale = if (size.width == Dimension.Undefined || size.height == Dimension.Undefined) {
            Scale.FIT
        } else {
            request.scale
        }

        // Disable allowRgb565 if there are transformations or the requested config is ALPHA_8.
        // ALPHA_8 is a mask config where each pixel is 1 byte so it wouldn't make sense to use
        // RGB_565 as an optimization in that case.
        val allowRgb565 = request.allowRgb565 &&
            request.transformations.isEmpty() &&
            config != Bitmap.Config.ALPHA_8

        return Options(
            context = request.context,
            config = config,
            colorSpace = request.colorSpace,
            size = size,
            scale = scale,
            allowInexactSize = request.allowInexactSize,
            allowRgb565 = allowRgb565,
            premultipliedAlpha = request.premultipliedAlpha,
            diskCacheKey = request.diskCacheKey,
            headers = request.headers,
            tags = request.tags,
            parameters = request.parameters,
            memoryCachePolicy = request.memoryCachePolicy,
            diskCachePolicy = request.diskCachePolicy,
            networkCachePolicy = request.networkCachePolicy,
        )
    }

    /**
     * Return 'true' if [requestedConfig] is a valid (i.e. can be returned to its [Target])
     * config for [request].
     */
    fun isConfigValidForHardware(request: ImageRequest, requestedConfig: Bitmap.Config): Boolean {
        // Short circuit if the requested bitmap config is software.
        if (!requestedConfig.isHardware) {
            return true
        }

        // Ensure the request allows hardware bitmaps.
        if (!request.allowHardware) {
            return false
        }

        // Prevent hardware bitmaps for non-hardware accelerated targets.
        val target = request.target
        if (target is ViewTarget<*> && target.view.run { isAttachedToWindow && !isHardwareAccelerated }) {
            return false
        }

        return true
    }

    fun updateOptionsOnWorkerThread(options: Options): Options {
        var changed = false
        var bitmapConfig = options.config
        var networkCachePolicy = options.networkCachePolicy

        if (!isBitmapConfigValidWorkerThread(options)) {
            bitmapConfig = Bitmap.Config.ARGB_8888
            changed = true
        }

        if (options.networkCachePolicy.readEnabled && !systemCallbacks.isOnline) {
            // Disable fetching from the network if we know we're offline.
            networkCachePolicy = CachePolicy.DISABLED
            changed = true
        }

        if (changed) {
            return options.copy(
                config = bitmapConfig,
                networkCachePolicy = networkCachePolicy,
            )
        } else {
            return options
        }
    }

    /** Return 'true' if we can allocate a hardware bitmap. */
    @WorkerThread
    private fun isBitmapConfigValidWorkerThread(options: Options): Boolean {
        return !options.config.isHardware || hardwareBitmapService.allowHardwareWorkerThread()
    }

    /**
     * Return 'true' if [request]'s requested bitmap config is valid (i.e. can be returned
     * to its [Target]).
     *
     * This check is similar to [isConfigValidForHardware] except this method also checks
     * that we are able to allocate a new hardware bitmap.
     */
    private fun isConfigValidForHardwareAllocation(request: ImageRequest, size: Size): Boolean {
        // Short circuit if the requested bitmap config is software.
        if (!request.bitmapConfig.isHardware) {
            return true
        }

        return isConfigValidForHardware(request, request.bitmapConfig) &&
            hardwareBitmapService.allowHardwareMainThread(size)
    }

    /** Return 'true' if [ImageRequest.bitmapConfig] is valid given its [Transformation]s. */
    private fun isConfigValidForTransformations(request: ImageRequest): Boolean {
        return request.transformations.isEmpty() ||
            request.bitmapConfig in VALID_TRANSFORMATION_CONFIGS
    }
}
