/*
 * Decompiled with CFR 0.152.
 */
package io.clientcore.core.http.pipeline;

import io.clientcore.core.http.models.HttpHeaderName;
import io.clientcore.core.http.models.HttpHeaders;
import io.clientcore.core.http.models.HttpRequest;
import io.clientcore.core.http.models.Response;
import io.clientcore.core.http.pipeline.HttpPipelineNextPolicy;
import io.clientcore.core.http.pipeline.HttpPipelinePolicy;
import io.clientcore.core.http.pipeline.HttpPipelinePosition;
import io.clientcore.core.http.pipeline.HttpRetryCondition;
import io.clientcore.core.http.pipeline.HttpRetryOptions;
import io.clientcore.core.implementation.http.HttpRequestAccessHelper;
import io.clientcore.core.implementation.http.RetryUtils;
import io.clientcore.core.instrumentation.InstrumentationContext;
import io.clientcore.core.instrumentation.logging.ClientLogger;
import io.clientcore.core.instrumentation.logging.LoggingEvent;
import io.clientcore.core.models.binarydata.BinaryData;
import io.clientcore.core.utils.CoreUtils;
import io.clientcore.core.utils.DateTimeRfc1123;
import io.clientcore.core.utils.configuration.Configuration;
import java.io.IOException;
import java.time.DateTimeException;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

public final class HttpRetryPolicy
implements HttpPipelinePolicy {
    private static final ClientLogger LOGGER = new ClientLogger(HttpRetryPolicy.class);
    private final int maxRetries;
    private final Function<HttpHeaders, Duration> delayFromHeaders;
    private final Duration baseDelay;
    private final Duration maxDelay;
    private final Duration fixedDelay;
    private final Predicate<HttpRetryCondition> shouldRetryCondition;
    private static final int DEFAULT_MAX_RETRIES;
    private static final Duration DEFAULT_BASE_DELAY;
    private static final Duration DEFAULT_MAX_DELAY;
    private static final double JITTER_FACTOR = 0.05;
    private static final HttpHeaderName RETRY_AFTER_MS_HEADER;
    private static final HttpHeaderName X_MS_RETRY_AFTER_MS_HEADER;

    public HttpRetryPolicy() {
        this(DEFAULT_BASE_DELAY, DEFAULT_MAX_DELAY, null, DEFAULT_MAX_RETRIES, null, null);
    }

    public HttpRetryPolicy(HttpRetryOptions retryOptions) {
        this(retryOptions.getBaseDelay(), retryOptions.getMaxDelay(), retryOptions.getFixedDelay(), retryOptions.getMaxRetries(), retryOptions.getDelayFromHeaders(), retryOptions.getShouldRetryCondition());
    }

    HttpRetryPolicy(Duration baseDelay, Duration maxDelay, Duration fixedDelay, int maxRetries, Function<HttpHeaders, Duration> delayFromHeaders, Predicate<HttpRetryCondition> shouldRetryCondition) {
        if (fixedDelay == null && baseDelay == null) {
            this.baseDelay = DEFAULT_BASE_DELAY;
            this.maxDelay = DEFAULT_MAX_DELAY;
        } else {
            this.baseDelay = baseDelay;
            this.maxDelay = maxDelay;
        }
        this.fixedDelay = fixedDelay;
        this.maxRetries = maxRetries;
        this.delayFromHeaders = delayFromHeaders;
        this.shouldRetryCondition = shouldRetryCondition;
    }

    @Override
    public Response<BinaryData> process(HttpRequest httpRequest, HttpPipelineNextPolicy next) {
        return this.attempt(httpRequest, next, 0, null);
    }

    @Override
    public HttpPipelinePosition getPipelinePosition() {
        return HttpPipelinePosition.RETRY;
    }

    private Duration getWellKnownRetryDelay(HttpHeaders responseHeaders, int tryCount, Supplier<OffsetDateTime> nowSupplier) {
        Duration retryDelay = HttpRetryPolicy.getRetryAfterFromHeaders(responseHeaders, nowSupplier);
        if (retryDelay != null) {
            return retryDelay;
        }
        return this.calculateRetryDelay(tryCount);
    }

    private Response<BinaryData> attempt(HttpRequest httpRequest, HttpPipelineNextPolicy next, int tryCount, List<Exception> suppressed) {
        Response<BinaryData> response;
        HttpRequestAccessHelper.setTryCount(httpRequest, tryCount);
        InstrumentationContext instrumentationContext = httpRequest.getContext().getInstrumentationContext();
        ClientLogger logger = this.getLogger(httpRequest);
        try {
            response = next.copy().process();
        }
        catch (RuntimeException err) {
            if (this.shouldRetryException(err, tryCount, suppressed)) {
                Duration delayDuration = this.calculateRetryDelay(tryCount);
                this.logRetry(logger.atVerbose(), tryCount, delayDuration, err, false, instrumentationContext);
                boolean interrupted = false;
                long millis = delayDuration.toMillis();
                if (millis > 0L) {
                    try {
                        Thread.sleep(millis);
                    }
                    catch (InterruptedException ie) {
                        interrupted = true;
                        err.addSuppressed(ie);
                    }
                }
                if (interrupted) {
                    throw logger.logThrowableAsError(err);
                }
                LinkedList<Exception> suppressedLocal = suppressed == null ? new LinkedList<Exception>() : suppressed;
                suppressedLocal.add(err);
                return this.attempt(httpRequest, next, tryCount + 1, suppressedLocal);
            }
            this.logRetry(logger.atWarning(), tryCount, null, err, true, instrumentationContext);
            if (suppressed != null) {
                suppressed.forEach(err::addSuppressed);
            }
            throw logger.logThrowableAsError(err);
        }
        if (this.shouldRetryResponse(response, tryCount, suppressed)) {
            Duration delayDuration = this.determineDelayDuration(response, tryCount, this.delayFromHeaders);
            this.logRetry(logger.atVerbose(), tryCount, delayDuration, null, false, instrumentationContext);
            response.close();
            long millis = delayDuration.toMillis();
            if (millis > 0L) {
                try {
                    Thread.sleep(millis);
                }
                catch (InterruptedException ie) {
                    throw LOGGER.logThrowableAsError(new RuntimeException(ie));
                }
            }
            return this.attempt(httpRequest, next, tryCount + 1, suppressed);
        }
        if (tryCount >= this.maxRetries) {
            this.logRetry(logger.atWarning(), tryCount, null, null, true, instrumentationContext);
        }
        return response;
    }

    private Duration determineDelayDuration(Response<BinaryData> response, int tryCount, Function<HttpHeaders, Duration> delayFromHeaders) {
        if (delayFromHeaders == null) {
            return this.getWellKnownRetryDelay(response.getHeaders(), tryCount, OffsetDateTime::now);
        }
        Duration delay = delayFromHeaders.apply(response.getHeaders());
        if (delay != null) {
            return delay;
        }
        return this.calculateRetryDelay(tryCount);
    }

    private boolean shouldRetryResponse(Response<BinaryData> response, int tryCount, List<Exception> retriedExceptions) {
        if (this.shouldRetryCondition != null) {
            return tryCount < this.maxRetries && this.shouldRetryCondition.test(new HttpRetryCondition(response, null, tryCount, retriedExceptions));
        }
        return tryCount < this.maxRetries && this.defaultShouldRetryCondition(new HttpRetryCondition(response, null, tryCount, retriedExceptions));
    }

    private boolean shouldRetryException(Exception exception, int tryCount, List<Exception> retriedExceptions) {
        if (tryCount >= this.maxRetries) {
            return false;
        }
        Throwable causalThrowable = exception.getCause();
        HttpRetryCondition requestRetryCondition = new HttpRetryCondition(null, exception, tryCount, retriedExceptions);
        while (causalThrowable instanceof IOException || causalThrowable instanceof TimeoutException) {
            if (this.shouldRetryCondition != null) {
                if (this.shouldRetryCondition.test(requestRetryCondition)) {
                    return true;
                }
            } else {
                return this.defaultShouldRetryCondition(requestRetryCondition);
            }
            causalThrowable = causalThrowable.getCause();
        }
        return false;
    }

    private void logRetry(LoggingEvent log, int tryCount, Duration delayDuration, Throwable throwable, boolean lastTry, InstrumentationContext context) {
        if (log.isEnabled()) {
            log.addKeyValue("http.request.resend_count", tryCount).addKeyValue("retry.max_attempt_count", this.maxRetries).addKeyValue("retry.was_last_attempt", lastTry).setEventName("http.retry").setInstrumentationContext(context).setThrowable(throwable);
            if (delayDuration != null) {
                log.addKeyValue("retry.delay", delayDuration.toMillis());
            }
            log.log();
        }
    }

    private Duration calculateRetryDelay(int retryAttempts) {
        if (this.fixedDelay != null) {
            return this.fixedDelay;
        }
        long baseDelayNanos = this.baseDelay.toNanos();
        long maxDelayNanos = this.maxDelay.toNanos();
        long delayWithJitterInNanos = ThreadLocalRandom.current().nextLong((long)((double)baseDelayNanos * 0.95), (long)((double)baseDelayNanos * 1.05));
        return Duration.ofNanos(Math.min((1L << retryAttempts) * delayWithJitterInNanos, maxDelayNanos));
    }

    private boolean defaultShouldRetryCondition(HttpRetryCondition requestRetryCondition) {
        if (requestRetryCondition.getResponse() != null) {
            return RetryUtils.isRetryable(requestRetryCondition.getResponse().getStatusCode());
        }
        return RetryUtils.isRetryable(requestRetryCondition.getException());
    }

    private ClientLogger getLogger(HttpRequest httpRequest) {
        ClientLogger logger = null;
        if (httpRequest.getContext() != null && httpRequest.getContext().getLogger() != null) {
            logger = httpRequest.getContext().getLogger();
        }
        return logger == null ? LOGGER : logger;
    }

    private static Duration getRetryAfterFromHeaders(HttpHeaders headers, Supplier<OffsetDateTime> nowSupplier) {
        Duration retryDelay = HttpRetryPolicy.tryGetRetryDelay(headers, X_MS_RETRY_AFTER_MS_HEADER, HttpRetryPolicy::tryGetDelayMillis);
        if (retryDelay != null) {
            return retryDelay;
        }
        retryDelay = HttpRetryPolicy.tryGetRetryDelay(headers, RETRY_AFTER_MS_HEADER, HttpRetryPolicy::tryGetDelayMillis);
        if (retryDelay != null) {
            return retryDelay;
        }
        retryDelay = HttpRetryPolicy.tryGetRetryDelay(headers, HttpHeaderName.RETRY_AFTER, headerValue -> HttpRetryPolicy.tryParseLongOrDateTime(headerValue, nowSupplier));
        return retryDelay;
    }

    private static Duration tryGetRetryDelay(HttpHeaders headers, HttpHeaderName headerName, Function<String, Duration> delayParser) {
        String headerValue = headers.getValue(headerName);
        return CoreUtils.isNullOrEmpty(headerValue) ? null : delayParser.apply(headerValue);
    }

    private static Duration tryGetDelayMillis(String value) {
        long delayMillis = HttpRetryPolicy.tryParseLong(value);
        return delayMillis >= 0L ? Duration.ofMillis(delayMillis) : null;
    }

    private static Duration tryParseLongOrDateTime(String value, Supplier<OffsetDateTime> nowSupplier) {
        long delaySeconds;
        try {
            OffsetDateTime retryAfter = new DateTimeRfc1123(value).getDateTime();
            delaySeconds = nowSupplier.get().until(retryAfter, ChronoUnit.SECONDS);
        }
        catch (DateTimeException ex) {
            delaySeconds = HttpRetryPolicy.tryParseLong(value);
        }
        return delaySeconds >= 0L ? Duration.ofSeconds(delaySeconds) : null;
    }

    private static long tryParseLong(String value) {
        try {
            return Long.parseLong(value);
        }
        catch (NumberFormatException ex) {
            return -1L;
        }
    }

    static {
        DEFAULT_BASE_DELAY = Duration.ofMillis(800L);
        DEFAULT_MAX_DELAY = Duration.ofSeconds(8L);
        String envDefaultMaxRetries = Configuration.getGlobalConfiguration().get("MAX_RETRY_ATTEMPTS");
        int defaultMaxRetries = 3;
        if (!CoreUtils.isNullOrEmpty(envDefaultMaxRetries)) {
            try {
                defaultMaxRetries = Integer.parseInt(envDefaultMaxRetries);
                if (defaultMaxRetries < 0) {
                    defaultMaxRetries = 3;
                }
            }
            catch (NumberFormatException ignored) {
                LOGGER.atVerbose().addKeyValue("property", "MAX_RETRY_ATTEMPTS").log("Invalid property value. Using 3 retries as the maximum.");
            }
        }
        DEFAULT_MAX_RETRIES = defaultMaxRetries;
        RETRY_AFTER_MS_HEADER = HttpHeaderName.fromString("retry-after-ms");
        X_MS_RETRY_AFTER_MS_HEADER = HttpHeaderName.fromString("x-ms-retry-after-ms");
    }
}

