/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.scheduler.startup;

import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.startup.StartupProcess;
import io.camunda.zeebe.scheduler.startup.StartupProcessException;
import io.camunda.zeebe.scheduler.startup.StartupStep;
import io.camunda.zeebe.scheduler.testing.TestActorFuture;
import io.camunda.zeebe.scheduler.testing.TestConcurrencyControl;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.awaitility.Awaitility;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.mockito.InOrder;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class StartupProcessTest {
    private static final Logger LOGGER = LoggerFactory.getLogger(StartupProcessTest.class);
    private static final Object STARTUP_CONTEXT = new Object(){

        public String toString() {
            return "startupContext";
        }
    };
    private static final Object SHUTDOWN_CONTEXT = new Object(){

        public String toString() {
            return "shutdownContext";
        }
    };
    private static final ConcurrencyControl TEST_CONCURRENCY_CONTROL = new TestConcurrencyControl();

    StartupProcessTest() {
    }

    private final class InvocationCountingStartupStep
    implements StartupStep<Object> {
        private int startupInvocationCounter = 0;
        private int shutdownInvocationCounter = 0;

        private InvocationCountingStartupStep(StartupProcessTest startupProcessTest) {
        }

        int getStartupInvocationCounter() {
            return this.startupInvocationCounter;
        }

        int getShutdownInvocationCounter() {
            return this.shutdownInvocationCounter;
        }

        public String getName() {
            return "InvocationCountingStartupStep";
        }

        public ActorFuture<Object> startup(Object o) {
            ++this.startupInvocationCounter;
            return TestActorFuture.completedFuture(o);
        }

        public ActorFuture<Object> shutdown(Object o) {
            ++this.shutdownInvocationCounter;
            return TestActorFuture.completedFuture(o);
        }
    }

    private final class WaitingStartupStep
    implements StartupStep<Object> {
        private final CountDownLatch startupCountdownLatch;
        private final boolean completeWithException;

        WaitingStartupStep(StartupProcessTest startupProcessTest, CountDownLatch startupCountdownLatch, boolean completeWithException) {
            this.startupCountdownLatch = startupCountdownLatch;
            this.completeWithException = completeWithException;
        }

        public String getName() {
            return "WaitingStartupStep";
        }

        public ActorFuture<Object> startup(Object o) {
            TestActorFuture<Object> startupFuture = new TestActorFuture<Object>();
            Thread startupThread = new Thread(() -> {
                try {
                    this.startupCountdownLatch.await();
                }
                catch (InterruptedException e) {
                    LOGGER.error(e.getMessage(), (Throwable)e);
                }
                finally {
                    if (!this.completeWithException) {
                        startupFuture.complete(o);
                    } else {
                        startupFuture.completeExceptionally(new Throwable("completed exceptionally"));
                    }
                }
            });
            startupThread.start();
            return startupFuture;
        }

        public ActorFuture<Object> shutdown(Object o) {
            return TestActorFuture.completedFuture(o);
        }
    }

    @Nested
    class EmptyList {
        private final StartupProcess<Object> sut = new StartupProcess(Collections.emptyList());

        EmptyList(StartupProcessTest this$0) {
        }

        @Test
        void shouldReturnContextImmediatelyOnStartup() {
            ActorFuture startupFuture = this.sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT);
            Assertions.assertThat((Object)startupFuture.join()).isSameAs(STARTUP_CONTEXT);
        }

        @Test
        void shouldReturnContextImmediatelyOnShutdown() {
            this.sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT).join();
            ActorFuture shutdownFuture = this.sut.shutdown(TEST_CONCURRENCY_CONTROL, SHUTDOWN_CONTEXT);
            Assertions.assertThat((Object)shutdownFuture.join()).isSameAs(SHUTDOWN_CONTEXT);
        }
    }

    @Nested
    class IllegalStatesAndArguments {
        IllegalStatesAndArguments() {
        }

        @Test
        void shouldThrowNPEWhenCalledWithNoSteps() {
            Assertions.assertThatThrownBy(() -> new StartupProcess(null)).isInstanceOf(NullPointerException.class);
        }

        @Test
        void shouldThrowNPEWhenCalledWithNoLogger() {
            Assertions.assertThatThrownBy(() -> new StartupProcess(null, Collections.emptyList())).isInstanceOf(NullPointerException.class);
        }

        @Test
        void shouldThrowIllegalStateIfStartupIsCalledMoreThanOnce() {
            StartupProcess sut = new StartupProcess(Collections.emptyList());
            sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT).join();
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT)).isInstanceOf(IllegalStateException.class)).hasMessage("startup(...) must only be called once");
        }

        @Test
        void shouldPerformShutdownOnlyOnceIfShutdownIsCalledMultipleTimes() {
            InvocationCountingStartupStep step = new InvocationCountingStartupStep(StartupProcessTest.this);
            StartupProcess sut = new StartupProcess(Collections.singletonList(step));
            sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT).join();
            sut.shutdown(TEST_CONCURRENCY_CONTROL, SHUTDOWN_CONTEXT).join();
            sut.shutdown(TEST_CONCURRENCY_CONTROL, SHUTDOWN_CONTEXT).join();
            Assertions.assertThat((int)step.getShutdownInvocationCounter()).isEqualTo(1);
        }
    }

    @Nested
    class MainUseCase {
        private static final String INPUT_STEP1 = "inputStep1";
        private static final String INPUT_STEP2 = "inputStep2";
        private static final String RESULT_STEP1 = "resultStep1";
        private static final String RESULT_STEP2 = "resultStep2";
        private final Exception testException1 = new Exception("TEST_EXCEPTION1");
        private final Exception testException2 = new Exception("TEST_EXCEPTION1");
        private StartupStep mockStep1;
        private StartupStep mockStep2;

        MainUseCase() {
        }

        @BeforeEach
        void setup() {
            this.mockStep1 = (StartupStep)Mockito.mock(StartupStep.class);
            this.mockStep2 = (StartupStep)Mockito.mock(StartupStep.class);
            Mockito.when((Object)this.mockStep1.getName()).thenReturn((Object)"step1");
            Mockito.when((Object)this.mockStep2.getName()).thenReturn((Object)"step2");
        }

        @Test
        void shouldCallStartupStepsInOrder() {
            Mockito.when((Object)this.mockStep1.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            Mockito.when((Object)this.mockStep2.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            StartupProcess sut = new StartupProcess(List.of(this.mockStep1, this.mockStep2));
            sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT).join();
            InOrder invocationRecorder = Mockito.inOrder((Object[])new Object[]{this.mockStep1, this.mockStep2});
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep1)).startup(STARTUP_CONTEXT);
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep2)).startup(STARTUP_CONTEXT);
        }

        @Test
        void shouldCallShutdownStepsInReverseOrder() {
            Mockito.when((Object)this.mockStep1.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            Mockito.when((Object)this.mockStep1.shutdown(SHUTDOWN_CONTEXT)).thenReturn(TestActorFuture.completedFuture(SHUTDOWN_CONTEXT));
            Mockito.when((Object)this.mockStep2.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            Mockito.when((Object)this.mockStep2.shutdown(SHUTDOWN_CONTEXT)).thenReturn(TestActorFuture.completedFuture(SHUTDOWN_CONTEXT));
            StartupProcess sut = new StartupProcess(List.of(this.mockStep1, this.mockStep2));
            sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT).join();
            sut.shutdown(TEST_CONCURRENCY_CONTROL, SHUTDOWN_CONTEXT).join();
            InOrder invocationRecorder = Mockito.inOrder((Object[])new Object[]{this.mockStep1, this.mockStep2});
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep2)).shutdown(SHUTDOWN_CONTEXT);
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep1)).shutdown(SHUTDOWN_CONTEXT);
        }

        @Test
        void shouldCallSubsequentStartupStepWithResultOfPreviousStep() {
            Mockito.when((Object)this.mockStep1.startup((Object)INPUT_STEP1)).thenReturn(TestActorFuture.completedFuture(RESULT_STEP1));
            Mockito.when((Object)this.mockStep2.startup((Object)RESULT_STEP1)).thenReturn(TestActorFuture.completedFuture(RESULT_STEP2));
            StartupProcess sut = new StartupProcess(List.of(this.mockStep1, this.mockStep2));
            Object actualResult = sut.startup(TEST_CONCURRENCY_CONTROL, (Object)INPUT_STEP1).join();
            InOrder invocationRecorder = Mockito.inOrder((Object[])new Object[]{this.mockStep1, this.mockStep2});
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep1)).startup((Object)INPUT_STEP1);
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep2)).startup((Object)RESULT_STEP1);
            Assertions.assertThat((Object)actualResult).isSameAs((Object)RESULT_STEP2);
        }

        @Test
        void shouldCallSubsequentShutdownStepWithResultOfPreviousStep() {
            Mockito.when((Object)this.mockStep1.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            Mockito.when((Object)this.mockStep2.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            Mockito.when((Object)this.mockStep2.shutdown((Object)INPUT_STEP2)).thenReturn(TestActorFuture.completedFuture(RESULT_STEP2));
            Mockito.when((Object)this.mockStep1.shutdown((Object)RESULT_STEP2)).thenReturn(TestActorFuture.completedFuture(RESULT_STEP1));
            StartupProcess sut = new StartupProcess(List.of(this.mockStep1, this.mockStep2));
            sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT).join();
            Object actualResult = sut.shutdown(TEST_CONCURRENCY_CONTROL, (Object)INPUT_STEP2).join();
            InOrder invocationRecorder = Mockito.inOrder((Object[])new Object[]{this.mockStep1, this.mockStep2});
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep2)).shutdown((Object)INPUT_STEP2);
            ((StartupStep)invocationRecorder.verify((Object)this.mockStep1)).shutdown((Object)RESULT_STEP2);
            Assertions.assertThat((Object)actualResult).isSameAs((Object)RESULT_STEP1);
        }

        @Test
        void shouldAbortStartupIfOneStepThrewAnException() {
            Exception testException = new Exception("TEST_EXCEPTION");
            Mockito.when((Object)this.mockStep1.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.failedFuture(testException));
            Mockito.when((Object)this.mockStep2.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            StartupProcess sut = new StartupProcess(List.of(this.mockStep1, this.mockStep2));
            ActorFuture actualResult = sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT);
            ((StartupStep)Mockito.verify((Object)this.mockStep2, (VerificationMode)Mockito.never())).startup(STARTUP_CONTEXT);
            Assertions.assertThat((boolean)actualResult.isCompletedExceptionally()).isTrue();
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ActorFuture)actualResult).join()).isInstanceOf(ExecutionException.class)).cause().isInstanceOf(StartupProcessException.class);
        }

        @Test
        void shouldContinueShutdownEvenIfStepsThrowExceptions() {
            Mockito.when((Object)this.mockStep1.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            Mockito.when((Object)this.mockStep2.startup(STARTUP_CONTEXT)).thenReturn(TestActorFuture.completedFuture(STARTUP_CONTEXT));
            Mockito.when((Object)this.mockStep1.shutdown(SHUTDOWN_CONTEXT)).thenReturn(TestActorFuture.failedFuture(this.testException1));
            Mockito.when((Object)this.mockStep2.shutdown(SHUTDOWN_CONTEXT)).thenReturn(TestActorFuture.failedFuture(this.testException2));
            StartupProcess sut = new StartupProcess(List.of(this.mockStep1, this.mockStep2));
            sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT).join();
            ActorFuture actualResult = sut.shutdown(TEST_CONCURRENCY_CONTROL, SHUTDOWN_CONTEXT);
            ((StartupStep)Mockito.verify((Object)this.mockStep2)).shutdown(SHUTDOWN_CONTEXT);
            ((StartupStep)Mockito.verify((Object)this.mockStep1)).shutdown(SHUTDOWN_CONTEXT);
            Assertions.assertThat((boolean)actualResult.isCompletedExceptionally()).isTrue();
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ActorFuture)actualResult).join()).isInstanceOf(ExecutionException.class)).cause().isInstanceOf(StartupProcessException.class);
        }

        @Test
        void shouldAbortOngoingStartupWhenShutdownIsCalled() {
            CountDownLatch step1CountdownLatch = new CountDownLatch(1);
            WaitingStartupStep step1 = new WaitingStartupStep(StartupProcessTest.this, step1CountdownLatch, false);
            StartupProcess sut = new StartupProcess(List.of(step1, this.mockStep2));
            ActorFuture startupFuture = sut.startup(TEST_CONCURRENCY_CONTROL, STARTUP_CONTEXT);
            ActorFuture shutdownFuture = sut.shutdown(TEST_CONCURRENCY_CONTROL, SHUTDOWN_CONTEXT);
            step1CountdownLatch.countDown();
            Mockito.verifyNoInteractions((Object[])new Object[]{this.mockStep2});
            Awaitility.await().until(() -> startupFuture.isDone());
            Awaitility.await().until(() -> shutdownFuture.isDone());
            Assertions.assertThat((boolean)startupFuture.isCompletedExceptionally()).isTrue();
            Assertions.assertThat((Object)shutdownFuture.join()).isSameAs(SHUTDOWN_CONTEXT);
        }
    }
}

