/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.scheduler.retry;

import io.camunda.zeebe.scheduler.Actor;
import io.camunda.zeebe.scheduler.ActorControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.retry.AbortableRetryStrategy;
import io.camunda.zeebe.scheduler.retry.BackOffRetryStrategy;
import io.camunda.zeebe.scheduler.retry.EndlessRetryStrategy;
import io.camunda.zeebe.scheduler.retry.RecoverableRetryStrategy;
import io.camunda.zeebe.scheduler.retry.RetryStrategy;
import io.camunda.zeebe.scheduler.testing.ControlledActorSchedulerExtension;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.CompletableFutureAssert;
import org.assertj.core.api.FutureAssert;
import org.awaitility.Awaitility;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

final class RetryStrategyTest {
    @RegisterExtension
    private final ControlledActorSchedulerExtension schedulerRule = new ControlledActorSchedulerExtension(builder -> builder.setIoBoundActorThreadCount(0).setCpuBoundActorThreadCount(1));
    private ActorFuture<Boolean> resultFuture;

    RetryStrategyTest() {
    }

    @ParameterizedTest
    @ValueSource(strings={"endless", "recoverable", "abortable", "backoff"})
    void shouldRunUntilDone(TestCase<?> test) {
        AtomicInteger count = new AtomicInteger(0);
        this.schedulerRule.submitActor(test.actor);
        test.actor.run(() -> {
            this.resultFuture = test.strategy.runWithRetry(() -> count.incrementAndGet() == 10);
        });
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((int)count.get()).isEqualTo(10);
        Assertions.assertThat(this.resultFuture).succeedsWithin(Duration.ZERO).isEqualTo((Object)true);
    }

    @ParameterizedTest
    @ValueSource(strings={"endless", "recoverable", "abortable", "backoff"})
    void shouldStopWhenAbortConditionReturnsTrue(TestCase<?> test) {
        AtomicInteger count = new AtomicInteger(0);
        this.schedulerRule.submitActor(test.actor);
        test.actor.run(() -> {
            this.resultFuture = test.strategy.runWithRetry(() -> false, () -> count.incrementAndGet() == 10);
        });
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((int)count.get()).isEqualTo(10);
        Assertions.assertThat(this.resultFuture).succeedsWithin(Duration.ZERO).isEqualTo((Object)false);
    }

    @ParameterizedTest
    @ValueSource(strings={"recoverable", "abortable"})
    void shouldAbortOnOtherException(TestCase<?> test) {
        RuntimeException failure = new RuntimeException("expected");
        this.schedulerRule.submitActor(test.actor);
        test.actor.run(() -> {
            this.resultFuture = test.strategy.runWithRetry(() -> {
                throw failure;
            });
        });
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(this.resultFuture).failsWithin(Duration.ZERO).withThrowableOfType(ExecutionException.class).withCause((Throwable)failure);
    }

    @ParameterizedTest
    @ValueSource(strings={"endless", "recoverable", "abortable"})
    void shouldNotInterleaveRetry(TestCase<?> test) {
        AtomicReference firstFuture = new AtomicReference();
        AtomicReference secondFuture = new AtomicReference();
        AtomicInteger executionAttempt = new AtomicInteger(0);
        AtomicInteger firstResult = new AtomicInteger();
        AtomicInteger secondResult = new AtomicInteger();
        this.schedulerRule.submitActor(test.actor);
        int retryCounts = 5;
        test.actor.run(() -> firstFuture.set(test.strategy.runWithRetry(() -> {
            firstResult.set(executionAttempt.getAndIncrement());
            return executionAttempt.get() >= 5;
        })));
        test.actor.run(() -> secondFuture.set(test.strategy.runWithRetry(() -> {
            secondResult.set(executionAttempt.getAndIncrement());
            return true;
        })));
        this.schedulerRule.workUntilDone();
        ((FutureAssert)Assertions.assertThat((Future)((Future)firstFuture.get())).isDone()).isNotEqualTo(secondFuture.get());
        Assertions.assertThat((Future)((Future)secondFuture.get())).isDone();
        Assertions.assertThat((AtomicInteger)firstResult).hasValue(4);
        Assertions.assertThat((AtomicInteger)secondResult).hasValue(5);
    }

    @ParameterizedTest
    @ValueSource(strings={"endless", "recoverable", "abortable"})
    void shouldYieldThreadOnRetry(TestCase<?> test) {
        LinkedTransferQueue barrier = new LinkedTransferQueue();
        CompletableFuture future = new CompletableFuture();
        ControllableActor secondActor = new ControllableActor();
        this.schedulerRule.submitActor(test.actor);
        this.schedulerRule.submitActor(secondActor);
        test.strategy.runWithRetry(() -> {
            boolean isDone = future.isDone();
            barrier.offer(true);
            return isDone;
        });
        secondActor.run(() -> {
            try {
                barrier.poll(1L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            future.complete(null);
        });
        Awaitility.await((String)("workUntilDone should be finite if each actor yields the thread, used retry strategy " + test.strategy.getClass().getName())).atMost(Duration.ofSeconds(30L)).untilAsserted(this.schedulerRule::workUntilDone);
        ((CompletableFutureAssert)Assertions.assertThat(future).as("future is completed iff second actor can run", new Object[0])).succeedsWithin(Duration.ofSeconds(2L));
    }

    private record TestCase<T extends RetryStrategy>(ControllableActor actor, T strategy) {
        static TestCase<?> of(String type) {
            return switch (type.toLowerCase()) {
                case "endless" -> TestCase.of(EndlessRetryStrategy::new);
                case "recoverable" -> TestCase.of(RecoverableRetryStrategy::new);
                case "abortable" -> TestCase.of(AbortableRetryStrategy::new);
                case "backoff" -> TestCase.of((ActorControl actor) -> new BackOffRetryStrategy(actor, Duration.ZERO));
                default -> throw new IllegalArgumentException("Expected one of ['endless', 'recoverable', 'abortable', or 'backoff'], but got " + type);
            };
        }

        private static <T extends RetryStrategy> TestCase<T> of(Function<ActorControl, T> provider) {
            ControllableActor actor = new ControllableActor();
            RetryStrategy strategy = (RetryStrategy)provider.apply(actor.getActor());
            return new TestCase<RetryStrategy>(actor, strategy);
        }
    }

    private static final class ControllableActor
    extends Actor {
        private ControllableActor() {
        }

        public ActorControl getActor() {
            return this.actor;
        }
    }
}

