/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.scheduler.lifecycle;

import io.camunda.zeebe.scheduler.ActorTask;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.future.CompletableActorFuture;
import io.camunda.zeebe.scheduler.lifecycle.LifecycleRecordingActor;
import io.camunda.zeebe.scheduler.testing.ControlledActorSchedulerRule;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.assertj.core.api.Assertions;
import org.assertj.core.util.Lists;
import org.awaitility.Awaitility;
import org.junit.Rule;
import org.junit.Test;

public final class ActorLifecyclePhasesTest {
    @Rule
    public final ControlledActorSchedulerRule schedulerRule = new ControlledActorSchedulerRule();

    @Test
    public void shouldStartActor() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor();
        ActorFuture<Void> startedFuture = this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(startedFuture).isDone();
        Assertions.assertThat(actor.phases).isEqualTo((Object)Lists.newArrayList((Object[])new ActorTask.ActorLifecyclePhase[]{ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED}));
    }

    @Test
    public void shouldCloseActor() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor();
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((Future)closeFuture).isDone();
        Assertions.assertThat(actor.phases).isEqualTo(LifecycleRecordingActor.FULL_LIFECYCLE);
    }

    @Test
    public void shouldDoFullLifecycleIfClosedConcurrently() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor();
        this.schedulerRule.submitActor(actor);
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((Future)closeFuture).isDone();
        Assertions.assertThat(actor.phases).isEqualTo(LifecycleRecordingActor.FULL_LIFECYCLE);
    }

    @Test
    public void shouldCloseOnFailureWhileActorStarting() {
        final RuntimeException failure = new RuntimeException("foo");
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarting() {
                super.onActorStarting();
                throw failure;
            }
        };
        ActorFuture<Void> startedFuture = this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((boolean)startedFuture.isCompletedExceptionally()).isTrue();
        Assertions.assertThat((Throwable)startedFuture.getException()).isEqualTo((Object)failure);
        Assertions.assertThat(actor.phases).isEqualTo((Object)Lists.newArrayList((Object[])new ActorTask.ActorLifecyclePhase[]{ActorTask.ActorLifecyclePhase.STARTING}));
    }

    @Test
    public void shouldCloseOnFailureWhileActorClosing() {
        final RuntimeException failure = new RuntimeException("foo");
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorClosing() {
                super.onActorClosing();
                throw failure;
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((boolean)closeFuture.isCompletedExceptionally()).isTrue();
        Assertions.assertThat((Throwable)closeFuture.getException()).isEqualTo((Object)failure);
        Assertions.assertThat(actor.phases).isEqualTo((Object)Lists.newArrayList((Object[])new ActorTask.ActorLifecyclePhase[]{ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.CLOSE_REQUESTED, ActorTask.ActorLifecyclePhase.CLOSING}));
    }

    @Test
    public void shouldPropagateFailureWhileActorStartingAndRun() {
        final RuntimeException failure = new RuntimeException("foo");
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarting() {
                super.onActorStarting();
                this.actor.run(() -> {
                    throw failure;
                });
            }
        };
        ActorFuture<Void> startedFuture = this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((boolean)startedFuture.isCompletedExceptionally()).isTrue();
        Assertions.assertThat((Throwable)startedFuture.getException()).isEqualTo((Object)failure);
        Assertions.assertThat(actor.phases).isEqualTo((Object)Lists.newArrayList((Object[])new ActorTask.ActorLifecyclePhase[]{ActorTask.ActorLifecyclePhase.STARTING}));
    }

    @Test
    public void shouldPropagateFailureWhileActorClosingAndRun() {
        final RuntimeException failure = new RuntimeException("foo");
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorClosing() {
                super.onActorClosing();
                this.actor.run(() -> {
                    throw failure;
                });
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((boolean)closeFuture.isCompletedExceptionally()).isTrue();
        Assertions.assertThat((Throwable)closeFuture.getException()).isEqualTo((Object)failure);
        Assertions.assertThat(actor.phases).isEqualTo((Object)Lists.newArrayList((Object[])new ActorTask.ActorLifecyclePhase[]{ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.CLOSE_REQUESTED, ActorTask.ActorLifecyclePhase.CLOSING}));
    }

    @Test
    public void shouldDiscardJobsOnFailureWhileActorStarting() {
        final RuntimeException failure = new RuntimeException("foo");
        final AtomicBoolean isInvoked = new AtomicBoolean();
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarting() {
                super.onActorStarting();
                this.actor.run(() -> isInvoked.set(true));
                throw failure;
            }
        };
        ActorFuture<Void> startedFuture = this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((boolean)startedFuture.isCompletedExceptionally()).isTrue();
        Assertions.assertThat((AtomicBoolean)isInvoked).isFalse();
    }

    @Test
    public void shouldNotCloseOnFailureWhileActorStarted() {
        final AtomicInteger invocations = new AtomicInteger();
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarted() {
                super.onActorStarted();
                this.actor.submit(this::triggerFailure);
            }

            protected void handleFailure(Throwable failure) {
                int inv = invocations.getAndIncrement();
                if (inv < 10) {
                    this.actor.submit(this::triggerFailure);
                }
            }

            private void triggerFailure() {
                throw new RuntimeException("fail");
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Awaitility.await().until(() -> invocations.get() >= 10);
        actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(actor.phases).isEqualTo(LifecycleRecordingActor.FULL_LIFECYCLE);
    }

    @Test
    public void shouldActorSpecificHandleException() {
        final AtomicInteger invocations = new AtomicInteger();
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarted() {
                super.onActorStarted();
                this.actor.run(() -> {
                    throw new RuntimeException("foo");
                });
            }

            public void handleFailure(Throwable failure) {
                invocations.incrementAndGet();
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((int)invocations.get()).isEqualTo(1);
    }

    @Test
    public void shouldHandleFailureWhenExceptionOnFutureContinuation() {
        AtomicInteger invocations = new AtomicInteger();
        CompletableActorFuture future = new CompletableActorFuture();
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this, (ActorFuture)future, invocations){
            final /* synthetic */ ActorFuture val$future;
            final /* synthetic */ AtomicInteger val$invocations;
            {
                this.val$future = actorFuture;
                this.val$invocations = atomicInteger;
            }

            @Override
            public void onActorStarted() {
                super.onActorStarted();
                this.actor.runOnCompletion(this.val$future, (v, t) -> {
                    throw new RuntimeException("foo");
                });
                this.actor.run(() -> this.val$future.complete(null));
            }

            public void handleFailure(Throwable failure) {
                this.val$invocations.incrementAndGet();
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((int)invocations.get()).isEqualTo(1);
    }

    @Test
    public void shouldNotExecuteNextJobsOnFail() {
        final AtomicInteger invocations = new AtomicInteger();
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarted() {
                super.onActorStarted();
                this.actor.submit(() -> this.actor.fail((Throwable)new RuntimeException("foo")));
                this.actor.submit(invocations::incrementAndGet);
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((int)invocations.get()).isEqualTo(0);
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.FAILED));
    }

    @Test
    public void shouldCompleteCloseFutureWhenFailingInStarted() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarted() {
                super.onActorStarted();
                this.actor.fail((Throwable)new RuntimeException("foo"));
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.FAILED));
        Assertions.assertThat((Future)closeFuture).failsWithin(Duration.ofSeconds(1L));
    }

    @Test
    public void shouldCompleteCloseFutureOnExceptionOnClosing() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorClosing() {
                super.onActorClosing();
                throw new RuntimeException("foo");
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.CLOSE_REQUESTED, ActorTask.ActorLifecyclePhase.CLOSING));
        Assertions.assertThat((Future)closeFuture).failsWithin(Duration.ofSeconds(1L));
    }

    @Test
    public void shouldCompleteCloseFutureWhenFailingOnClosing() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorClosing() {
                super.onActorClosing();
                this.actor.fail((Throwable)new RuntimeException("foo"));
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.CLOSE_REQUESTED, ActorTask.ActorLifecyclePhase.CLOSING, ActorTask.ActorLifecyclePhase.FAILED));
        Assertions.assertThat((Future)closeFuture).failsWithin(Duration.ofSeconds(1L));
    }

    @Test
    public void shouldCompleteFuturesWhenFailingOnStarting() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarting() {
                super.onActorStarting();
                this.actor.fail((Throwable)new RuntimeException("foo"));
            }
        };
        ActorFuture<Void> startFuture = this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.FAILED));
        Assertions.assertThat(startFuture).failsWithin(Duration.ofSeconds(1L));
        Assertions.assertThat((Future)closeFuture).failsWithin(Duration.ofSeconds(1L));
    }

    @Test
    public void shouldCompleteFuturesOnExceptionOnStarting() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorStarting() {
                super.onActorStarting();
                throw new RuntimeException("hello");
            }
        };
        ActorFuture<Void> startFuture = this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING));
        Assertions.assertThat(startFuture).failsWithin(Duration.ofSeconds(1L));
        Assertions.assertThat((Future)closeFuture).failsWithin(Duration.ofSeconds(1L));
    }

    @Test
    public void shouldCompleteCloseFutureWhenFailingOnCloseRequested() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorCloseRequested() {
                super.onActorCloseRequested();
                this.actor.fail((Throwable)new RuntimeException("hello"));
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.CLOSE_REQUESTED, ActorTask.ActorLifecyclePhase.FAILED));
        Assertions.assertThat((Future)closeFuture).failsWithin(Duration.ofSeconds(1L));
    }

    @Test
    public void shouldCompleteCloseFutureWhenExceptionOnCloseRequested() {
        LifecycleRecordingActor actor = new LifecycleRecordingActor(this){

            @Override
            public void onActorCloseRequested() {
                super.onActorCloseRequested();
                throw new RuntimeException("hello");
            }
        };
        this.schedulerRule.submitActor(actor);
        this.schedulerRule.workUntilDone();
        ActorFuture closeFuture = actor.closeAsync();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(actor.phases).isEqualTo(List.of(ActorTask.ActorLifecyclePhase.STARTING, ActorTask.ActorLifecyclePhase.STARTED, ActorTask.ActorLifecyclePhase.CLOSE_REQUESTED));
        Assertions.assertThat((Future)closeFuture).failsWithin(Duration.ofSeconds(1L));
    }
}

