/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.scheduler.functional;

import io.camunda.zeebe.scheduler.Actor;
import io.camunda.zeebe.scheduler.ActorThread;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.future.CompletableActorFuture;
import io.camunda.zeebe.scheduler.testing.ControlledActorSchedulerExtension;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.assertj.core.api.ThrowableAssertAlternative;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

final class ActorFutureTest {
    @RegisterExtension
    final ControlledActorSchedulerExtension schedulerRule = new ControlledActorSchedulerExtension();

    ActorFutureTest() {
    }

    @Test
    void shouldInvokeCallbackOnFutureCompletion() {
        final CompletableActorFuture future = new CompletableActorFuture();
        final AtomicInteger callbackInvocations = new AtomicInteger(0);
        Actor waitingActor = new Actor(this){

            protected void onActorStarted() {
                this.actor.runOnCompletion((ActorFuture)future, (r, t) -> callbackInvocations.incrementAndGet());
            }
        };
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.complete(null);
            }
        };
        this.schedulerRule.submitActor(waitingActor);
        this.schedulerRule.workUntilDone();
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((AtomicInteger)callbackInvocations).hasValue(1);
    }

    @Test
    void shouldInvokeCallbackOnBlockPhaseForFutureCompletion() {
        final CompletableActorFuture future = new CompletableActorFuture();
        final AtomicInteger callbackInvocations = new AtomicInteger(0);
        Actor waitingActor = new Actor(this){

            protected void onActorStarted() {
                this.actor.runOnCompletionBlockingCurrentPhase((ActorFuture)future, (r, t) -> callbackInvocations.incrementAndGet());
            }
        };
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.complete(null);
            }
        };
        this.schedulerRule.submitActor(waitingActor);
        this.schedulerRule.workUntilDone();
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((AtomicInteger)callbackInvocations).hasValue(1);
    }

    @Test
    void shouldInvokeCallbackOnAllFutureCompletedSuccessfully() {
        final CompletableActorFuture future1 = new CompletableActorFuture();
        final CompletableActorFuture future2 = new CompletableActorFuture();
        final ArrayList invocations = new ArrayList();
        final ArrayList results = new ArrayList();
        Actor waitingActor = new Actor(this){

            protected void onActorStarted() {
                this.actor.runOnCompletion(Arrays.asList(future1, future2), t -> {
                    invocations.add(t);
                    results.add((String)future1.join());
                    results.add((String)future2.join());
                });
            }
        };
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future1.complete((Object)"foo");
                future2.complete((Object)"bar");
            }
        };
        this.schedulerRule.submitActor(waitingActor);
        this.schedulerRule.workUntilDone();
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        ((ListAssert)Assertions.assertThat(invocations).hasSize(1)).containsNull();
        Assertions.assertThat(results).contains((Object[])new String[]{"foo", "bar"});
    }

    @Test
    void shouldInvokeCallbackOnEmptyFutureList() {
        final List futures = Collections.emptyList();
        final ArrayList invocations = new ArrayList();
        Actor waitingActor = new Actor(this){

            protected void onActorStarted() {
                this.actor.runOnCompletion((Collection)futures, invocations::add);
            }
        };
        this.schedulerRule.submitActor(waitingActor);
        this.schedulerRule.workUntilDone();
        ((ListAssert)Assertions.assertThat(invocations).hasSize(1)).containsNull();
    }

    @Test
    void shouldInvokeCallbackOnAllFutureCompletedExceptionally() {
        final CompletableActorFuture future1 = new CompletableActorFuture();
        final CompletableActorFuture future2 = new CompletableActorFuture();
        final ArrayList invocations = new ArrayList();
        Actor waitingActor = new Actor(this){

            protected void onActorStarted() {
                this.actor.runOnCompletion(Arrays.asList(future1, future2), invocations::add);
            }
        };
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future1.completeExceptionally((Throwable)new RuntimeException("foo"));
                future2.completeExceptionally((Throwable)new RuntimeException("bar"));
            }
        };
        this.schedulerRule.submitActor(waitingActor);
        this.schedulerRule.workUntilDone();
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(invocations).hasSize(1);
        Assertions.assertThat((String)((Throwable)invocations.getFirst()).getMessage()).isEqualTo("bar");
    }

    @Test
    void shouldNotBlockExecutionWhenRegisteredOnFuture() {
        BlockedCallActor actor = new BlockedCallActor();
        this.schedulerRule.submitActor(actor);
        actor.waitOnFuture();
        this.schedulerRule.workUntilDone();
        ActorFuture<Integer> future = actor.call(42);
        this.schedulerRule.workUntilDone();
        Integer result = (Integer)future.join();
        Assertions.assertThat((Integer)result).isEqualTo(42);
    }

    @Test
    void shouldNotBlockExecutionOnRunOnCompletion() {
        BlockedCallActorWithRunOnCompletion actor = new BlockedCallActorWithRunOnCompletion();
        this.schedulerRule.submitActor(actor);
        actor.waitOnFuture();
        this.schedulerRule.workUntilDone();
        ActorFuture<Integer> future = actor.call(42);
        this.schedulerRule.workUntilDone();
        Integer result = (Integer)future.join();
        Assertions.assertThat((Integer)result).isEqualTo(42);
    }

    @Test
    void shouldInvokeCallbackOnCompletedFuture() {
        final AtomicReference futureResult = new AtomicReference();
        this.schedulerRule.submitActor(new Actor(this){

            protected void onActorStarted() {
                this.actor.runOnCompletion((ActorFuture)CompletableActorFuture.completed((Object)"foo"), (r, t) -> futureResult.set(r));
            }
        });
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((String)((String)futureResult.get())).isEqualTo("foo");
    }

    @Test
    void shouldInvokeCallbackOnBlockPhaseForCompletedFuture() {
        final AtomicReference futureResult = new AtomicReference();
        this.schedulerRule.submitActor(new Actor(this){

            protected void onActorStarted() {
                this.actor.runOnCompletionBlockingCurrentPhase((ActorFuture)CompletableActorFuture.completed((Object)"foo"), (r, t) -> futureResult.set(r));
            }
        });
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((String)((String)futureResult.get())).isEqualTo("foo");
    }

    @Test
    void shouldReturnCompletedFutureWithNullValue() {
        CompletableActorFuture completed = CompletableActorFuture.completed(null);
        Assertions.assertThat((Future)completed).isDone();
        Assertions.assertThat((Object)((Void)completed.join())).isNull();
    }

    @Test
    void shouldReturnCompletedFuture() {
        Object result = new Object();
        CompletableActorFuture completed = CompletableActorFuture.completed((Object)result);
        Assertions.assertThat((Future)completed).isDone();
        Assertions.assertThat((Object)completed.join()).isEqualTo(result);
    }

    @Test
    void shouldReturnCompletedExceptionallyFuture() {
        RuntimeException result = new RuntimeException("Something bad happened!");
        CompletableActorFuture completed = CompletableActorFuture.completedExceptionally((Throwable)result);
        Assertions.assertThat((Future)completed).isDone();
        Assertions.assertThat((boolean)completed.isCompletedExceptionally()).isTrue();
        Assertions.assertThatThrownBy(() -> ((CompletableActorFuture)completed).join()).hasMessageContaining("Something bad happened!");
    }

    @Test
    void shouldInvokeCallbacksAfterCloseIsCalled() {
        CompletableActorFuture f1 = new CompletableActorFuture();
        CompletableActorFuture f2 = new CompletableActorFuture();
        Object result1 = new Object();
        Object result2 = new Object();
        TestActor actor = new TestActor();
        this.schedulerRule.submitActor(actor);
        ArrayList receivedObjects = new ArrayList();
        actor.awaitFuture(f1, (o, t) -> receivedObjects.add(o));
        actor.awaitFuture(f2, (o, t) -> receivedObjects.add(o));
        this.schedulerRule.workUntilDone();
        f1.complete(result1);
        f2.complete(result2);
        actor.close();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat(receivedObjects).containsExactly(new Object[]{result1, result2});
    }

    @Test
    void joinShouldThrowExecutionException() {
        CompletableActorFuture future = new CompletableActorFuture();
        RuntimeException throwable = new RuntimeException();
        future.completeExceptionally((Throwable)throwable);
        AbstractThrowableAssert thrownBy = Assertions.assertThatThrownBy(() -> ((CompletableActorFuture)future).join());
        thrownBy.isInstanceOf(ExecutionException.class);
        thrownBy.hasCause((Throwable)throwable);
    }

    @Test
    void shouldCompleteFutureAndWaitOnNonActorThread() throws Exception {
        final CompletableActorFuture future = new CompletableActorFuture();
        this.schedulerRule.submitActor(new Actor(this){

            protected void onActorStarted() {
                future.complete((Object)250);
            }
        });
        new Thread(){

            @Override
            public void run() {
                ActorFutureTest.this.schedulerRule.workUntilDone();
            }
        }.start();
        Integer value = (Integer)future.get();
        Assertions.assertThat((Integer)value).isEqualTo(250);
    }

    @Test
    void shouldCompleteFutureExceptionallyAndWaitOnNonActorThread() {
        final CompletableActorFuture future = new CompletableActorFuture();
        this.schedulerRule.submitActor(new Actor(this){

            protected void onActorStarted() {
                future.completeExceptionally((Throwable)new IllegalArgumentException("moep"));
            }
        });
        new Thread(){

            @Override
            public void run() {
                ActorFutureTest.this.schedulerRule.workUntilDone();
            }
        }.start();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((CompletableActorFuture)future).get()).isInstanceOf(ExecutionException.class)).hasMessage("moep");
    }

    @Test
    void shouldReturnValueOnNonActorThread() throws Exception {
        CompletableActorFuture future = CompletableActorFuture.completed((Object)"value");
        String value = (String)future.get(5L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((String)value).isEqualTo("value");
    }

    @Test
    void shouldThrowExceptionOnNonActorThread() {
        CompletableActorFuture future = CompletableActorFuture.completedExceptionally((Throwable)new IllegalArgumentException("moep"));
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> future.get(5L, TimeUnit.MILLISECONDS)).isInstanceOf(ExecutionException.class)).hasMessage("moep");
    }

    @Test
    void shouldThrowTimeoutOnNonActorThread() {
        CompletableActorFuture future = new CompletableActorFuture();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> future.get(5L, TimeUnit.MILLISECONDS)).isInstanceOf(TimeoutException.class)).hasMessage("Timeout after: 5 MILLISECONDS");
    }

    @Test
    void shouldFailToStaticallyCreateExceptionallyCompletedFutureWithNull() {
        RuntimeException result = null;
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> CompletableActorFuture.completedExceptionally((Throwable)result)).isInstanceOf(NullPointerException.class)).hasMessageContaining("Throwable must not be null.");
    }

    @Test
    void shouldFailToExceptionallyCompleteFutureWithNull() {
        CompletableActorFuture future = new CompletableActorFuture();
        RuntimeException result = null;
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> future.completeExceptionally((Throwable)result)).isInstanceOf(NullPointerException.class)).hasMessageContaining("Throwable must not be null.");
    }

    @Test
    void shouldFailToExceptionallyCompleteFutureWithNullAndMessage() {
        CompletableActorFuture future = new CompletableActorFuture();
        RuntimeException result = null;
        String message = "foo";
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> future.completeExceptionally("foo", (Throwable)result)).isInstanceOf(NullPointerException.class)).hasMessageContaining("Throwable must not be null.");
    }

    @Test
    void shouldRunOnComplete() {
        ActorB actorB = new ActorB();
        this.schedulerRule.submitActor(actorB);
        ActorA actorA = new ActorA(actorB);
        this.schedulerRule.submitActor(actorA);
        ActorFuture<Integer> future = actorA.sumValues();
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((boolean)future.isDone()).isTrue();
        Assertions.assertThat((Integer)((Integer)future.join())).isEqualTo(51967);
    }

    @Test
    void shouldInvokeCallbackOnFutureCompletionIfCallerIsNotActor() {
        final CompletableActorFuture future = new CompletableActorFuture();
        AtomicInteger callbackInvocations = new AtomicInteger(0);
        future.onComplete((r, t) -> callbackInvocations.incrementAndGet());
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.complete(null);
            }
        };
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((AtomicInteger)callbackInvocations).hasValue(1);
    }

    @Test
    void shouldInvokeCallbackOnFutureCompletionOnProvidedExecutor() {
        final CompletableActorFuture future = new CompletableActorFuture();
        AtomicInteger callbackInvocations = new AtomicInteger(0);
        AtomicInteger executorCount = new AtomicInteger(0);
        Executor decoratedExecutor = runnable -> {
            executorCount.getAndIncrement();
            runnable.run();
        };
        future.onComplete((r, t) -> callbackInvocations.incrementAndGet(), decoratedExecutor);
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.complete(null);
            }
        };
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((AtomicInteger)callbackInvocations).hasValue(1);
        Assertions.assertThat((AtomicInteger)executorCount).hasValue(1);
    }

    @Test
    void shouldInvokeCallbackOnFutureCompletionExceptionIfCallerIsNotActor() {
        final CompletableActorFuture future = new CompletableActorFuture();
        AtomicReference callBackError = new AtomicReference();
        future.onComplete((r, t) -> callBackError.set(t));
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.completeExceptionally((Throwable)new RuntimeException("Expected"));
            }
        };
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)((Throwable)callBackError.get())).isInstanceOf(RuntimeException.class)).hasMessageContaining("Expected");
    }

    @Test
    void shouldInvokeCallbackOnFutureCompletionErrorOnProvidedExecutor() {
        final CompletableActorFuture future = new CompletableActorFuture();
        AtomicReference callBackError = new AtomicReference();
        AtomicInteger executorCount = new AtomicInteger(0);
        Executor decoratedExecutor = runnable -> {
            executorCount.getAndIncrement();
            runnable.run();
        };
        future.onComplete((r, t) -> callBackError.set(t), decoratedExecutor);
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.completeExceptionally((Throwable)new RuntimeException("Expected"));
            }
        };
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)((Throwable)callBackError.get())).isInstanceOf(RuntimeException.class)).hasMessageContaining("Expected");
        Assertions.assertThat((AtomicInteger)executorCount).hasValue(1);
    }

    @Test
    void shouldInvokeCallbackAddedAfterCompletionIfCallerIsNonActor() {
        final CompletableActorFuture future = new CompletableActorFuture();
        AtomicInteger futureResult = new AtomicInteger();
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.complete((Object)1);
            }
        };
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        AtomicInteger executorCount = new AtomicInteger(0);
        Executor decoratedExecutor = runnable -> {
            executorCount.getAndIncrement();
            runnable.run();
        };
        future.onComplete((r, t) -> futureResult.set((int)r), decoratedExecutor);
        Assertions.assertThat((int)futureResult.get()).isOne();
        Assertions.assertThat((AtomicInteger)executorCount).hasValue(1);
    }

    @Test
    void shouldInvokeCallbackAddedAfterCompletionErrorIfCallerIsNonActor() {
        final CompletableActorFuture future = new CompletableActorFuture();
        AtomicReference futureResult = new AtomicReference();
        Actor completingActor = new Actor(this){

            protected void onActorStarted() {
                future.completeExceptionally((Throwable)new RuntimeException("Expected"));
            }
        };
        this.schedulerRule.submitActor(completingActor);
        this.schedulerRule.workUntilDone();
        AtomicInteger executorCount = new AtomicInteger(0);
        Executor decoratedExecutor = runnable -> {
            executorCount.getAndIncrement();
            runnable.run();
        };
        future.onComplete((r, t) -> futureResult.set(t), decoratedExecutor);
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)((Throwable)futureResult.get())).isInstanceOf(RuntimeException.class)).hasMessageContaining("Expected");
        Assertions.assertThat((AtomicInteger)executorCount).hasValue(1);
    }

    @Test
    void shouldChainWithAndThen() {
        AtomicInteger executorCount = new AtomicInteger(0);
        Executor decoratedExecutor = runnable -> {
            executorCount.getAndIncrement();
            runnable.run();
        };
        CompletableActorFuture future1 = new CompletableActorFuture();
        CompletableActorFuture future2 = new CompletableActorFuture();
        ActorFuture chainedFuture = future1.andThen(r -> future2, decoratedExecutor);
        Assertions.assertThat((Future)chainedFuture).isNotDone();
        Assertions.assertThat((AtomicInteger)executorCount).hasValue(0);
        future1.complete(null);
        Assertions.assertThat((Future)chainedFuture).isNotDone();
        Assertions.assertThat((AtomicInteger)executorCount).hasValue(1);
        future2.complete(null);
        Assertions.assertThat((Future)chainedFuture).isDone();
        Assertions.assertThat((AtomicInteger)executorCount).hasValue(2);
    }

    @Test
    void andThenChainPropagatesInitialException() {
        CompletableActorFuture future1 = new CompletableActorFuture();
        CompletableActorFuture future2 = new CompletableActorFuture();
        ActorFuture chained = future1.andThen(r -> future2, Runnable::run);
        RuntimeException expectedException = new RuntimeException("expected");
        future1.completeExceptionally((Throwable)expectedException);
        Assertions.assertThat((Future)chained).failsWithin(1L, TimeUnit.SECONDS).withThrowableThat().withCause((Throwable)expectedException);
    }

    @Test
    void andThenChainPropagatesValue() {
        ActorFuture chained = CompletableActorFuture.completed((Object)"expected").andThen(CompletableActorFuture::completed, Runnable::run);
        Assertions.assertThat((Future)chained).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)"expected");
    }

    @Test
    void shouldChainThenApply() {
        CompletableActorFuture future = new CompletableActorFuture();
        ActorFuture chained = future.thenApply(value -> value + 1, Runnable::run);
        future.complete((Object)1);
        Assertions.assertThat((Future)chained).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)2);
    }

    @Test
    void shouldShortCircuitThenApplyOnFailure() {
        CompletableActorFuture future = new CompletableActorFuture();
        AtomicBoolean called = new AtomicBoolean(false);
        RuntimeException failure = new RuntimeException("foo");
        ActorFuture chained = future.thenApply(value -> {
            called.set(true);
            return value;
        }, Runnable::run);
        future.completeExceptionally((Throwable)failure);
        Assertions.assertThat((Future)chained).failsWithin(Duration.ofSeconds(1L)).withThrowableThat().havingCause().isSameAs((Object)failure);
        Assertions.assertThat((AtomicBoolean)called).isFalse();
    }

    @Test
    void shouldChainAndCompleteIntermediateFuturesOnApply() {
        CompletableActorFuture original = new CompletableActorFuture();
        ActorFuture firstElement = original.thenApply(value -> value + 1, Runnable::run);
        ActorFuture secondElement = firstElement.thenApply(value -> value + 1, Runnable::run);
        original.complete((Object)1);
        Assertions.assertThat((Future)firstElement).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)2);
        Assertions.assertThat((Future)secondElement).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)3);
    }

    @Test
    void shouldApplyOnExecutor() {
        CompletableActorFuture future = new CompletableActorFuture();
        AtomicBoolean onExecutor = new AtomicBoolean(false);
        AtomicBoolean calledOnExecutor = new AtomicBoolean(false);
        ActorFuture chained = future.thenApply(value -> {
            calledOnExecutor.set(onExecutor.get());
            return value + 1;
        }, task -> {
            onExecutor.set(true);
            task.run();
            onExecutor.set(false);
        });
        future.complete((Object)1);
        Assertions.assertThat((Future)chained).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)2);
        Assertions.assertThat((AtomicBoolean)calledOnExecutor).isTrue();
    }

    @Test
    void shouldShortCircuitMiddleOfChainWithActor() {
        CompletableActorFuture original = new CompletableActorFuture();
        TestActor testActor = new TestActor();
        RuntimeException failure = new RuntimeException("foo");
        AtomicReference firstElement = new AtomicReference();
        AtomicReference secondElement = new AtomicReference();
        AtomicReference thirdElement = new AtomicReference();
        this.schedulerRule.submitActor(testActor);
        this.schedulerRule.workUntilDone();
        testActor.run(() -> {
            firstElement.set(original.thenApply(value -> value + 1, (Executor)((Object)testActor)));
            secondElement.set(((ActorFuture)firstElement.get()).thenApply(value -> {
                throw failure;
            }, (Executor)((Object)testActor)));
            thirdElement.set(((ActorFuture)secondElement.get()).thenApply(value -> value + 1, (Executor)((Object)testActor)));
        });
        this.schedulerRule.workUntilDone();
        original.complete((Object)1);
        this.schedulerRule.workUntilDone();
        Assertions.assertThat((Future)original).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)1);
        Assertions.assertThat((Future)((Future)firstElement.get())).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)2);
        ((ThrowableAssertAlternative)Assertions.assertThat((Future)((Future)secondElement.get())).failsWithin(Duration.ofSeconds(1L)).withThrowableThat().isInstanceOf(ExecutionException.class)).havingRootCause().isSameAs((Object)failure);
        ((ThrowableAssertAlternative)Assertions.assertThat((Future)((Future)thirdElement.get())).failsWithin(Duration.ofSeconds(1L)).withThrowableThat().isInstanceOf(ExecutionException.class)).havingRootCause().isSameAs((Object)failure);
    }

    @Test
    void shouldShortCircuitMiddleOfChain() {
        CompletableActorFuture original = new CompletableActorFuture();
        RuntimeException failure = new RuntimeException("foo");
        ActorFuture firstElement = original.thenApply(value -> value + 1, Runnable::run);
        ActorFuture secondElement = firstElement.thenApply(value -> {
            throw failure;
        }, Runnable::run);
        ActorFuture thirdElement = secondElement.thenApply(value -> value + 1, Runnable::run);
        original.complete((Object)1);
        Assertions.assertThat((Future)original).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)1);
        Assertions.assertThat((Future)firstElement).succeedsWithin(Duration.ofSeconds(1L)).isEqualTo((Object)2);
        ((ThrowableAssertAlternative)Assertions.assertThat((Future)secondElement).failsWithin(Duration.ofSeconds(1L)).withThrowableThat().isInstanceOf(ExecutionException.class)).havingRootCause().isSameAs((Object)failure);
        ((ThrowableAssertAlternative)Assertions.assertThat((Future)thirdElement).failsWithin(Duration.ofSeconds(1L)).withThrowableThat().isInstanceOf(ExecutionException.class)).havingRootCause().isSameAs((Object)failure);
    }

    private static final class BlockedCallActor
    extends Actor {
        private BlockedCallActor() {
        }

        public void waitOnFuture() {
            this.actor.call(() -> this.actor.runOnCompletionBlockingCurrentPhase((ActorFuture)new CompletableActorFuture(), (r, t) -> {}));
        }

        public ActorFuture<Integer> call(int returnValue) {
            return this.actor.call(() -> returnValue);
        }
    }

    private static final class BlockedCallActorWithRunOnCompletion
    extends Actor {
        private BlockedCallActorWithRunOnCompletion() {
        }

        public void waitOnFuture() {
            this.actor.call(() -> this.actor.runOnCompletion((ActorFuture)new CompletableActorFuture(), (r, t) -> {}));
        }

        public ActorFuture<Integer> call(int returnValue) {
            return this.actor.call(() -> returnValue);
        }
    }

    private static final class TestActor
    extends Actor {
        private TestActor() {
        }

        public <T> void awaitFuture(ActorFuture<T> f, BiConsumer<T, Throwable> onCompletion) {
            this.actor.call(() -> this.actor.runOnCompletionBlockingCurrentPhase(f, onCompletion));
        }

        public void close() {
            this.actor.close();
        }
    }

    private static final class ActorB
    extends Actor {
        private ActorB() {
        }

        public ActorFuture<Integer> getValue() {
            return this.actor.call(() -> 51966);
        }
    }

    private static final class ActorA
    extends Actor {
        private final ActorB actorB;

        ActorA(ActorB actorB) {
            this.actorB = actorB;
        }

        ActorFuture<Integer> sumValues() {
            CompletableActorFuture future = new CompletableActorFuture();
            this.actor.call(() -> this.actorB.getValue().onComplete((v, t) -> {
                future.complete((Object)(v + 1));
                ActorThread current = ActorThread.current();
                assert (current != null) : "Expected to run in actor thread!";
                assert (current.getCurrentTask().getActor() == this) : "Expected to run in same actor!";
            }));
            return future;
        }
    }
}

