/*
 * Copyright © 2017 camunda services GmbH (info@camunda.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.camunda.zeebe.client.api.command;

import io.camunda.zeebe.client.api.response.DocumentReferenceResponse;
import java.io.InputStream;
import java.time.Duration;
import java.util.Map;

public interface CreateDocumentCommandStep1 extends DocumentBuilderStep1 {

  /**
   * Sets the content of the document.
   *
   * @param content the content of the document as stream
   */
  @Override
  CreateDocumentCommandStep2 content(InputStream content);

  /**
   * Sets the content of the document.
   *
   * @param content the content of the document as byte array
   */
  @Override
  CreateDocumentCommandStep2 content(byte[] content);

  /**
   * Sets the content of the document.
   *
   * @param content the content of the document as string
   */
  @Override
  CreateDocumentCommandStep2 content(String content);

  interface CreateDocumentCommandStep2
      extends FinalCommandStep<DocumentReferenceResponse>, DocumentBuilderStep2 {

    /**
     * Sets the document ID. If not set, the document ID will be generated by the document store.
     *
     * <p>The document ID must be unique within the store. If a document with the same ID already
     * exists, the request will fail.
     *
     * <p>A custom document ID can be used to prevent duplicate documents from being created in an
     * environment where requests can be retried (e.g. job workers).
     *
     * @param documentId the document ID
     */
    CreateDocumentCommandStep2 documentId(String documentId);

    /**
     * Sets the store ID. If not set, the default store will be used.
     *
     * <p>The store ID is the identifier of the document store where the document should be stored.
     * Documents with the same ID can exist in different stores.
     *
     * @param storeId the store ID
     */
    CreateDocumentCommandStep2 storeId(String storeId);

    /**
     * Sets the content type of the document.
     *
     * @param contentType the content type of the document
     */
    @Override
    CreateDocumentCommandStep2 contentType(String contentType);

    /**
     * Sets the file name of the document.
     *
     * @param name the file name of the document
     */
    @Override
    CreateDocumentCommandStep2 fileName(String name);

    /**
     * Sets the time-to-live of the document. The document will be automatically deleted after the
     * time-to-live is exceeded.
     *
     * <p>Depending on the document store, a maximum and a default time-to-live strategy may be
     * enforced.
     *
     * @param timeToLive the time-to-live of the document
     */
    @Override
    CreateDocumentCommandStep2 timeToLive(Duration timeToLive);

    /**
     * Adds a custom key-value pair to the document metadata.
     *
     * @param key custom metadata key
     * @param value custom metadata value
     */
    @Override
    CreateDocumentCommandStep2 customMetadata(String key, Object value);

    /**
     * Adds custom key-value pairs to the document metadata.
     *
     * @param customMetadata custom metadata key-value pairs
     */
    @Override
    CreateDocumentCommandStep2 customMetadata(Map<String, Object> customMetadata);

    /**
     * Sets the process definition that the document is associated with.
     *
     * @param processDefinitionId the process definition ID
     */
    CreateDocumentCommandStep2 processDefinitionId(String processDefinitionId);

    /**
     * Sets the process instance key that the document is associated with.
     *
     * @param processInstanceKey the process instance key
     */
    CreateDocumentCommandStep2 processInstanceKey(long processInstanceKey);
  }
}
