/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.protocols.raft.roles;

import com.google.common.base.Preconditions;
import io.atomix.protocols.raft.RaftException;
import io.atomix.protocols.raft.RaftServer;
import io.atomix.protocols.raft.cluster.RaftMember;
import io.atomix.protocols.raft.cluster.impl.RaftMemberContext;
import io.atomix.protocols.raft.protocol.AppendRequest;
import io.atomix.protocols.raft.protocol.AppendResponse;
import io.atomix.protocols.raft.protocol.ConfigureRequest;
import io.atomix.protocols.raft.protocol.ConfigureResponse;
import io.atomix.protocols.raft.protocol.InstallRequest;
import io.atomix.protocols.raft.protocol.InstallResponse;
import io.atomix.protocols.raft.protocol.RaftRequest;
import io.atomix.protocols.raft.roles.AbstractAppender;
import io.atomix.protocols.raft.roles.LeaderRole;
import io.atomix.protocols.raft.storage.snapshot.Snapshot;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

final class LeaderAppender
extends AbstractAppender {
    private static final long MAX_HEARTBEAT_WAIT = 60000L;
    private static final int MINIMUM_BACKOFF_FAILURE_COUNT = 5;
    private final LeaderRole leader;
    private final long leaderTime;
    private final long leaderIndex;
    private final long heartbeatInterval;
    private long heartbeatTime;
    private int heartbeatFailures;
    private CompletableFuture<Long> heartbeatFuture;
    private CompletableFuture<Long> nextHeartbeatFuture;
    private final Map<Long, CompletableFuture<Long>> appendFutures = new HashMap<Long, CompletableFuture<Long>>();

    LeaderAppender(LeaderRole leader) {
        super(leader.raft);
        this.leader = (LeaderRole)Preconditions.checkNotNull((Object)leader, (Object)"leader cannot be null");
        this.leaderTime = System.currentTimeMillis();
        this.leaderIndex = this.raft.getLogWriter().getNextIndex();
        this.heartbeatTime = this.leaderTime;
        this.heartbeatInterval = this.raft.getHeartbeatInterval().toMillis();
    }

    public long getTime() {
        return this.heartbeatTime;
    }

    public long getIndex() {
        return this.leaderIndex;
    }

    private int getQuorumIndex() {
        return this.raft.getCluster().getQuorum() - 2;
    }

    public CompletableFuture<Long> appendEntries() {
        this.raft.checkThread();
        if (this.raft.getCluster().getRemoteMemberStates().isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        if (this.heartbeatFuture == null) {
            CompletableFuture<Long> newHeartbeatFuture = new CompletableFuture<Long>();
            this.heartbeatFuture = newHeartbeatFuture;
            this.heartbeatTime = System.currentTimeMillis();
            for (RaftMemberContext member : this.raft.getCluster().getRemoteMemberStates()) {
                this.appendEntries(member);
            }
            return newHeartbeatFuture;
        }
        if (this.nextHeartbeatFuture == null) {
            this.nextHeartbeatFuture = new CompletableFuture();
            return this.nextHeartbeatFuture;
        }
        return this.nextHeartbeatFuture;
    }

    public CompletableFuture<Long> appendEntries(long index) {
        this.raft.checkThread();
        if (index == 0L) {
            return this.appendEntries();
        }
        if (index <= this.raft.getCommitIndex()) {
            return CompletableFuture.completedFuture(index);
        }
        if (this.raft.getCluster().getActiveMemberStates().isEmpty() && this.raft.getCluster().getPassiveMemberStates().isEmpty()) {
            long previousCommitIndex = this.raft.getCommitIndex();
            this.raft.setCommitIndex(index);
            this.completeCommits(previousCommitIndex, index);
            return CompletableFuture.completedFuture(index);
        }
        if (this.raft.getCluster().getActiveMemberStates().isEmpty()) {
            long previousCommitIndex = this.raft.getCommitIndex();
            this.raft.setCommitIndex(index);
            this.completeCommits(previousCommitIndex, index);
            return CompletableFuture.completedFuture(index);
        }
        return this.appendFutures.computeIfAbsent(index, i -> {
            for (RaftMemberContext member : this.raft.getCluster().getActiveMemberStates()) {
                this.appendEntries(member);
            }
            return new CompletableFuture();
        });
    }

    @Override
    protected void appendEntries(RaftMemberContext member) {
        if (!this.open) {
            return;
        }
        if (member.getFailureCount() >= 5) {
            if ((double)(System.currentTimeMillis() - member.getHeartbeatStartTime()) > Math.min((double)this.heartbeatInterval * Math.pow(2.0, member.getFailureCount()), 60000.0)) {
                this.sendAppendRequest(member, this.buildAppendEmptyRequest(member));
            }
        } else if (member.getConfigTerm() < this.raft.getTerm() || member.getConfigIndex() < this.raft.getCluster().getConfiguration().index()) {
            if (member.canConfigure()) {
                this.sendConfigureRequest(member, this.buildConfigureRequest(member));
            }
        } else if (member.getMember().getType() == RaftMember.Type.RESERVE || member.getMember().getType() == RaftMember.Type.PASSIVE) {
            if (member.canAppend()) {
                this.sendAppendRequest(member, this.buildAppendEmptyRequest(member));
            }
        } else if (member.getMember().getType() == RaftMember.Type.ACTIVE) {
            Snapshot snapshot = this.raft.getSnapshotStore().getSnapshotByIndex(member.getLogReader().getCurrentIndex());
            if (snapshot != null && member.getSnapshotIndex() < snapshot.index()) {
                if (member.canInstall()) {
                    this.sendInstallRequest(member, this.buildInstallRequest(member));
                }
            } else if (member.canAppend()) {
                this.sendAppendRequest(member, this.buildAppendRequest(member, -1L));
            }
        } else if (member.canAppend()) {
            this.sendAppendRequest(member, this.buildAppendRequest(member, -1L));
        }
    }

    @Override
    protected boolean hasMoreEntries(RaftMemberContext member) {
        return member.getMember().getType() != RaftMember.Type.RESERVE && member.getMember().getType() != RaftMember.Type.PASSIVE && member.getLogReader().hasNext();
    }

    private long getHeartbeatTime() {
        int quorumIndex = this.getQuorumIndex();
        if (quorumIndex >= 0) {
            return this.raft.getCluster().getActiveMemberStates((m1, m2) -> Long.compare(m2.getHeartbeatTime(), m1.getHeartbeatTime())).get(quorumIndex).getHeartbeatTime();
        }
        return System.currentTimeMillis();
    }

    private void updateHeartbeatTime(RaftMemberContext member, Throwable error) {
        if (this.heartbeatFuture == null) {
            return;
        }
        this.raft.checkThread();
        if (error != null && member.getHeartbeatStartTime() == this.heartbeatTime) {
            int votingMemberSize = this.raft.getCluster().getActiveMemberStates().size() + (this.raft.getCluster().getMember().getType() == RaftMember.Type.ACTIVE ? 1 : 0);
            int quorumSize = (int)Math.floor(votingMemberSize / 2) + 1;
            if (member.getMember().getType() == RaftMember.Type.ACTIVE && ++this.heartbeatFailures > votingMemberSize - quorumSize) {
                this.heartbeatFuture.completeExceptionally(new RaftException.ProtocolException("Failed to reach consensus", new Object[0]));
                this.completeHeartbeat();
            }
        } else {
            member.setHeartbeatTime(System.currentTimeMillis());
            if (this.heartbeatTime <= this.getHeartbeatTime()) {
                this.heartbeatFuture.complete(null);
                this.completeHeartbeat();
            }
        }
    }

    private void completeHeartbeat() {
        this.heartbeatFailures = 0;
        this.heartbeatFuture = this.nextHeartbeatFuture;
        this.nextHeartbeatFuture = null;
        if (this.heartbeatFuture != null) {
            this.heartbeatTime = System.currentTimeMillis();
            for (RaftMemberContext member : this.raft.getCluster().getRemoteMemberStates()) {
                this.appendEntries(member);
            }
        }
    }

    private void commitEntries() {
        this.raft.checkThread();
        List<RaftMemberContext> members = this.raft.getCluster().getActiveMemberStates((m1, m2) -> Long.compare(m2.getMatchIndex() != 0L ? m2.getMatchIndex() : 0L, m1.getMatchIndex() != 0L ? m1.getMatchIndex() : 0L));
        if (members.isEmpty()) {
            long previousCommitIndex = this.raft.getCommitIndex();
            long commitIndex = this.raft.getLogWriter().getLastIndex();
            this.raft.setCommitIndex(commitIndex);
            this.completeCommits(previousCommitIndex, commitIndex);
            return;
        }
        long commitIndex = members.get(this.getQuorumIndex()).getMatchIndex();
        long previousCommitIndex = this.raft.getCommitIndex();
        if (commitIndex > 0L && commitIndex > previousCommitIndex && this.leaderIndex > 0L && commitIndex >= this.leaderIndex) {
            this.raft.setCommitIndex(commitIndex);
            this.completeCommits(previousCommitIndex, commitIndex);
        }
    }

    private void completeCommits(long previousCommitIndex, long commitIndex) {
        for (long i = previousCommitIndex + 1L; i <= commitIndex; ++i) {
            CompletableFuture<Long> future = this.appendFutures.remove(i);
            if (future == null) continue;
            future.complete(i);
        }
    }

    @Override
    protected void sendAppendRequest(RaftMemberContext member, AppendRequest request) {
        member.setHeartbeatStartTime(this.heartbeatTime);
        super.sendAppendRequest(member, request);
    }

    @Override
    protected void handleAppendRequestFailure(RaftMemberContext member, AppendRequest request, Throwable error) {
        super.handleAppendRequestFailure(member, request, error);
        this.updateHeartbeatTime(member, error);
    }

    @Override
    protected void handleAppendResponseFailure(RaftMemberContext member, AppendRequest request, Throwable error) {
        this.updateHeartbeatTime(member, error);
        super.handleAppendResponseFailure(member, request, error);
    }

    @Override
    protected void handleAppendResponse(RaftMemberContext member, AppendRequest request, AppendResponse response) {
        this.updateHeartbeatTime(member, null);
        super.handleAppendResponse(member, request, response);
    }

    @Override
    protected void handleAppendResponseOk(RaftMemberContext member, AppendRequest request, AppendResponse response) {
        this.succeedAttempt(member);
        if (response.succeeded()) {
            member.appendSucceeded();
            this.updateMatchIndex(member, response);
            if (!request.entries().isEmpty()) {
                this.commitEntries();
            }
            if (this.hasMoreEntries(member)) {
                this.appendEntries(member);
            }
        } else if (response.term() > this.raft.getTerm()) {
            this.raft.setTerm(response.term());
            this.raft.setLeader(null);
            this.raft.transition(RaftServer.Role.FOLLOWER);
        } else {
            member.appendFailed();
            this.resetMatchIndex(member, response);
            this.resetNextIndex(member);
            if (this.hasMoreEntries(member)) {
                this.appendEntries(member);
            }
        }
    }

    @Override
    protected void handleAppendResponseError(RaftMemberContext member, AppendRequest request, AppendResponse response) {
        if (response.term() > this.raft.getTerm()) {
            this.log.debug("Received higher term from {}", (Object)member.getMember().memberId());
            this.raft.setTerm(response.term());
            this.raft.setLeader(null);
            this.raft.transition(RaftServer.Role.FOLLOWER);
        } else {
            super.handleAppendResponseError(member, request, response);
        }
    }

    @Override
    protected void failAttempt(RaftMemberContext member, RaftRequest request, Throwable error) {
        super.failAttempt(member, request, error);
        if (System.currentTimeMillis() - Math.max(this.getHeartbeatTime(), this.leaderTime) > this.raft.getElectionTimeout().toMillis() * 2L) {
            this.log.warn("Suspected network partition. Stepping down");
            this.raft.setLeader(null);
            this.raft.transition(RaftServer.Role.FOLLOWER);
        }
    }

    @Override
    protected void handleConfigureResponse(RaftMemberContext member, ConfigureRequest request, ConfigureResponse response) {
        this.updateHeartbeatTime(member, null);
        super.handleConfigureResponse(member, request, response);
    }

    @Override
    protected void handleConfigureRequestFailure(RaftMemberContext member, ConfigureRequest request, Throwable error) {
        super.handleConfigureRequestFailure(member, request, error);
        this.updateHeartbeatTime(member, error);
    }

    @Override
    protected void handleConfigureResponseFailure(RaftMemberContext member, ConfigureRequest request, Throwable error) {
        this.updateHeartbeatTime(member, error);
        super.handleConfigureResponseFailure(member, request, error);
    }

    @Override
    protected void handleInstallResponse(RaftMemberContext member, InstallRequest request, InstallResponse response) {
        this.updateHeartbeatTime(member, null);
        super.handleInstallResponse(member, request, response);
    }

    @Override
    protected void handleInstallRequestFailure(RaftMemberContext member, InstallRequest request, Throwable error) {
        super.handleInstallRequestFailure(member, request, error);
        this.updateHeartbeatTime(member, error);
    }

    @Override
    protected void handleInstallResponseFailure(RaftMemberContext member, InstallRequest request, Throwable error) {
        this.updateHeartbeatTime(member, error);
        super.handleInstallResponseFailure(member, request, error);
    }
}

