package io.atlassian.blobstore.client.api;

import java.io.InputStream;

import com.atlassian.fugue.Either;
import com.atlassian.fugue.Option;
import com.atlassian.util.concurrent.Promise;
import com.google.common.base.Function;

/**
 * Service exposing get, put and delete methods for blobs.
 *
 * All parameters are not null, and {@link NullPointerException} will be thrown
 * if nulls are supplied as parameters.
 */
public interface BlobStoreService {

  /**
   * Gets a blob, and applies the function to it to produce a result.
   * <p>
   * Internally, the input stream will be closed, the function does not need to
   * handle closing the stream.
   *
   * @param key the blob's logical key
   * @param options request options, like caching parameters
   * @param f callback function. This function cannot return null. If you don't
   * want to return anything from the function use {@link Unit}.
   * @param <A> return type of the callback function. This must not be Void. If
   * you don't want to return anything from the function use {@link Unit}.
   * @return A promise of either the result of the callback function or a
   * {@link Failure}. Possible values are
   * <ul>
   * <li>Either.left(Failure): a non-recoverable error occurred and the callback
   * function was not called.</li> <li>Either.right(value): the callback was
   * called on the blob with Some(InputStream) if a blob was found for logical
   * key 'key', or None if not blob was found. <code>value</code> was returned
   * by the callback.</li>
   * </ul>
   */
  <A> Promise<Either<Failure, A>> get(String key, Access options, Function<Option<GetResult>, A> f);

  /**
   * Stores the given input stream into a new blob and returns the the blob's
   * {@link PutResult}.
   *
   * @param key the blob's logical key
   * @param stream data to upload
   * @param contentLength in bytes
   *
   * @return A promise of either the {@link PutResult} of the written blob or a
   * Failure. Possible values are
   * <ul>
   * <li>Either.left(Failure): the blob was not stored</li>
   * <li>Either.right(PutResult): the blob was stored, the PutResult object can
   * be used to get further information on the details of the operation</li>
   * </ul>
   */
  Promise<Either<Failure, PutResult>> put(String key, InputStream stream, Long contentLength);

  /**
   * Logically deletes a blob. Doesn't actually delete the content, simply
   * removes the key/name mapping from the content hash.
   *
   * @param key the blob's logical key
   * @param hash the current blob's content hash
   * @return A promise of either the result of the delete operation or a
   * Failure. Possible values are
   * <ul>
   * <li>Either.left(Failure): the blob was not deleted</li>
   * <li>Either.right(false): the blob was not deleted because it was not found</li>
   * <li>Either.right(true): the blob was successfully deleted</li>
   * </ul>
   */
  Promise<Either<Failure, Boolean>> delete(String key, String hash);

  /**
   * @deprecated will likely throw {@link UnsupportedOperationException}
   * @see #delete(String, String)
   */
  @Deprecated Promise<Either<Failure, Boolean>> delete(String key);

  /**
   * Move (rename) a blob.
   *
   * @param sourceKey the blob's current logical key
   * @param destinationKey the blob's future logical key
   * @return A promise of either the {@link PutResult} of the moved blob or a
   * Failure.
   * <ul>
   * <li>Either.left(Failure): the blob was not moved</li>
   * <li>Either.right(PutResult): the blob was moved, the PutResult object can
   * be used to get further information on the details of the operation</li>
   * </ul>
   */
  Promise<Either<Failure, PutResult>> move(String sourceKey, String destinationKey);

  /**
   * Copy a blob.
   *
   * @param sourceKey the blob's current logical key
   * @param destinationKey the blob's future logical key
   * @return A promise of either the {@link PutResult} of the copied blob or a
   * Failure.
   * <ul>
   * <li>Either.left(Failure): the blob was not copied</li>
   * <li>Either.right(PutResult): the blob was copied, the PutResult object can
   * be used to get further information on the details of the operation</li>
   * </ul>
   */
  Promise<Either<Failure, PutResult>> copy(String sourceKey, String destinationKey);

  /**
   * Gets the header information of a blob.
   *
   * @param key the blob's logical key
   * @param options request options, like caching parameters
   * @return A promise of the {@link HeadResult} of the the blob if it exists,
   * None if it does not exist, or a Failure in the event of any non-recoverable
   * errors. Possible values are
   * <ul>
   * <li>Either.left(errorMessage): non-recoverable error, could not determine
   * whether blob exists or access its metadata.</li>
   * <li>Either.right(None): the Blobstore service was successfully contacted,
   * but the blob with logical key 'key' does not exist</li>
   * <li>Either.right(Some(HeadResult)): a blob is stored at logical key 'key'
   * and meta data is in HeadResult instance.</li>
   * </ul>
   */
  Promise<Either<Failure, Option<HeadResult>>> head(String key, Access options);

  /**
   * Reset the blobstore service and reconfigure it from the latest
   * configuration.
   */
  void reset();
}
