/*
 * Decompiled with CFR 0.152.
 */
package io.apiman.manager.api.service;

import com.google.common.collect.Lists;
import io.apiman.common.logging.ApimanLoggerFactory;
import io.apiman.common.logging.IApimanLogger;
import io.apiman.common.util.Preconditions;
import io.apiman.gateway.engine.beans.IPolicyProbeResponse;
import io.apiman.gateway.engine.beans.Policy;
import io.apiman.manager.api.beans.apis.ApiGatewayBean;
import io.apiman.manager.api.beans.apis.ApiPlanBean;
import io.apiman.manager.api.beans.apis.ApiStatus;
import io.apiman.manager.api.beans.apis.ApiVersionBean;
import io.apiman.manager.api.beans.clients.ClientStatus;
import io.apiman.manager.api.beans.clients.ClientVersionBean;
import io.apiman.manager.api.beans.contracts.ContractBean;
import io.apiman.manager.api.beans.contracts.ContractStatus;
import io.apiman.manager.api.beans.contracts.NewContractBean;
import io.apiman.manager.api.beans.events.ApimanEventHeaders;
import io.apiman.manager.api.beans.events.ContractCreatedEvent;
import io.apiman.manager.api.beans.gateways.GatewayBean;
import io.apiman.manager.api.beans.idm.PermissionType;
import io.apiman.manager.api.beans.idm.UserDto;
import io.apiman.manager.api.beans.idm.UserMapper;
import io.apiman.manager.api.beans.orgs.OrganizationBean;
import io.apiman.manager.api.beans.plans.PlanStatus;
import io.apiman.manager.api.beans.plans.PlanVersionBean;
import io.apiman.manager.api.beans.policies.PolicyBean;
import io.apiman.manager.api.beans.policies.PolicyType;
import io.apiman.manager.api.beans.summary.ContractSummaryBean;
import io.apiman.manager.api.beans.summary.PolicySummaryBean;
import io.apiman.manager.api.core.IClientValidator;
import io.apiman.manager.api.core.IStorage;
import io.apiman.manager.api.core.IStorageQuery;
import io.apiman.manager.api.core.exceptions.StorageException;
import io.apiman.manager.api.events.EventService;
import io.apiman.manager.api.gateway.GatewayAuthenticationException;
import io.apiman.manager.api.gateway.IGatewayLink;
import io.apiman.manager.api.gateway.IGatewayLinkFactory;
import io.apiman.manager.api.rest.exceptions.ApiNotFoundException;
import io.apiman.manager.api.rest.exceptions.ClientNotFoundException;
import io.apiman.manager.api.rest.exceptions.ContractAlreadyExistsException;
import io.apiman.manager.api.rest.exceptions.ContractNotFoundException;
import io.apiman.manager.api.rest.exceptions.InvalidClientStatusException;
import io.apiman.manager.api.rest.exceptions.NotAuthorizedException;
import io.apiman.manager.api.rest.exceptions.OrganizationNotFoundException;
import io.apiman.manager.api.rest.exceptions.PlanNotFoundException;
import io.apiman.manager.api.rest.exceptions.SystemErrorException;
import io.apiman.manager.api.rest.exceptions.i18n.Messages;
import io.apiman.manager.api.rest.exceptions.util.ExceptionFactory;
import io.apiman.manager.api.rest.impl.audit.AuditUtils;
import io.apiman.manager.api.rest.impl.util.DataAccessUtilMixin;
import io.apiman.manager.api.security.ISecurityContext;
import io.apiman.manager.api.service.ActionService;
import io.apiman.manager.api.service.ClientAppService;
import io.apiman.manager.api.service.PlanService;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;

@ApplicationScoped
@Transactional
public class ContractService
implements DataAccessUtilMixin {
    private static final IApimanLogger LOGGER = ApimanLoggerFactory.getLogger(ContractService.class);
    private IStorage storage;
    private IStorageQuery query;
    private EventService eventService;
    private ClientAppService clientAppService;
    private PlanService planService;
    private ISecurityContext securityContext;
    private IClientValidator clientValidator;
    private IGatewayLinkFactory gatewayLinkFactory;
    private ActionService actionService;

    @Inject
    public ContractService(IStorage storage, IStorageQuery query, EventService eventService, ClientAppService clientAppService, PlanService planService, ISecurityContext securityContext, IClientValidator clientValidator, IGatewayLinkFactory gatewayLinkFactory, ActionService actionService) {
        this.storage = storage;
        this.query = query;
        this.eventService = eventService;
        this.clientAppService = clientAppService;
        this.planService = planService;
        this.securityContext = securityContext;
        this.clientValidator = clientValidator;
        this.gatewayLinkFactory = gatewayLinkFactory;
        this.actionService = actionService;
    }

    public ContractService() {
    }

    public ContractBean createContract(String organizationId, String clientId, String version, NewContractBean bean) throws OrganizationNotFoundException, ClientNotFoundException, ApiNotFoundException, PlanNotFoundException, ContractAlreadyExistsException, NotAuthorizedException {
        ContractBean contract = this.tryAction(() -> this.createContractInternal(organizationId, clientId, version, bean));
        LOGGER.debug("Created new contract {0}: {1}", new Object[]{contract.getId(), contract});
        this.fireContractCreatedEvent(this.securityContext.getCurrentUser(), contract);
        return contract;
    }

    protected ContractBean createContractInternal(String clientOrgId, String clientId, String clientVersion, NewContractBean bean) throws Exception {
        ClientVersionBean cvb = this.clientAppService.getClientVersion(clientOrgId, clientId, clientVersion);
        if (cvb.getStatus() == ClientStatus.Retired) {
            throw ExceptionFactory.invalidClientStatusException();
        }
        ApiVersionBean avb = this.storage.getApiVersion(bean.getApiOrgId(), bean.getApiId(), bean.getApiVersion());
        if (avb == null) {
            throw ExceptionFactory.apiNotFoundException((String)bean.getApiId());
        }
        if (avb.getStatus() != ApiStatus.Published) {
            throw ExceptionFactory.invalidApiStatusException();
        }
        Set plans = Optional.ofNullable(avb.getPlans()).orElse(Collections.emptySet());
        ApiPlanBean apiPlanBean = plans.stream().filter(apb -> apb.getPlanId().equals(bean.getPlanId())).findFirst().orElseThrow(() -> ExceptionFactory.planNotFoundException((String)bean.getPlanId()));
        PlanVersionBean pvb = this.planService.getPlanVersion(bean.getApiOrgId(), bean.getPlanId(), apiPlanBean.getVersion());
        if (pvb.getStatus() != PlanStatus.Locked) {
            throw ExceptionFactory.invalidPlanStatusException();
        }
        ContractBean contract = new ContractBean();
        contract.setClient(cvb);
        contract.setApi(avb);
        contract.setPlan(pvb);
        contract.setCreatedBy(this.securityContext.getCurrentUser());
        contract.setCreatedOn(new Date());
        OrganizationBean planOrg = pvb.getPlan().getOrganization();
        if (!apiPlanBean.isRequiresApproval().booleanValue() || this.securityContext.hasPermission(PermissionType.planAdmin, planOrg.getId())) {
            LOGGER.debug("Contract valid immediately \u2705: {0}", new Object[]{contract});
            contract.setStatus(ContractStatus.Created);
        } else {
            LOGGER.debug("Contract requires approval \u270b: {0}", new Object[]{contract});
            contract.setStatus(ContractStatus.AwaitingApproval);
        }
        try {
            this.storage.createContract(contract);
        }
        catch (IllegalStateException ise) {
            throw ExceptionFactory.contractDuplicateException();
        }
        this.storage.createAuditEntry(AuditUtils.contractCreatedFromClient(contract, this.securityContext));
        this.storage.createAuditEntry(AuditUtils.contractCreatedToApi(contract, this.securityContext));
        ClientStatus oldStatus = cvb.getStatus();
        ClientStatus newStatus = this.clientValidator.determineStatus(cvb);
        if (oldStatus != newStatus) {
            cvb.setStatus(newStatus);
            this.clientAppService.fireClientStatusChangeEvent(cvb, oldStatus);
        }
        cvb.setModifiedBy(this.securityContext.getCurrentUser());
        cvb.setModifiedOn(new Date());
        this.storage.updateClientVersion(cvb);
        return contract;
    }

    private boolean contractAlreadyExists(String organizationId, String clientId, String version, NewContractBean bean) {
        try {
            List contracts = this.query.getClientContracts(organizationId, clientId, version);
            for (ContractSummaryBean contract : contracts) {
                if (!contract.getApiOrganizationId().equals(bean.getApiOrgId()) || !contract.getApiId().equals(bean.getApiId()) || !contract.getApiVersion().equals(bean.getApiVersion()) || !contract.getPlanId().equals(bean.getPlanId())) continue;
                return true;
            }
            return false;
        }
        catch (StorageException e) {
            return false;
        }
    }

    public List<ContractSummaryBean> getClientContractSummaries(String orgId, String clientId, String version) throws ClientNotFoundException, ContractNotFoundException, NotAuthorizedException, StorageException {
        return this.query.getClientContracts(orgId, clientId, version);
    }

    @Transactional
    public ContractBean getContract(Long contractId) throws ClientNotFoundException, ContractNotFoundException, NotAuthorizedException {
        return this.tryAction(() -> {
            ContractBean contract = this.storage.getContract(contractId);
            if (contract == null) {
                throw ExceptionFactory.contractNotFoundException((Long)contractId);
            }
            LOGGER.debug(String.format("Got contract %s: %s", contract.getId(), contract));
            return contract;
        });
    }

    @Transactional
    public void deleteAllContracts(String organizationId, String clientId, String version) throws ClientNotFoundException, NotAuthorizedException {
        ArrayList contractsToDelete = Lists.newArrayList((Iterator)this.tryAction(() -> this.storage.getAllContracts(organizationId, clientId, version)));
        try {
            ClientStatus clientStatus;
            if (!(contractsToDelete.isEmpty() || (clientStatus = ((ContractBean)contractsToDelete.get(0)).getClient().getStatus()) != ClientStatus.Registered && clientStatus != ClientStatus.AwaitingApproval)) {
                this.actionService.unregisterClient(organizationId, clientId, version);
            }
            this.deleteContractsInternal(organizationId, clientId, version, contractsToDelete, contractsToDelete);
        }
        catch (Exception e) {
            throw new SystemErrorException((Throwable)e);
        }
    }

    @Transactional
    public void deleteContract(String organizationId, String clientId, String version, Long contractId) throws ClientNotFoundException, ContractNotFoundException, NotAuthorizedException, InvalidClientStatusException {
        try {
            ArrayList allContracts = Lists.newArrayList((Iterator)this.tryAction(() -> this.storage.getAllContracts(organizationId, clientId, version)));
            ContractBean contractToDelete = this.storage.getContract(contractId);
            if (allContracts.size() <= 1) {
                ClientStatus clientStatus = contractToDelete.getClient().getStatus();
                if (clientStatus == ClientStatus.Registered || clientStatus == ClientStatus.AwaitingApproval) {
                    this.actionService.unregisterClient(organizationId, clientId, version);
                }
                this.deleteContractsInternal(organizationId, clientId, version, allContracts, List.of(contractToDelete));
            } else {
                ClientStatus newStatus = this.deleteContractsInternal(organizationId, clientId, version, allContracts, List.of(contractToDelete));
                if (newStatus == ClientStatus.Registered || newStatus == ClientStatus.AwaitingApproval) {
                    this.actionService.registerClient(organizationId, clientId, version);
                }
            }
        }
        catch (Exception e) {
            throw new SystemErrorException((Throwable)e);
        }
    }

    private ClientStatus deleteContractsInternal(String organizationId, String clientId, String clientVersion, List<ContractBean> allContracts, List<ContractBean> contractsToDelete) throws Exception {
        Preconditions.checkArgument((allContracts.size() > 0 ? 1 : 0) != 0, (Object)"Must have at least 1 contract if you want to delete");
        Preconditions.checkArgument((contractsToDelete.size() > 0 ? 1 : 0) != 0, (Object)"Must nominate at least 1 contract to delete");
        for (ContractBean contract : contractsToDelete) {
            Long contractId = contract.getId();
            if (!contract.getClient().getClient().getOrganization().getId().equals(organizationId)) {
                throw ExceptionFactory.contractNotFoundException((Long)contractId);
            }
            if (!contract.getClient().getClient().getId().equals(clientId)) {
                throw ExceptionFactory.contractNotFoundException((Long)contractId);
            }
            if (!contract.getClient().getVersion().equals(clientVersion)) {
                throw ExceptionFactory.contractNotFoundException((Long)contractId);
            }
            this.storage.deleteContract(contract);
            this.storage.createAuditEntry(AuditUtils.contractBrokenFromClient(contract, this.securityContext));
            this.storage.createAuditEntry(AuditUtils.contractBrokenToApi(contract, this.securityContext));
        }
        ClientVersionBean clientV = this.clientAppService.getClientVersion(organizationId, clientId, clientVersion);
        clientV.setModifiedBy(this.securityContext.getCurrentUser());
        clientV.setModifiedOn(new Date());
        ClientStatus newStatus = this.clientValidator.determineStatus(clientV, allContracts);
        LOGGER.debug("New status for client version {0} is: {1}", new Object[]{clientV, newStatus});
        clientV.setStatus(newStatus);
        this.storage.updateClientVersion(clientV);
        LOGGER.debug("Deleted contract(s): {0}", new Object[]{contractsToDelete});
        return newStatus;
    }

    public List<IPolicyProbeResponse> probePolicy(Long contractId, long policyId, String rawPayload) throws ClientNotFoundException, ContractNotFoundException {
        ContractBean contract = this.getContract(contractId);
        ApiVersionBean avb = contract.getApi();
        OrganizationBean apiOrg = avb.getApi().getOrganization();
        String apiKey = contract.getClient().getApikey();
        Set gatewayIds = contract.getApi().getGateways().stream().map(ApiGatewayBean::getGatewayId).collect(Collectors.toSet());
        if (gatewayIds.size() == 0) {
            return List.of();
        }
        List<PolicyBean> policyChain = this.aggregateContractPolicies(contract);
        int idxFound = -1;
        int policyChainSize = policyChain.size();
        for (int i = 0; i < policyChainSize; ++i) {
            PolicyBean policy = policyChain.get(i);
            if (!policy.getId().equals(policyId)) continue;
            idxFound = i;
        }
        if (idxFound == -1) {
            throw new IllegalArgumentException("Provided policy ID not found in contract " + policyId);
        }
        List gateways = this.tryAction(() -> this.storage.getGateways(gatewayIds));
        LOGGER.debug("Gateways for contract {0}: {1}", new Object[]{contractId, gateways});
        ArrayList<IPolicyProbeResponse> probeResponses = new ArrayList<IPolicyProbeResponse>(gateways.size());
        for (GatewayBean gateway : gateways) {
            IGatewayLink link = this.gatewayLinkFactory.create(gateway);
            try {
                probeResponses.add(link.probe(apiOrg.getId(), avb.getApi().getId(), avb.getVersion(), idxFound, apiKey, rawPayload));
            }
            catch (GatewayAuthenticationException e) {
                throw new SystemErrorException((Throwable)e);
            }
        }
        LOGGER.debug("Probe responses for contract {0}: {1}", new Object[]{contractId, probeResponses});
        return probeResponses;
    }

    public List<PolicyBean> aggregateContractPolicies(ContractBean contractBean) {
        try {
            PolicyType[] types;
            ArrayList<PolicyBean> policies = new ArrayList<PolicyBean>();
            for (PolicyType policyType : types = new PolicyType[]{PolicyType.Client, PolicyType.Plan, PolicyType.Api}) {
                String ver;
                String id;
                String org;
                switch (policyType) {
                    case Client: {
                        org = contractBean.getClient().getClient().getOrganization().getId();
                        id = contractBean.getClient().getClient().getId();
                        ver = contractBean.getClient().getVersion();
                        break;
                    }
                    case Plan: {
                        org = contractBean.getPlan().getPlan().getOrganization().getId();
                        id = contractBean.getPlan().getPlan().getId();
                        ver = contractBean.getPlan().getVersion();
                        break;
                    }
                    case Api: {
                        org = contractBean.getApi().getApi().getOrganization().getId();
                        id = contractBean.getApi().getApi().getId();
                        ver = contractBean.getApi().getVersion();
                        break;
                    }
                    default: {
                        throw new RuntimeException("Missing case for switch!");
                    }
                }
                List clientPolicies = this.query.getPolicies(org, id, ver, policyType);
                for (PolicySummaryBean policySummaryBean : clientPolicies) {
                    policies.add(this.storage.getPolicy(policyType, org, id, ver, policySummaryBean.getId()));
                }
            }
            return policies;
        }
        catch (Exception e) {
            throw ExceptionFactory.actionException((String)Messages.i18n.format("ErrorAggregatingPolicies", new Object[]{e}));
        }
    }

    public List<Policy> aggregateContractPolicies(ContractSummaryBean contractBean) {
        try {
            PolicyType[] types;
            ArrayList<Policy> policies = new ArrayList<Policy>();
            for (PolicyType policyType : types = new PolicyType[]{PolicyType.Client, PolicyType.Plan, PolicyType.Api}) {
                String ver;
                String id;
                String org;
                switch (policyType) {
                    case Client: {
                        org = contractBean.getClientOrganizationId();
                        id = contractBean.getClientId();
                        ver = contractBean.getClientVersion();
                        break;
                    }
                    case Plan: {
                        org = contractBean.getApiOrganizationId();
                        id = contractBean.getPlanId();
                        ver = contractBean.getPlanVersion();
                        break;
                    }
                    case Api: {
                        org = contractBean.getApiOrganizationId();
                        id = contractBean.getApiId();
                        ver = contractBean.getApiVersion();
                        break;
                    }
                    default: {
                        throw new RuntimeException("Missing case for switch!");
                    }
                }
                List clientPolicies = this.query.getPolicies(org, id, ver, policyType);
                for (PolicySummaryBean policySummaryBean : clientPolicies) {
                    PolicyBean policyBean = this.storage.getPolicy(policyType, org, id, ver, policySummaryBean.getId());
                    Policy policy = new Policy();
                    policy.setPolicyJsonConfig(policyBean.getConfiguration());
                    policy.setPolicyImpl(policyBean.getDefinition().getPolicyImpl());
                    policies.add(policy);
                }
            }
            return policies;
        }
        catch (Exception e) {
            throw ExceptionFactory.actionException((String)Messages.i18n.format("ErrorAggregatingPolicies", new Object[]{contractBean.getClientId() + "->" + contractBean.getApiDescription()}), (Exception)e);
        }
    }

    private void fireContractCreatedEvent(String userId, ContractBean contract) {
        LOGGER.debug("Firing contract create event from user {0} on contract {1}", new Object[]{userId, contract});
        UserDto requester = UserMapper.INSTANCE.toDto(this.tryAction(() -> this.storage.getUser(userId)));
        ApimanEventHeaders.Builder builder = ApimanEventHeaders.builder().setId(UUID.randomUUID().toString()).setSource(URI.create("/apiman/events/contracts/created"));
        if (contract.getStatus() == ContractStatus.AwaitingApproval) {
            builder.setSubject("approval-required");
        } else {
            builder.setSubject("created");
        }
        PlanVersionBean plan = contract.getPlan();
        ClientVersionBean cvb = contract.getClient();
        ApiVersionBean avb = contract.getApi();
        OrganizationBean orgA = avb.getApi().getOrganization();
        OrganizationBean orgC = cvb.getClient().getOrganization();
        ContractCreatedEvent contractCreatedEvent = ContractCreatedEvent.builder().setHeaders(builder.build()).setUser(requester).setClientOrgId(orgC.getId()).setClientId(cvb.getClient().getId()).setClientVersion(cvb.getVersion()).setApiOrgId(orgA.getId()).setApiId(avb.getApi().getId()).setApiVersion(avb.getVersion()).setContractId(String.valueOf(contract.getId())).setPlanId(plan.getPlan().getId()).setPlanVersion(plan.getVersion()).setApprovalRequired(Boolean.valueOf(contract.getStatus() == ContractStatus.AwaitingApproval)).build();
        LOGGER.debug("Sending contract created event {0}", new Object[]{contractCreatedEvent});
        this.eventService.fireEvent(contractCreatedEvent);
    }
}

