package com.payu.india.Tasks.base;

import android.os.Binder;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.MainThread;
import androidx.annotation.WorkerThread;

import com.payu.india.Payu.PayuConstants;

import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;

public abstract class PayUAsyncTask<Params, Progress, Result> {

    private volatile PayUAsyncTask.Status mStatus = PayUAsyncTask.Status.PENDING;
    private final AtomicBoolean mCancelled = new AtomicBoolean();
    private final AtomicBoolean mTaskInvoked = new AtomicBoolean();

    private final WorkerRunnable<Params, Result> mWorker;
    private final FutureTask<Result> mFuture;

    private final Executor executor = Executors.newSingleThreadExecutor();
    private final Handler mainHandler;

    /**
     * Creates a new asynchronous task. This constructor must be invoked on the UI thread.
     */
    public PayUAsyncTask() {
        this((Looper) null);
    }

    /**
     * Creates a new asynchronous task. This constructor must be invoked on the UI thread.
     *
     * @hide
     */
    public PayUAsyncTask(@androidx.annotation.Nullable Handler handler) {
        this(handler != null ? handler.getLooper() : null);
    }

    public PayUAsyncTask(@androidx.annotation.Nullable Looper callbackLooper) {
        mainHandler = callbackLooper == null || callbackLooper == Looper.getMainLooper()
                ? new Handler(Looper.getMainLooper())
                : new Handler(callbackLooper);

        mWorker = new WorkerRunnable<Params, Result>() {
            public Result call() throws Exception {
                mTaskInvoked.set(true);
                Result result = null;
                if (Thread.currentThread().isAlive()) {
                    try {
                        mStatus = PayUAsyncTask.Status.RUNNING;
                        result = doInBackground(mParams);
                        Binder.flushPendingCommands();
                    } catch (Throwable tr) {
                        mCancelled.set(true);
                        throw tr;
                    } finally {
                        postResult(result);
                    }
                }
                return result;
            }
        };

        mFuture = new FutureTask<Result>(mWorker) {
            @Override
            protected void done() {
                try {
                    postResultIfNotInvoked(get());
                } catch (InterruptedException e) {
                    // log if needed
                } catch (ExecutionException e) {
                    throw new RuntimeException("An error occurred while executing doInBackground()",
                            e.getCause());
                } catch (CancellationException e) {
                    postResultIfNotInvoked(null);
                }
            }
        };
    }

    @WorkerThread
    protected abstract Result doInBackground(Params... params);

    @MainThread
    public final PayUAsyncTask<Params, Progress, Result> execute(Params... params) {
        return executeOnExecutor(executor, params);
    }


    @MainThread
    public final PayUAsyncTask<Params, Progress, Result> executeOnExecutor(Executor exec,
                                                                           Params... params) {
        if (mStatus != PayUAsyncTask.Status.PENDING) {
            switch (mStatus) {
                case RUNNING:
                    throw new IllegalStateException(PayuConstants.ASYNC_TASK_ILLEGAL_STATE_EXCEPTION_MESSAGE_1);
                case FINISHED:
                    throw new IllegalStateException(PayuConstants.ASYNC_TASK_ILLEGAL_STATE_EXCEPTION_MESSAGE_2);
            }
        }

        mStatus = PayUAsyncTask.Status.RUNNING;
        onPreExecute();
        mWorker.mParams = params;
        exec.execute(mFuture);
        return this;
    }


    private void postResultIfNotInvoked(Result result) {
        final boolean wasTaskInvoked = mTaskInvoked.get();
        if (!wasTaskInvoked) {
            postResult(result);
        }
    }

    private void postResult(Result result) {
        mainHandler.post(() -> {
            onPostExecute(result);
        });
    }

    /**
     * Returns the current status of this task.
     *
     * @return The current status.
     */
    public final PayUAsyncTask.Status getStatus() {
        return mStatus;
    }

    public final boolean isCancelled() {
        return mCancelled.get();
    }

    public final boolean cancel(boolean mayInterruptIfRunning) {
        mCancelled.set(true);
        return mFuture.cancel(mayInterruptIfRunning);
    }


    /**
     * Waits if necessary for the computation to complete, and then
     * retrieves its result.
     *
     * @return The computed result.
     * @throws CancellationException If the computation was cancelled.
     * @throws ExecutionException    If the computation threw an exception.
     * @throws InterruptedException  If the current thread was interrupted
     *                               while waiting.
     */
    public final Result get() throws InterruptedException, ExecutionException {
        return mFuture.get();
    }


    /**
     * Waits if necessary for at most the given time for the computation
     * to complete, and then retrieves its result.
     *
     * @param timeout Time to wait before cancelling the operation.
     * @param unit    The time unit for the timeout.
     * @return The computed result.
     * @throws CancellationException If the computation was cancelled.
     * @throws ExecutionException    If the computation threw an exception.
     * @throws InterruptedException  If the current thread was interrupted
     *                               while waiting.
     * @throws TimeoutException      If the wait timed out.
     */
    public final Result get(long timeout, TimeUnit unit) throws InterruptedException,
            ExecutionException, TimeoutException {
        return mFuture.get(timeout, unit);
    }


    @SafeVarargs
    @MainThread
    protected final void onProgressUpdate(Progress... values) {
    }

    @MainThread
    protected void onPreExecute() {
    }

    @MainThread
    protected void onPostExecute(Result result) {
        mStatus = Status.FINISHED;
    }

    /**
     * Indicates the current status of the task. Each status will be set only once
     * during the lifetime of a task.
     */
    public enum Status {
        /**
         * Indicates that the task has not been executed yet.
         */
        PENDING,
        /**
         * Indicates that the task is running.
         */
        RUNNING,
        /**
         * Indicates that {@link PayUAsyncTask#onPostExecute} has finished.
         */
        FINISHED,
    }

    private static abstract class WorkerRunnable<Params, Result> implements Callable<Result> {
        Params[] mParams;
    }

}
