package com.payu.india.Tasks;

import android.util.Log;

import com.payu.india.Interfaces.LookupApiListener;
import com.payu.india.Model.LookupDetails;
import com.payu.india.Model.McpConversionBean;
import com.payu.india.Model.PayuConfig;
import com.payu.india.Model.PayuResponse;
import com.payu.india.Model.PostData;
import com.payu.india.Payu.PayuConstants;
import com.payu.india.Payu.PayuErrors;
import com.payu.india.Payu.PayuUtils;
import com.payu.india.Tasks.base.PayUAsyncTask;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.net.ProtocolException;
import java.net.URL;
import java.util.ArrayList;

import javax.net.ssl.HttpsURLConnection;

/**
 * Async task which takes care of Getting currencies with conversion rates enabled on merchant access key
 * Takes PayuConfig as input and sends PayuResponse to the calling activity
 * Activity which calls {@link LookupTask} should implement {@link com.payu.india.Interfaces.LookupApiListener}
 */
public class LookupTask extends PayUAsyncTask<PayuConfig, String, PayuResponse> {

    private  final String TAG = getClass().getSimpleName();

    LookupApiListener mLookupApiListener;

    public LookupTask(LookupApiListener lookupApiListener) {
        mLookupApiListener = lookupApiListener;
    }

    @Override
    protected PayuResponse doInBackground(PayuConfig... params) {
        PayuResponse payuResponse = new PayuResponse();
        PostData postData = new PostData();

        try {
            URL url = null;
            // get the payuConfig first
            PayuConfig payuConfig = params[0];

            // set the environment
            switch (payuConfig.getEnvironment()) {
                case PayuConstants.PRODUCTION_ENV:
                    url = new URL(PayuConstants.PRODUCTION_LOOKUP_URL);
                    break;
                case PayuConstants.STAGING_ENV:
                    url = new URL(PayuConstants.TEST_LOOKUP_URL);
                    break;
                default:
                    url = new URL(PayuConstants.PRODUCTION_LOOKUP_URL);
                    break;
            }


            HttpsURLConnection conn = PayuUtils.getHttpsConn(url.toString(), payuConfig.getData(), PayuConstants.REQUEST_APPLICATION_JSON);
            if (null != conn) {
                StringBuffer responseStringBuffer = PayuUtils.getStringBuffer(conn, false);

                JSONObject response = new JSONObject(responseStringBuffer.toString());
                payuResponse.setRawResponse(response);
                if (isJSONArrayAvailableForKey(response, PayuConstants.PAYU_MCP_CONVERSION_BINS)) {

                    JSONArray mcpArray = response.getJSONArray(PayuConstants.PAYU_MCP_CONVERSION_BINS);
                    LookupDetails lookupDetails = new LookupDetails();
                    ArrayList<McpConversionBean> mcpConversionBeans = new ArrayList<>();
                    for (int i = 0; i < mcpArray.length(); i++) {
                        JSONObject mcpObject = mcpArray.optJSONObject(i);
                        if (mcpObject != null) {
                            McpConversionBean mcpConversionBean = new McpConversionBean();
                            mcpConversionBean.setOfferAmount(mcpObject.optString(PayuConstants.PAYU_OFFER_AMOUNT));
                            mcpConversionBean.setOfferCurrency(mcpObject.optString(PayuConstants.PAYU_OFFER_CURRENCY));
                            mcpConversionBean.setOfferExchangeRate(mcpObject.optString(PayuConstants.PAYU_OFFER_EXCHANGE_RATE));
                            mcpConversionBean.setMerchantOrderId(mcpObject.optString(PayuConstants.PAYU_MERCHANT_ORDER_ID));
                            mcpConversionBean.setLookupId(mcpObject.optString(PayuConstants.PAYU_LOOK_UP_ID));
                            mcpConversionBeans.add(mcpConversionBean);
                        }
                    }
                    lookupDetails.setMcpConversionBeans(mcpConversionBeans);

                    if (isJSONArrayAvailableForKey(response, PayuConstants.PAYU_SUPPORTED_CARD_SCHEMES)) {
                        JSONArray cardSchemesArray = response.getJSONArray(PayuConstants.PAYU_SUPPORTED_CARD_SCHEMES);
                        ArrayList<String> supportedCardSchemes = new ArrayList<>();
                        for (int i = 0; i < cardSchemesArray.length(); i++) {
                            String cardScheme = cardSchemesArray.optString(i);
                            if (cardScheme != null && !cardScheme.isEmpty()) {
                                supportedCardSchemes.add(cardScheme);
                            }
                        }
                        lookupDetails.setSupportedCardSchemes(supportedCardSchemes);
                    }
                    payuResponse.setLookupDetails(lookupDetails);
                }

                if (response.has(PayuConstants.PAYU_RESULT_CODE) && response.getInt(PayuConstants.PAYU_RESULT_CODE) != 0) {
                    postData.setCode(response.getInt(PayuConstants.PAYU_RESULT_CODE));
                    postData.setResult(response.getString(PayuConstants.PAYU_RESULT_MESSAGE));
                    postData.setStatus(PayuConstants.ERROR);
                } else {
                    postData.setCode(PayuErrors.NO_ERROR);
                    postData.setStatus(PayuConstants.SUCCESS);
                }

                payuResponse.setResponseStatus(postData);
            }
        } catch (ProtocolException e) {
            Log.d(TAG, "ProtocolException " + e.getMessage());
        } catch (IOException e) {
            Log.d(TAG, "IOException " + e.getMessage());
        } catch (JSONException e) {
            Log.d(TAG, "JSONException " + e.getMessage());
        }
        return payuResponse;
    }

    @Override
    protected void onPostExecute(PayuResponse payuResponse) {
        super.onPostExecute(payuResponse);
        mLookupApiListener.onLookupApiResponse(payuResponse);
    }

    private boolean isJSONArrayAvailableForKey(JSONObject response, String key) throws JSONException {
        if (response == null || key == null || key.isEmpty())
            return false;

        return response.has(key)
                && response.optJSONArray(key) != null && response.getJSONArray(key).length() > 0;
    }
}