//    Copyright (c) 2014 - 2015 payu@india.com
//
//    Permission is hereby granted, free of charge, to any person obtaining a copy
//    of this software and associated documentation files (the "Software"), to deal
//    in the Software without restriction, including without limitation the rights
//    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//    copies of the Software, and to permit persons to whom the Software is
//    furnished to do so, subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be included in
//    all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//    THE SOFTWARE.

package com.payu.india.Tasks;

import android.util.Log;

import com.payu.india.Interfaces.CheckoutDetailsListener;
import com.payu.india.Model.Bnpl;
import com.payu.india.Model.Emi;
import com.payu.india.Model.MerchantInfo;
import com.payu.india.Model.MerchantParamInfo;
import com.payu.india.Model.PayUEmiTenures;
import com.payu.india.Model.PaymentDetails;
import com.payu.india.Model.PayuConfig;
import com.payu.india.Model.PayuOffer;
import com.payu.india.Model.PayuResponse;
import com.payu.india.Model.PostData;
import com.payu.india.Model.TaxSpecification;
import com.payu.india.Model.Upi;
import com.payu.india.Payu.PayuConstants;
import com.payu.india.Payu.PayuErrors;
import com.payu.india.Payu.PayuUtils;
import com.payu.india.Tasks.base.PayUAsyncTask;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.ProtocolException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;

import javax.net.ssl.HttpsURLConnection;


/**
 * Async task which takes care of Getting checkout details for a merchant
 * Takes PayuConfig as input and sends PayuResponse to the calling activity
 * Activity which calls {@link GetCheckoutDetailsTask} should implement {@link com.payu.india.Interfaces.CheckoutDetailsListener}
 */
public class GetCheckoutDetailsTask extends PayUAsyncTask<PayuConfig, String, PayuResponse> {

    private CheckoutDetailsListener mCheckoutDetailsListener;

    public GetCheckoutDetailsTask(CheckoutDetailsListener checkoutDetailsListener) {
        this.mCheckoutDetailsListener = checkoutDetailsListener;
    }

    @Override
    protected void onPreExecute() {
        super.onPreExecute();
    }

    @Override
    protected PayuResponse doInBackground(PayuConfig... params) {

        PayuResponse payuResponse = new PayuResponse();
        PostData postData = new PostData();

        try {
            URL url;
            // get the payuConfig first
            PayuConfig payuConfig = params[0];

            // set the environment
            switch (payuConfig.getEnvironment()) {
                case PayuConstants.PRODUCTION_ENV:
                    url = new URL(PayuConstants.PRODUCTION_FETCH_DATA_URL);
                    break;
                case PayuConstants.MOBILE_STAGING_ENV:
                    url = new URL(PayuConstants.MOBILE_TEST_FETCH_DATA_URL);
                    break;
                case PayuConstants.STAGING_ENV:
                    url = new URL(PayuConstants.TEST_FETCH_DATA_URL);
                    break;
                case PayuConstants.MOBILE_DEV_ENV:
                    url = new URL(PayuConstants.MOBILE_DEV_FETCH_DATA_URL);
                    break;
                case PayuConstants.BIZCHECKOUT_TEST_ENV:
                    url = new URL(PayuConstants.BIZ_CHECKOUT_TEST_FETCH_DATA_URL);
                    break;
                default:
                    url = new URL(PayuConstants.PRODUCTION_FETCH_DATA_URL);
                    break;
            }

            byte[] postParamsByte = payuConfig.getData().getBytes("UTF-8");

            HttpsURLConnection conn = PayuUtils.getHttpsConn(url.toString(), payuConfig.getData());
            if (null != conn) {
                StringBuffer responseStringBuffer = PayuUtils.getStringBuffer(conn, false);

                JSONObject response = new JSONObject(responseStringBuffer.toString());
                payuResponse.setRawResponse(response);
                if (response.has(PayuConstants.DETAILS) && response.optJSONObject(PayuConstants.DETAILS) != null) {

                    JSONObject details = response.getJSONObject(PayuConstants.DETAILS);
                    JSONArray offersArray = details.optJSONArray(PayuConstants.OFFERS);
                    if (offersArray != null && offersArray.length() != 0) {
                        ArrayList<PayuOffer> offersList = new ArrayList<>();
                        for (int i = 0; i < offersArray.length(); i++) {
                            JSONObject offerObject = offersArray.getJSONObject(i);
                            PayuOffer payuOffer = new PayuOffer();
                            payuOffer.setId(offerObject.optString(PayuConstants.ID));
                            payuOffer.setTitle(offerObject.optString(PayuConstants.TITLE));
                            payuOffer.setDescription(offerObject.optString(PayuConstants.DESCRIPTION));
                            payuOffer.setMinAmount(offerObject.optString(PayuConstants.PAYU_MIN_AMOUNT));
                            payuOffer.setDiscount(offerObject.optString(PayuConstants.DISCOUNT));
                            payuOffer.setDiscountUnit(offerObject.optString(PayuConstants.DISCOUNT_UNIT));
                            payuOffer.setOfferType(offerObject.optString(PayuConstants.OFFER_TYPE));
                            payuOffer.setValidOnDays(offerObject.optString(PayuConstants.VALID_ON_DAYS));
                            payuOffer.setOfferKey("@" + offerObject.optString(PayuConstants.ID));
                            offersList.add(payuOffer);
                        }
                        payuResponse.setOfferDetailsList(offersList);
                    }

                    JSONObject downInfoObject = details.optJSONObject(PayuConstants.DOWN_INFO);
                    if (downInfoObject != null && downInfoObject.length() != 0) {
                        JSONArray issuingBanks = downInfoObject.optJSONArray(PayuConstants.ISSUING_BANKS);
                        if (issuingBanks != null && issuingBanks.length() != 0) {
                            ArrayList<String> issuingBanksList = new ArrayList<>();
                            for (int i = 0; i < issuingBanks.length(); i++) {
                                if (issuingBanks.optString(i) != null && !issuingBanks.getString(i).isEmpty()) {
                                    issuingBanksList.add(issuingBanks.getString(i));
                                }
                            }
                            payuResponse.setDownIssuingBanks(issuingBanksList);
                        }
                    }
                    JSONObject configObject = details.optJSONObject(PayuConstants.CONFIG);

                    MerchantInfo merchantInfo = new MerchantInfo();
                    JSONObject merchant = details.optJSONObject(PayuConstants.P_MERCHANT);
                    if (merchant != null && merchant.length() != 0) {
                        if (merchant.optString(PayuConstants.LOGO) != null)
                            merchantInfo.setLogo(merchant.optString(PayuConstants.LOGO));
                        merchantInfo.setWalletIdentifier(merchant.optString(PayuConstants.WALLET_IDENTIFIER));
                        merchantInfo.setDisplayName(merchant.optString(PayuConstants.DISPLAYNAME));
                        String retryCount = merchant.optString(PayuConstants.RETRYALLOWED);
                        if (retryCount != null && !retryCount.isEmpty())
                            merchantInfo.setRetryCount(Integer.parseInt(retryCount));
                        JSONObject merchantParamObj = merchant.optJSONObject(PayuConstants.MERCHANT_PARAM);
                        MerchantParamInfo paramInfo = new MerchantParamInfo();
                        if (merchantParamObj != null && merchantParamObj.has(PayuConstants.PAYU_IS_ADS_ENABLE)) {
                            paramInfo.setAdsEnabled(merchantParamObj.optInt(PayuConstants.PAYU_IS_ADS_ENABLE) == 1);
                        }
                        if (configObject != null && configObject.has(PayuConstants.PAYU_IS_QUICKPAY_ENABLE)) {
                            paramInfo.setQuickPayEnabled(configObject.optBoolean(PayuConstants.PAYU_IS_QUICKPAY_ENABLE));
                        }
                        if (configObject != null && configObject.has(PayuConstants.DEVICE_FP)) {
                            paramInfo.setDeviceIdEnabled(configObject.optBoolean(PayuConstants.DEVICE_FP));
                        }
                        if (configObject != null && configObject.has(PayuConstants.PAYU_IS_OFFER_ENABLED)) {
                            paramInfo.setOfferEnabled(configObject.optBoolean(PayuConstants.PAYU_IS_OFFER_ENABLED));
                        }

                        merchantInfo.setMerchantParamInfo(paramInfo);
                        if (merchant.optJSONArray(PayuConstants.MINKASUIBIBOCODES) != null)
                            merchantInfo.setOneClickPayBankCodes(getArrayListFromJsonArray(merchant.getJSONArray(PayuConstants.MINKASUIBIBOCODES)));
                        payuResponse.setMerchantInfo(merchantInfo);
                    }

                    if (configObject != null) {
                        TaxSpecification taxSpecification = new TaxSpecification();
                        taxSpecification.setCcTaxValue(getTaxFromConfigObject(configObject, PayuConstants.PAYU_CC));
                        taxSpecification.setDcTaxValue(getTaxFromConfigObject(configObject, PayuConstants.PAYU_DC));
                        taxSpecification.setNbTaxValue(getTaxFromConfigObject(configObject, PayuConstants.PAYU_NB));
                        taxSpecification.setUpiTaxValue(getTaxFromConfigObject(configObject, PayuConstants.UPI));
                        taxSpecification.setEmiTaxValue(getTaxFromConfigObject(configObject, PayuConstants.EMI_IN_RESPONSE));
                        taxSpecification.setCashTaxValue(getTaxFromConfigObject(configObject, PayuConstants.PAYU_CASH));
                        taxSpecification.setEnachTaxValue(getTaxFromConfigObject(configObject, PayuConstants.ENACH));
                        taxSpecification.setLazypayTaxValue(getTaxFromConfigObject(configObject, PayuConstants.LAZYPAY));
                        taxSpecification.setSiTaxValue(getTaxFromConfigObject(configObject, PayuConstants.PAYU_SI));
                        taxSpecification.setNeftRtgsTaxValue(getTaxFromConfigObject(configObject, PayuConstants.PAYU_NEFTRTGS));
                        taxSpecification.setMealCardTaxValue(getTaxFromConfigObject(configObject, PayuConstants.MC));
                        taxSpecification.setBnplTaxValue(getTaxFromConfigObject(configObject, PayuConstants.P_BNPL));
                        payuResponse.setTaxSpecification(taxSpecification);
                    }

                    JSONObject paymentOptions = details.optJSONObject(PayuConstants.PAYMENT_OPTIONS);
                    if (paymentOptions != null && paymentOptions.length() > 0) {
                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_CC)) {
                            payuResponse.setCreditCard(prepareListWithKeyData(details, PayuConstants.PAYU_CC));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_DC)) {
                            payuResponse.setDebitCard(prepareListWithKeyData(details, PayuConstants.PAYU_DC));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_NB)) {
                            ArrayList<PaymentDetails> nbList = prepareListWithKeyData(details, PayuConstants.PAYU_NB);
                            Collections.sort(nbList, new PaymentDetailsComparator());
                            payuResponse.setNetBanks(nbList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.ENACH)) {
                            ArrayList<PaymentDetails> nbSiList = prepareListWithKeyData(details, PayuConstants.ENACH);
                            payuResponse.setSiBankList(nbSiList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_SI)) {
                            ArrayList<PaymentDetails> siList = prepareListWithKeyData(details, PayuConstants.PAYU_SI);
                            payuResponse.setStandingInstructions(siList);
                        }
                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_NEFTRTGS)) {
                            ArrayList<PaymentDetails> nfetrtgsList = prepareListWithKeyData(details, PayuConstants.PAYU_NEFTRTGS);
                            payuResponse.setNeftRtgs(nfetrtgsList);
                        }


                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.PAYU_CASH)) {
                            ArrayList<PaymentDetails> walletList = prepareListWithKeyData(details, PayuConstants.PAYU_CASH);
                            PaymentDetails paymentDetails = getPhonePeIntentFromList(walletList);
                            if (paymentDetails != null) payuResponse.setPhonePe(paymentDetails);

                            Collections.sort(walletList, new PaymentDetailsComparator());
                            payuResponse.setCashCard(walletList);
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.IVR)) {
                            payuResponse.setIvr(prepareListWithKeyData(details, PayuConstants.IVR));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.IVRDC)) {
                            payuResponse.setIvrdc(prepareListWithKeyData(details, PayuConstants.IVRDC));
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.UPI)) {
                            JSONObject upiObject = paymentOptions.getJSONObject(PayuConstants.UPI).getJSONObject(PayuConstants.PAYU_ALL);
                            if (upiObject.has((PayuConstants.UPI).toUpperCase())) {
                                JSONObject upiCollections = upiObject.getJSONObject((PayuConstants.UPI).toUpperCase());
                                Upi upi = new Upi();
                                upi.setTitle(upiCollections.optString(PayuConstants.TITLE));
                                upi.setAdditionalCharge(upiCollections.optString(PayuConstants.ADDITIONAL_CHARGE));
                                upi.setBankDown(getBankDownStatus(downInfoObject, PayuConstants.UPI, PayuConstants.UPI));
                                upi.setOfferDetailsList(getOffersList(offersArray, upiCollections.optJSONArray(PayuConstants.OFFERS)));
                                payuResponse.setUpi(upi);
                            }
                            if (upiObject.has((PayuConstants.TEZ).toUpperCase())) {
                                JSONObject upiCollections = upiObject.getJSONObject((PayuConstants.TEZ).toUpperCase());
                                Upi upi = new Upi();
                                upi.setTitle(upiCollections.optString(PayuConstants.TITLE));
                                upi.setAdditionalCharge(upiCollections.optString(PayuConstants.ADDITIONAL_CHARGE));
                                upi.setBankDown(getBankDownStatus(downInfoObject, PayuConstants.UPI, PayuConstants.TEZ));
                                upi.setOfferDetailsList(getOffersList(offersArray, upiCollections.optJSONArray(PayuConstants.OFFERS)));
                                payuResponse.setGoogleTez(upi);
                            }
                            if (upiObject.has((PayuConstants.TEZOMNI).toUpperCase())) {
                                JSONObject upiCollections = upiObject.getJSONObject((PayuConstants.TEZOMNI).toUpperCase());
                                Upi upi = new Upi();
                                upi.setTitle(upiCollections.optString(PayuConstants.TITLE));
                                upi.setAdditionalCharge(upiCollections.optString(PayuConstants.ADDITIONAL_CHARGE));
                                upi.setBankDown(getBankDownStatus(downInfoObject, PayuConstants.UPI, PayuConstants.TEZ));
                                upi.setOfferDetailsList(getOffersList(offersArray, upiCollections.optJSONArray(PayuConstants.OFFERS)));
                                payuResponse.setGoogleTezOmni(upi);
                            }
                            if (upiObject.has((PayuConstants.UPI_INTENT).toUpperCase())) {
                                JSONObject upiCollections = upiObject.getJSONObject((PayuConstants.UPI_INTENT).toUpperCase());
                                Upi upi = new Upi();
                                upi.setTitle(upiCollections.optString(PayuConstants.TITLE));
                                upi.setAdditionalCharge(upiCollections.optString(PayuConstants.ADDITIONAL_CHARGE));
                                upi.setBankDown(getBankDownStatus(downInfoObject, PayuConstants.UPI, PayuConstants.UPI_INTENT));
                                upi.setOfferDetailsList(getOffersList(offersArray, upiCollections.optJSONArray(PayuConstants.OFFERS)));
                                payuResponse.setGenericIntent(upi);
                            }
                        }
                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.EMI_IN_RESPONSE)) {
                            ArrayList<Emi> ccemiList = getEmiList(details, PayuConstants.PAYU_CC);
                            if (ccemiList != null && !ccemiList.isEmpty()) {
                                payuResponse.setCcemi(ccemiList);
                            }
                            ArrayList<Emi> dcemiList = getEmiList(details, PayuConstants.PAYU_DC);
                            if (dcemiList != null && !dcemiList.isEmpty()) {
                                payuResponse.setDcemi(dcemiList);
                            }

                            ArrayList<Emi> cardlessemiList = new ArrayList<>();
                            ArrayList<Emi> otheremiList = getEmiList(details, PayuConstants.PAYU_OTHER);
                            if (otheremiList != null && !otheremiList.isEmpty())
                                cardlessemiList.addAll(otheremiList);

                            ArrayList<Emi> cardlessList = getEmiList(details, PayuConstants.PAYU_CARDLESS);
                            if (cardlessList != null && !cardlessList.isEmpty())
                                cardlessemiList.addAll(cardlessList);

                            if (!cardlessemiList.isEmpty()) {
                                payuResponse.setCardlessemi(cardlessemiList);
                            }
                        }

                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.LAZYPAY)) {
                            payuResponse.setLazyPay(prepareListWithKeyData(details, PayuConstants.LAZYPAY));
                        }
                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.P_BNPL)) {
                            ArrayList<Bnpl> bnplList = prepareBnplListWithKeyData(details, PayuConstants.P_BNPL);
                            if (bnplList != null && !bnplList.isEmpty()) {
                                payuResponse.setBnpl(bnplList);
                            }
                        }
                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.CLW)) {
                            payuResponse.setClosedLoopWallet(prepareListWithKeyData(details, PayuConstants.CLW));
                        }
                        if (isJSONObjectAvailableForKey(paymentOptions, PayuConstants.MC)) {
                            payuResponse.setMealCard(prepareListWithKeyData(details, PayuConstants.MC));
                        }

                    }
                }

                if (response.has(PayuConstants.STATUS) && response.getString(PayuConstants.STATUS).contentEquals(PayuConstants.STRING_ZERO)) {
                    postData = new PostData();
                    postData.setCode(PayuErrors.INVALID_HASH);
                    postData.setStatus(PayuConstants.ERROR);
                    postData.setResult(response.getString(PayuConstants.MSG));
                } else {
                    postData.setCode(PayuErrors.NO_ERROR);
                    // Result status should be success
                    postData.setResult(PayuErrors.DETAILS_FETCHED_SUCCESSFULLY);
                    postData.setStatus(PayuConstants.SUCCESS);
                }
            }
        } catch (ProtocolException e) {
            postData.setCode(PayuErrors.PROTOCOL_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        } catch (UnsupportedEncodingException e) {
            postData.setCode(PayuErrors.UN_SUPPORTED_ENCODING_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        } catch (JSONException e) {
            postData.setCode(PayuErrors.JSON_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        } catch (IOException e) {
            postData.setCode(PayuErrors.IO_EXCEPTION);
            postData.setStatus(PayuConstants.ERROR);
            postData.setResult(e.getMessage());
        }

        payuResponse.setResponseStatus(postData);
        return payuResponse;
    }

    private ArrayList<Emi> getEmiList(JSONObject details, String emiType) throws JSONException {
        JSONObject ccObject = details.getJSONObject(PayuConstants.PAYMENT_OPTIONS).getJSONObject(PayuConstants.EMI_IN_RESPONSE).getJSONObject(PayuConstants.PAYU_ALL).optJSONObject(emiType);
        if (ccObject != null && ccObject.optJSONObject(PayuConstants.PAYU_ALL) != null) {
            JSONObject allObject = ccObject.getJSONObject(PayuConstants.PAYU_ALL);
            Iterator<String> keysIterator = allObject.keys();
            ArrayList<Emi> emiList = new ArrayList<>();
            while (keysIterator.hasNext()) {
                String code = keysIterator.next();
                JSONObject emiObject = allObject.getJSONObject(code);
                Emi emi = new Emi();
                emi.setBankName(code);
                emi.setBankTitle(emiObject.optString(PayuConstants.TITLE));
                emi.setShortTitle(emiObject.optString(PayuConstants.SHORT_TITLE));
                emi.setMinAmount(emiObject.optString(PayuConstants.EMI_MINIMUM_AMOUNT));
                emi.setMaxAmount(emiObject.optString(PayuConstants.EMI_MAXIMUM_AMOUNT));
                JSONObject tenureOptions = emiObject.getJSONObject(PayuConstants.TENURE_OPTIONS);
                JSONObject bankEligibility = emiObject.optJSONObject(PayuConstants.ELIGIBILITY);
                if (bankEligibility != null) {
                    emi.setStatus(bankEligibility.optBoolean(PayuConstants.ELIGIBILITY_STATUS));
                    emi.setReason(bankEligibility.optString(PayuConstants.ELIGIBILITY_REASON));
                }
                ArrayList<PayUEmiTenures> payUEmiTenuresList = new ArrayList<>();
                Iterator<String> tenureKeysIterator = tenureOptions.keys();
                while (tenureKeysIterator.hasNext()) {
                    String bankCode = tenureKeysIterator.next();
                    JSONObject tenureObject = tenureOptions.getJSONObject(bankCode);
                    JSONObject eligibility = tenureObject.optJSONObject(PayuConstants.ELIGIBILITY);
                    PayUEmiTenures payUEmiTenures = new PayUEmiTenures();
                    payUEmiTenures.setBankCode(bankCode);
                    payUEmiTenures.setAdditionalCharge(tenureObject.optString(PayuConstants.ADDITIONAL_CHARGE));
                    payUEmiTenures.setBankDown(getBankDownStatus(details.optJSONObject(PayuConstants.DOWN_INFO), PayuConstants.EMI_IN_RESPONSE, bankCode));
                    payUEmiTenures.setOfferDetailsList(getOffersList(details.optJSONArray(PayuConstants.OFFERS), tenureObject.optJSONArray(PayuConstants.OFFERS)));
                    payUEmiTenures.setMinAmount(tenureObject.optString(PayuConstants.EMI_MINIMUM_AMOUNT));
                    payUEmiTenures.setMaxAmount(tenureObject.optString(PayuConstants.EMI_MAXIMUM_AMOUNT));
                    payUEmiTenures.setTenure(tenureObject.optString(PayuConstants.EMI_TENURE));
                    payUEmiTenures.setInterestRate(tenureObject.optString(PayuConstants.EMI_INTEREST_RATE));
                    payUEmiTenures.setMonthlyEmi(tenureObject.optString(PayuConstants.EMI_MONTHLY));
                    payUEmiTenures.setInterestCharged(tenureObject.optString(PayuConstants.EMI_INTEREST_CHARGED));
                    payUEmiTenures.setPaybackAmount(tenureObject.optString(PayuConstants.EMI_PAYBACK_AMOUNT));
                    payUEmiTenures.setBankCharge(tenureObject.optString(PayuConstants.BANK_CHARGE));
                    if (eligibility != null) {
                        payUEmiTenures.setStatus(eligibility.optBoolean(PayuConstants.ELIGIBILITY_STATUS));
                        payUEmiTenures.setReason(eligibility.optString(PayuConstants.ELIGIBILITY_REASON));
                    }
                    payUEmiTenuresList.add(payUEmiTenures);
                }
                emi.setPayUEmiTenuresList(payUEmiTenuresList);
                emiList.add(emi);
            }
            return emiList;
        }
        return null;
    }

    @Override
    protected void onPostExecute(PayuResponse payuResponse) {
        super.onPostExecute(payuResponse);
        mCheckoutDetailsListener.onCheckoutDetailsResponse(payuResponse);
    }

    static class PaymentDetailsComparator implements Comparator<PaymentDetails> {

        @Override
        public int compare(PaymentDetails p1, PaymentDetails p2) {

            return p1.getBankName().compareTo(p2.getBankName());

        }
    }

    private PaymentDetails getPhonePeIntentFromList(ArrayList<PaymentDetails> paymentDetailsList) {
        if (paymentDetailsList == null || paymentDetailsList.size() == 0 || com.payu.paymentparamhelper.PayuConstants.PHONEPE_INTENT == null || com.payu.paymentparamhelper.PayuConstants.PHONEPE_INTENT.isEmpty())
            return null;

        for (PaymentDetails paymentDetails : paymentDetailsList) {
            if (paymentDetails.getBankCode().equalsIgnoreCase(com.payu.paymentparamhelper.PayuConstants.PHONEPE_INTENT))
                return paymentDetails;
        }
        return null;
    }

    private ArrayList<PaymentDetails> prepareListWithKeyData(JSONObject details, String key) throws JSONException {
        JSONObject allObject = details.getJSONObject(PayuConstants.PAYMENT_OPTIONS).getJSONObject(key).getJSONObject(PayuConstants.PAYU_ALL);
        ArrayList<PaymentDetails> paymentDetailsList = new ArrayList<>();
        Iterator<String> keysIterator = allObject.keys();
        while (keysIterator.hasNext()) {
            String bankCode = keysIterator.next();
            JSONObject ccObject = allObject.getJSONObject(bankCode);
            PaymentDetails paymentDetails = new PaymentDetails();
            paymentDetails.setBankCode(bankCode);
            paymentDetails.setBankName(ccObject.optString(PayuConstants.TITLE));
            paymentDetails.setAdditionalCharge(ccObject.optString(PayuConstants.ADDITIONAL_CHARGE));
            paymentDetails.setIsSecureWebview(ccObject.optBoolean(PayuConstants.SECURE_WEBVIEW));
            paymentDetails.setBankDown(getBankDownStatus(details.optJSONObject(PayuConstants.DOWN_INFO), key, bankCode));
            paymentDetails.setOfferDetailsList(getOffersList(details.optJSONArray(PayuConstants.OFFERS), ccObject.optJSONArray(PayuConstants.OFFERS)));
            JSONArray verificationModeJsonArray = ccObject.optJSONArray(PayuConstants.VERIFICATION_MODE);
            if (null != verificationModeJsonArray && verificationModeJsonArray.length() > 0) {
                paymentDetails.setVerificationModeList(getArrayListFromJsonArray(verificationModeJsonArray));
            }
            paymentDetailsList.add(paymentDetails);
        }
        return paymentDetailsList;
    }

    private ArrayList<Bnpl> prepareBnplListWithKeyData(JSONObject details, String key) throws JSONException {
        JSONObject allObject = details.getJSONObject(PayuConstants.PAYMENT_OPTIONS).getJSONObject(key).getJSONObject(PayuConstants.PAYU_ALL);
        ArrayList<Bnpl> paymentDetailsList = new ArrayList<>();
        Iterator<String> keysIterator = allObject.keys();
        while (keysIterator.hasNext()) {
            String bankCode = keysIterator.next();
            JSONObject ccObject = allObject.getJSONObject(bankCode);
            Bnpl paymentDetails = new Bnpl();
            paymentDetails.setBankCode(bankCode);
            paymentDetails.setBankName(ccObject.optString(PayuConstants.TITLE));
            paymentDetails.setAdditionalCharge(ccObject.optString(PayuConstants.ADDITIONAL_CHARGE));
            paymentDetails.setMinAmount(ccObject.optString(PayuConstants.EMI_MINIMUM_AMOUNT));
            paymentDetails.setMaxAmount(ccObject.optString(PayuConstants.EMI_MINIMUM_AMOUNT));
            JSONObject eligibility = ccObject.optJSONObject(PayuConstants.ELIGIBILITY);
            if (eligibility != null) {
                paymentDetails.setStatus(eligibility.optBoolean(PayuConstants.ELIGIBILITY_STATUS));
                paymentDetails.setReason(eligibility.optString(PayuConstants.ELIGIBILITY_REASON));
            }
            paymentDetailsList.add(paymentDetails);
        }
        return paymentDetailsList;
    }

    private ArrayList<String> getArrayListFromJsonArray(JSONArray jsonArray) {
        ArrayList<String> arrayList = new ArrayList<String>();
        if (jsonArray != null) {
            for (int i = 0; i < jsonArray.length(); i++) {
                try {
                    arrayList.add(jsonArray.get(i).toString());
                } catch (JSONException e) {
                    Log.d(getClass().getSimpleName(), "JSONException " + e.getMessage());
                }
            }
        }
        return arrayList;
    }

    private ArrayList<PayuOffer> getOffersList(JSONArray fullArray, JSONArray offerArray) {
        if (fullArray == null || fullArray.length() == 0 || offerArray == null || offerArray.length() == 0)
            return null;

        ArrayList<PayuOffer> offersList = new ArrayList<>();
        for (int i = 0; i < offerArray.length(); i++) {
            JSONObject jsonObject = offerArray.optJSONObject(i);
            if (jsonObject != null) {
                String id = jsonObject.optString(PayuConstants.ID);
                if (id == null || id.isEmpty() || id.equalsIgnoreCase("null")) continue;

                for (int j = 0; j < fullArray.length(); j++) {
                    JSONObject fullJsonObject = fullArray.optJSONObject(j);
                    if (fullJsonObject != null) {
                        String idFullArray = fullJsonObject.optString(PayuConstants.ID);
                        if (idFullArray != null && idFullArray.equalsIgnoreCase(id)) {
                            PayuOffer payuOffer = new PayuOffer();
                            payuOffer.setId(fullJsonObject.optString(PayuConstants.ID));
                            payuOffer.setTitle(fullJsonObject.optString(PayuConstants.TITLE));
                            payuOffer.setDescription(fullJsonObject.optString(PayuConstants.DESCRIPTION));
                            payuOffer.setMinAmount(fullJsonObject.optString(PayuConstants.PAYU_MIN_AMOUNT));
                            payuOffer.setDiscount(fullJsonObject.optString(PayuConstants.DISCOUNT));
                            payuOffer.setDiscountUnit(fullJsonObject.optString(PayuConstants.DISCOUNT_UNIT));
                            payuOffer.setOfferType(fullJsonObject.optString(PayuConstants.OFFER_TYPE));
                            payuOffer.setValidOnDays(fullJsonObject.optString(PayuConstants.VALID_ON_DAYS));
                            payuOffer.setOfferKey("@" + fullJsonObject.optString(PayuConstants.ID));
                            offersList.add(payuOffer);
                        }
                    }
                }
            }
        }
        return offersList;
    }

    private boolean getBankDownStatus(JSONObject downInfoObject, String key, String bankCode) {
        if (downInfoObject == null || downInfoObject.length() == 0 || key == null || key.equalsIgnoreCase(PayuConstants.PAYU_CC) || key.equalsIgnoreCase(PayuConstants.PAYU_DC))
            return false;

        JSONArray banksArray = downInfoObject.optJSONArray(key);
        if (banksArray != null && banksArray.length() != 0) {
            for (int i = 0; i < banksArray.length(); i++) {
                if (banksArray.optString(i) != null && banksArray.optString(i).equalsIgnoreCase(bankCode))
                    return true;
            }
        }
        return false;
    }

    private String getTaxFromConfigObject(JSONObject configObject, String key) throws JSONException {
        if (configObject == null || configObject.length() == 0 || configObject.optJSONObject(PayuConstants.TAX_SPECIFICATION) == null)
            return null;

        JSONObject taxSpecification = configObject.getJSONObject(PayuConstants.TAX_SPECIFICATION);
        //check if tax is applied for specific mode
        if (taxSpecification.has(key)) return taxSpecification.getString(key);
        else if (taxSpecification.has(PayuConstants.DEFAULT))
            return taxSpecification.getString(PayuConstants.DEFAULT);
        else return null;
    }


    private boolean isJSONObjectAvailableForKey(JSONObject paymentOptions, String key) {
        return paymentOptions.has(key) && paymentOptions.optJSONObject(key) != null && paymentOptions.optJSONObject(key).optJSONObject(PayuConstants.PAYU_ALL) != null;
    }
}
