//    Copyright (c) 2014 - 2015 payu@india.com
//
//    Permission is hereby granted, free of charge, to any person obtaining a copy
//    of this software and associated documentation files (the "Software"), to deal
//    in the Software without restriction, including without limitation the rights
//    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//    copies of the Software, and to permit persons to whom the Software is
//    furnished to do so, subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be included in
//    all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//    THE SOFTWARE.

package com.payu.india.Tasks;

import android.util.Log;

import com.payu.india.Interfaces.EligibleBinsForEMIApiListener;
import com.payu.india.Model.EligibleEmiBins;
import com.payu.india.Model.PayuConfig;
import com.payu.india.Model.PayuResponse;
import com.payu.india.Model.PostData;
import com.payu.india.Payu.PayuConstants;
import com.payu.india.Payu.PayuErrors;
import com.payu.india.Payu.PayuUtils;
import com.payu.india.Tasks.base.PayUAsyncTask;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;

import javax.net.ssl.HttpsURLConnection;

/**
 * Async task which takes care of getting Eligible Bins for EMI
 * Takes PayuConfig as input and sends PayuResponse to the calling activity
 * Activity which calls {@link EligibleBinsForEMITask } should implement {@link EligibleBinsForEMIApiListener}
 */
public class EligibleBinsForEMITask extends PayUAsyncTask<PayuConfig, String, PayuResponse> {

    private  final String TAG = getClass().getSimpleName();

    EligibleBinsForEMIApiListener apiListener;

    public EligibleBinsForEMITask(EligibleBinsForEMIApiListener apiListener) {
        this.apiListener = apiListener;
    }

    @Override
    protected PayuResponse doInBackground(PayuConfig... params) {
        PayuResponse payuResponse = new PayuResponse();
        PostData postData = new PostData();

        try {
            URL url = null;
            // get the payuConfig first
            PayuConfig payuConfig = params[0];

            // set the environment
            switch (payuConfig.getEnvironment()) {
                case PayuConstants.PRODUCTION_ENV:
                    url = new URL(PayuConstants.PRODUCTION_FETCH_DATA_URL);
                    break;
                case PayuConstants.MOBILE_STAGING_ENV:
                    url = new URL(PayuConstants.MOBILE_TEST_FETCH_DATA_URL);
                    break;
                case PayuConstants.STAGING_ENV:
                    url = new URL(PayuConstants.TEST_FETCH_DATA_URL);
                    break;
                case PayuConstants.MOBILE_DEV_ENV:
                    url = new URL(PayuConstants.MOBILE_DEV_FETCH_DATA_URL);
                    break;
                default:
                    url = new URL(PayuConstants.PRODUCTION_FETCH_DATA_URL);
                    break;
            }


            HttpsURLConnection conn = PayuUtils.getHttpsConn(url.toString(), payuConfig.getData());
            if (null != conn) {
                StringBuffer responseStringBuffer = PayuUtils.getStringBuffer(conn, false);

                JSONObject jsonResponse = new JSONObject(responseStringBuffer.toString());
                payuResponse.setRawResponse(jsonResponse);
                if (jsonResponse.has(PayuConstants.MSG)) {
                    postData.setResult(jsonResponse.getString(PayuConstants.MSG));
                }
                if (jsonResponse.has(PayuConstants.STATUS) && jsonResponse.getInt(PayuConstants.STATUS) == 0) {
                    postData.setCode(PayuErrors.INVALID_HASH);
                    postData.setStatus(PayuConstants.ERROR);
                } else {
                    postData.setCode(PayuErrors.NO_ERROR);
                    postData.setStatus(PayuConstants.SUCCESS);
                }

                if (jsonResponse.has(PayuConstants.DETAILS) && jsonResponse.get(PayuConstants.DETAILS) != null) {
                    JSONObject detailsObject = jsonResponse.getJSONObject(PayuConstants.DETAILS);
                    Iterator<String> keys = detailsObject.keys();
                    ArrayList<EligibleEmiBins> eligibleEmiBinsList = new ArrayList<>();
                    EligibleEmiBins eligibleEmiBins;
                    while (keys.hasNext()) {
                        eligibleEmiBins = new EligibleEmiBins();
                        String bankShortName = keys.next();
                        if (detailsObject.optJSONObject(bankShortName) == null) {
                            /**
                             * Sample response for this block parsing -
                             * 	"details": {
                             * 		"isEligible": 1,
                             * 		"bank": "HDFC",
                             * 		"minAmount": 3000
                             *        }
                             * */
                            eligibleEmiBins.setIsEligible(detailsObject.optInt(PayuConstants.IS_ELIGIBLE));
                            eligibleEmiBins.setBankShortName(detailsObject.optString(PayuConstants.BANK));
                            eligibleEmiBins.setMinAmount(detailsObject.optInt(PayuConstants.MIN_AMOUNT));
                            if (detailsObject.has(PayuConstants.MSG))
                                eligibleEmiBins.setIsEligible(0);
                            eligibleEmiBinsList.add(eligibleEmiBins);
                            break;
                        } else {
                            /*
                            * Sample response for this block parsing -
                            * 	"details": {
		"HDFC": {
			"minAmount": 3000,
			"cardBins": ["360825", "360826", "360886", "361010", "361011", "361012", "361135", "361147", "361152", "361153", "367123", "368357", "401403", "405028", "416317", "417410", "418136", "430570", "434155", "434168", "434678", "435376", "435386", "435393", "436152", "436520", "437546", "451104", "457262", "457704", "461786", "461787", "463917", "485498", "485499", "489377", "517635", "517652", "518159", "522852", "523996", "524111", "524181", "524216", "524368", "524487", "524931", "528945", "532118", "532135", "533744", "536311", "541608", "545226", "545964", "552088", "552260", "552274", "552344", "552365", "552385", "552389", "552394", "553583", "555551", "555589", "556042", "558818", "558983", "559197", "559300", "652850", "652894", "652895"]
		},
		"HSBC": {
			"minAmount": 2000,
			"cardBins": ["418356", "418357", "431104", "438459", "447692", "456342", "456719", "461707", "461709", "461714", "461716", "461721", "461722", "486269", "490900", "512042", "517798", "540259", "542207", "554837", "554851"]
		},
		"ICICI": {
			"minAmount": 1500,
			"cardBins": ["374740", "374741", "374742", "376944", "377040", "377041", "402368", "402853", "405533", "407352", "407651", "407659", "410059", "423226", "423227", "431581", "437551", "444341", "447746", "447747", "447758", "450172", "455452", "461133", "462986", "470573", "474846", "486630", "517637", "517653", "517719", "523951", "524193", "524376", "525996", "540282", "545207", "552418"]
		},
		"HDFCD": {
			"minAmount": 5000,
			"cardBins": "No cardbins present"
		},
		"AXIS": {
			"minAmount": 2500,
			"cardBins": ["405995", "407439", "411146", "418212", "421137", "430832", "430833", "430834", "436560", "451456", "451457", "461116", "461117", "461118", "464118", "470011", "471860", "471861", "471863", "490906", "524178", "524240", "524508", "524512", "528028", "529629", "530562", "533467", "559340", "559341", "559342"]
		},
		"ICICID": {
			"minAmount": 5000,
			"cardBins": "No cardbins present"
		}
	}
                            * */
                            JSONObject bankObject = detailsObject.optJSONObject(bankShortName);
                            eligibleEmiBins.setBankShortName(bankShortName);
                            eligibleEmiBins.setMinAmount(bankObject.optInt(PayuConstants.MIN_AMOUNT));
                            eligibleEmiBins.setIsEligible(1); //setting it 1 here because we do not get for this response but in actual its eligible
                            ArrayList<String> cardBinsList = new ArrayList<>();
                            if (bankObject.has(PayuConstants.CARD_BINS) && bankObject.optJSONArray(PayuConstants.CARD_BINS) != null && bankObject.optJSONArray(PayuConstants.CARD_BINS).length() > 0) {
                                JSONArray cardBinsArray = bankObject.getJSONArray(PayuConstants.CARD_BINS);
                                for (int i = 0; i < cardBinsArray.length(); i++) {
                                    cardBinsList.add(cardBinsArray.getString(i));
                                }
                            }
                            eligibleEmiBins.setCardBins(cardBinsList);
                            eligibleEmiBinsList.add(eligibleEmiBins);
                        }
                    }
                    payuResponse.setEligibleEmiBins(eligibleEmiBinsList);
                }
            }
        } catch (MalformedURLException e) {
            Log.d(TAG, "MalformedURLException " + e.getMessage());
        } catch (ProtocolException e) {
            Log.d(TAG, "ProtocolException " + e.getMessage());
        } catch (IOException e) {
            Log.d(TAG, "IOException " + e.getMessage());
        } catch (JSONException e) {
            Log.d(TAG, "JSONException " + e.getMessage());
        }

        // gotta return payuResponse
        payuResponse.setResponseStatus(postData);
        return payuResponse;
    }

    @Override
    protected void onPostExecute(PayuResponse payuResponse) {
        super.onPostExecute(payuResponse);
        apiListener.onEligibleBinsForEMIApiResponse(payuResponse);
    }
}
