package com.payu.india.Payu;

import static android.content.Context.MODE_PRIVATE;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.security.KeyPairGeneratorSpec;
import android.security.keystore.KeyGenParameterSpec;
import android.security.keystore.KeyProperties;
import android.util.Base64;
import android.util.Log;

import java.security.KeyStore;
import java.util.HashMap;
import java.util.Map;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;


/**
 * Created by shoaib on 10/13/15.
 */
public class SharedPreferenceUtil {

    private static String CIPHER_TRANSFORMATION = PayuConstants.PAYU_AES_GCM_NO_PADDING;
    private static String ENCRYPTION_ALGORITHM = "AES";
    private static String KEYSTORE_PROVIDER_ANDROID_KEYSTORE = "AndroidKeyStore";
    private static String mAlias = "PayUKeyAlias";
    private static String TAG = "SharedPreferenceUtil";


    private static SecretKey createKeys(Context context) {
        try {
            KeyGenerator kpGenerator = KeyGenerator.getInstance(ENCRYPTION_ALGORITHM, KEYSTORE_PROVIDER_ANDROID_KEYSTORE);

            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
                KeyPairGeneratorSpec spec = new KeyPairGeneratorSpec.Builder(context)
                        .setAlias(mAlias)
                        .build();
                kpGenerator.init(spec);
            } else {
                KeyGenParameterSpec spec = new KeyGenParameterSpec.Builder(mAlias,
                        KeyProperties.PURPOSE_ENCRYPT | KeyProperties.PURPOSE_DECRYPT)
                        .setBlockModes(KeyProperties.BLOCK_MODE_CBC)
                        .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_PKCS7)
                        .build();
                kpGenerator.init(spec);
            }

            return kpGenerator.generateKey();
        } catch (Exception e) {
            Log.d(TAG, "createKey:Exception" + e.getLocalizedMessage());
        }
        return null;
    }

    private static SecretKey getKeys(Context context) {
        try {
            KeyStore ks = KeyStore.getInstance(KEYSTORE_PROVIDER_ANDROID_KEYSTORE);
            // If no input stream is needed, we still need to call load()
            ks.load(null);
            // Load the existing key pair from the Android Key Store
            KeyStore.SecretKeyEntry existingKey = (KeyStore.SecretKeyEntry) ks.getEntry(mAlias, null);
            if (existingKey != null) {
                return existingKey.getSecretKey();
            } else {
                return createKeys(context);
            }
        } catch (Exception e) {
            Log.d(TAG, "getKeys:Exception" + e.getLocalizedMessage());
            return createKeys(context);
        }
    }

    // Method to encrypt the data
    private static String encrypt(Context context, String data) {
        try {
            SecretKey key = getKeys(context);
            if (key != null) {
                Cipher cipher = Cipher.getInstance(CIPHER_TRANSFORMATION);
                cipher.init(Cipher.ENCRYPT_MODE, key);

                byte[] cipherText = cipher.doFinal(data.getBytes());
                byte[] iv = cipher.getIV();
                byte[] encryptedDataWithIV = new byte[iv.length + cipherText.length];
                System.arraycopy(iv, 0, encryptedDataWithIV, 0, iv.length);
                System.arraycopy(cipherText, 0, encryptedDataWithIV, iv.length, cipherText.length);

                return Base64.encodeToString(encryptedDataWithIV, Base64.DEFAULT);
            }
        } catch (Exception e) {
            Log.d(TAG, "encrypt:Exception" + e.getLocalizedMessage());
        }
        return null;
    }

    // Method to decrypt the data
    private static String decrypt(Context context, String encryptedData) {
        try {
            SecretKey key = getKeys(context);
            if (key != null) {
                String ex = encryptedData.replace("\n", "");
                byte[] cipherText = Base64.decode(ex, Base64.DEFAULT);

                Cipher cipher = Cipher.getInstance(CIPHER_TRANSFORMATION);
                byte[] iv = new byte[cipher.getBlockSize()];
                System.arraycopy(cipherText, 0, iv, 0, iv.length);

                cipher.init(Cipher.DECRYPT_MODE, key, new IvParameterSpec(iv));

                byte[] data = new byte[cipherText.length - iv.length];
                System.arraycopy(cipherText, iv.length, data, 0, data.length);

                return new String(cipher.doFinal(data));
            }
        } catch (Exception e) {
            Log.d(TAG, "decrypt:Exception" + e.getLocalizedMessage());
        }
        return null;
    }


    public static SharedPreferences getSharedPref(
            Context context,
            String fileName) {
        getKeys(context);
        return context.getSharedPreferences(fileName, MODE_PRIVATE);
    }

    public static void addStringToSharedPreference(Context context, String spFileName, String spKey, String spValue) {
        SharedPreferences sharedPref = getSharedPref(context, spFileName);
        SharedPreferences.Editor editor = sharedPref.edit();
        if (spValue != null) {
            editor.putString(spKey, encrypt(
                    context,
                    spValue
            ));
            editor.apply();
        }
    }

    public static void addIntToSharedPreference(Context context, String spFileName, String spKey, int spValue) {
        try {
            SharedPreferences sharedPref = getSharedPref(context, spFileName);
            SharedPreferences.Editor editor = sharedPref.edit();
            editor.putString(spKey, encrypt(
                    context,
                    String.valueOf(spValue)
            ));
            editor.apply();
        } catch (Exception e) {
            Log.d(TAG, "addIntToSharedPreference:Exception" + e.getLocalizedMessage());
        }
    }

    public static void addBooleanToSharedPreference(Context context, String spFileName, String spKey, boolean spValue) {
        SharedPreferences sharedPref = getSharedPref(context, spFileName);
        SharedPreferences.Editor editor = sharedPref.edit();
        editor.putString(spKey, encrypt(
                context,
                String.valueOf(spValue)
        ));
        editor.apply();
    }

    public static String getStringFromSharedPreference(Context context, String spFileName, String spKey, String defaultValue) {
        try {
            SharedPreferences sharedPref = getSharedPref(context, spFileName);
            String encryptedString = sharedPref.getString(spKey, defaultValue);
            String decodedString = decrypt(
                    context,
                    encryptedString
            );
            if (decodedString == null) return defaultValue;
            return decodedString;
        } catch (Exception e) {
            removeAllFromSharedPref(context, spFileName);
            return defaultValue;
        }
    }

    public static int getIntFromSharedPreference(Context context, String spFileName, String spKey, int defaultValue) {
        try {
            SharedPreferences sharedPref = getSharedPref(context, spFileName);
            String encryptedData = sharedPref.getString(spKey, String.valueOf(defaultValue));
            String decryptedData = decrypt(context, encryptedData);
            if (decryptedData == null) return defaultValue;
            return Integer.parseInt(decryptedData);
        } catch (Exception e) {
            removeAllFromSharedPref(context, spFileName);
            return defaultValue;
        }
    }

    public static boolean getBooleanFromSharedPreference(Context context, String spFileName, String spKey, boolean defaultValue) {
        try {
            SharedPreferences sharedPref = getSharedPref(context, spFileName);
            String encryptedData = sharedPref.getString(spKey, String.valueOf(defaultValue));
            String decryptedData = decrypt(context, encryptedData);
            if (decryptedData == null) return defaultValue;
            return Boolean.getBoolean(decryptedData);
        } catch (Exception e) {
            removeAllFromSharedPref(context, spFileName);
            return defaultValue;
        }
    }

    /**
     * Simple helper method read a preference and return all the values as hashmap<String, Object> format.
     *
     * @param context  context
     * @param prefName name of the perference file.
     * @return hashmap.
     */
    public static Map<String, ?> getSharedPrefMap(Context context, String prefName) {
        try {
            HashMap<String, Object> sharedPrefHashMap = new HashMap<>();
            SharedPreferences snoozeSharedPref = getSharedPref(context, prefName);
            Map<String, ?> allEntries = snoozeSharedPref.getAll();

            return allEntries;
        } catch (Exception e) {
            Map<String, ?> allEntries = new HashMap<>();
            removeAllFromSharedPref(context, prefName);
            return allEntries;
        }
    }

    /**
     * Simple helper method to remove all the keys from given shared pref file.
     *
     * @param context
     * @param prefName
     */
    public static void removeAllFromSharedPref(Context context, String prefName) {
        try {
            SharedPreferences settings = getSharedPref(context, prefName);
            settings.edit().clear().apply();
        } catch (Exception e) {
            Log.d(TAG, "removeAllFromSharedPref:Exception" + e.getLocalizedMessage());
        }
    }

}
