package com.payu.india.Model.fetchConvenienceFee;

import android.util.Log;

import com.payu.india.Payu.PayuConstants;
import com.payu.paymentparamhelper.V2ApiBase;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * Represents a request to fetch conversion fee details from the PayU API. This class {@link com.payu.india.Tasks.FetchConvFeeTask} contains
 * payment-related information such as the PayU ID, merchant key, transaction amount, source of the transaction,
 * timestamp, and request ID. It also supports dynamic attributes for both common and variable dynamic data.
 * <p>
 * The class uses the Builder design pattern to facilitate the creation of a `FetchConvFeeRequest` instance, ensuring
 * that all necessary fields are set before the object is constructed.
 * </p>
 *
 * Example usage:
 * <pre>
 * FetchConvFeeRequest request = new FetchConvFeeRequest.Builder()
 *     .setPayuId("examplePayuId")
 *     .setMerchantKey("exampleMerchantKey")
 *     .setAmount(100.50)
 *     .setSource("website")
 *     .setTimeStamp("2025-03-24T10:00:00Z")
 *     .setRequestId("request123")
 *     .setCommonDynamicAttributes(new CommonDynamicAttributes(...))
 *     .setVariableDynamicAttributesList(new ArrayList<>())
 *     .build();
 * </pre>
 *
 * This class provides the following functionalities:
 * <ul>
 *   <li>Building a request object with required fields using the Builder pattern.</li>
 *   <li>Converting the request object into a JSON format string using {@link #getJson()}.</li>
 *   <li>Handling common and variable dynamic attributes as part of the request.</li>
 * </ul>
 *
 * <p>
 * Methods:
 * <ul>
 *   <li>{@link FetchConvFeeRequest#getJson()} - Converts the request into a JSON format string to be sent to the PayU API.</li>
 * </ul>
 * </p>
 *
 * <p>
 * Related classes:
 * <ul>
 *   <li>{@link Builder} - The builder class used to construct the `FetchConvFeeRequest`.</li>
 *   <li>{@link CommonDynamicAttributes} - Represents common dynamic attributes in the request.</li>
 *   <li>{@link VariableDynamicAttributes} - Represents variable dynamic attributes in the request.</li>
 * </ul>
 * </p>
 *
 * @see Builder
 * @see CommonDynamicAttributes
 * @see VariableDynamicAttributes
 */
public class FetchConvFeeRequest extends V2ApiBase {
    private final String payuId;
    private final String merchantKey;
    private final double amount;
    private final String source;
    private final String timeStamp;
    private final String requestId;
    private final CommonDynamicAttributes commonDynamicAttributes;
    private final List<VariableDynamicAttributes> variableDynamicAttributesList;

    /**
     * Constructs a new {@link FetchConvFeeRequest} using the provided builder.
     *
     * @param builder The builder that provides the necessary attributes for creating the request.
     */
    public FetchConvFeeRequest(Builder builder) {
        this.payuId = builder.payuId;
        this.merchantKey = builder.merchantKey;
        this.amount = builder.amount;
        this.source = builder.source;
        this.timeStamp = builder.timeStamp;
        this.requestId = builder.requestId;
        this.commonDynamicAttributes = builder.commonDynamicAttributes;
        this.variableDynamicAttributesList = builder.variableDynamicAttributesList;
    }

    /**
     * The Builder class is used to construct a {@link FetchConvFeeRequest} with the necessary attributes.
     * It provides setter methods for all required fields and a {@link #build()} method to create the final object.
     */
    public static class Builder {
        private String payuId;
        private String merchantKey;
        private double amount;
        private String source;
        private String timeStamp;
        private String requestId;
        private CommonDynamicAttributes commonDynamicAttributes;
        private List<VariableDynamicAttributes> variableDynamicAttributesList;

        /**
         * Sets the PayU ID for the request.
         *
         * @param payuId The PayU ID.
         * @return The builder instance.
         */
        public Builder setPayuId(String payuId) {
            this.payuId = payuId;
            return this;
        }

        /**
         * Sets the merchant key for the request.
         *
         * @param merchantKey The merchant key.
         * @return The builder instance.
         */
        public Builder setMerchantKey(String merchantKey) {
            this.merchantKey = merchantKey;
            return this;
        }

        /**
         * Sets the amount for the request.
         *
         * @param amount The amount.
         * @return The builder instance.
         */
        public Builder setAmount(double amount) {
            this.amount = amount;
            return this;
        }

        /**
         * Sets the source for the request.
         *
         * @param source The source.
         * @return The builder instance.
         */
        public Builder setSource(String source) {
            this.source = source;
            return this;
        }

        /**
         * Sets the timestamp for the request.
         *
         * @param timeStamp The timestamp.
         * @return The builder instance.
         */
        public Builder setTimeStamp(String timeStamp) {
            this.timeStamp = timeStamp;
            return this;
        }

        /**
         * Sets the common dynamic attributes for the request.
         *
         * @param commonDynamicAttributes The common dynamic attributes.
         * @return The builder instance.
         */
        public Builder setCommonDynamicAttributes(CommonDynamicAttributes commonDynamicAttributes) {
            this.commonDynamicAttributes = commonDynamicAttributes;
            return this;
        }

        /**
         * Sets the list of variable dynamic attributes for the request.
         *
         * @param variableDynamicAttributesList The list of variable dynamic attributes.
         * @return The builder instance.
         */
        public Builder setVariableDynamicAttributesList(List<VariableDynamicAttributes> variableDynamicAttributesList) {
            this.variableDynamicAttributesList = variableDynamicAttributesList;
            return this;
        }

        /**
         * Sets the request ID for the request.
         *
         * @param requestId The request ID.
         * @return The builder instance.
         */
        public Builder setRequestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        /**
         * Builds and returns the {@link FetchConvFeeRequest} object.
         *
         * @return A new instance of {@link FetchConvFeeRequest}.
         */
        public FetchConvFeeRequest build() {
            return new FetchConvFeeRequest(this);
        }
    }

    /**
     * Converts this {@link FetchConvFeeRequest} into a JSON string representation.
     *
     * @return The JSON string representing the request, or jsonObject if an error occurs during JSON conversion.
     */
    @Override
    public String getJson() {
        JSONObject jsonObject = new JSONObject();
        try {

            jsonObject.put(PayuConstants.PAYU_ID, payuId);
            jsonObject.put(PayuConstants.PAYU_MERCHANT_KEY, merchantKey);
            jsonObject.put(PayuConstants.AMOUNT, amount);
            jsonObject.put(PayuConstants.PAYU_SOURCE, source);
            jsonObject.put(PayuConstants.PAYU_TIME_STAMP, timeStamp);
            jsonObject.put(PayuConstants.PAYU_REQUESTID, requestId);
            JSONObject commonAttributes = getCommonDynamicAttributes(commonDynamicAttributes);
            if (commonAttributes != null)
                jsonObject.put(PayuConstants.PAYU_COMMON_DYNAMIC_ATTRIBUTES, commonAttributes);
            JSONArray jsonArray = getVariableDynamicAttributesJsonarray(variableDynamicAttributesList);
            if (jsonArray != null)
                jsonObject.put(PayuConstants.PAYU_VARIABLE_DYNAMIC_ATTRIBUTES, jsonArray);
        } catch (JSONException e) {
            Log.d("BinBaseDetailsRequest", "Exception " + e.getMessage());
        }

        return jsonObject.toString();
    }

    /**
     * Converts a list of {@link VariableDynamicAttributes} objects into a {@link JSONArray} representation.
     * Each {@link VariableDynamicAttributes} object is converted into a {@link JSONObject} containing its
     * combination key and Ibibo code, and added to the resulting JSON array.
     *
     * If any {@link JSONException} occurs while adding attributes to the JSON object, the method returns null.
     *
     * @param variableDynamicAttributesList The list of {@link VariableDynamicAttributes} to be converted.
     * @return A {@link JSONArray} containing the converted JSON objects, or null if an error occurs.
     */
    private JSONArray getVariableDynamicAttributesJsonarray(List<VariableDynamicAttributes> variableDynamicAttributesList) {
        JSONArray jsonArray = new JSONArray();
        for (VariableDynamicAttributes variableDynamicAttribute : variableDynamicAttributesList) {
            JSONObject jsonObject = new JSONObject();
            try {
                jsonObject.put(PayuConstants.PAYU_COMBINATION_KEY, variableDynamicAttribute.getCombinationKey());
                jsonObject.put(PayuConstants.IBIBOCODE, variableDynamicAttribute.getIbiboCode());
                jsonArray.put(jsonObject);
            } catch (JSONException e) {
                return null;
            }
        }
        return jsonArray;
    }

    /**
     * Converts a {@link CommonDynamicAttributes} object into a {@link JSONObject} representation.
     * The object is populated with the mode and action (set to "capture") from the provided
     * {@link CommonDynamicAttributes}.
     *
     * If a {@link JSONException} occurs while adding attributes to the JSON object, the method returns null.
     *
     * @param commonDynamicAttributes The {@link CommonDynamicAttributes} object containing the attributes to be converted.
     * @return A {@link JSONObject} containing the converted attributes, or null if an error occurs.
     */

    private JSONObject getCommonDynamicAttributes(CommonDynamicAttributes commonDynamicAttributes) {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put(PayuConstants.MODE, commonDynamicAttributes.getMode());
            jsonObject.put(PayuConstants.PAYU_ACTION, PayuConstants.PAYU_CAPTURE);
        } catch (JSONException e) {
            return null;
        }
        return jsonObject;
    }
}

